/*
 * To change this template, choose Tools | Templates
 * and on the template in the editor.
 */
package org.lsst.ccs.subsystems.fcs.drivers;

import java.util.concurrent.locks.Condition;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.lsst.ccs.bus.BadCommandException;
import org.lsst.ccs.bus.ErrorInCommandExecutionException;
import org.lsst.ccs.subsystems.fcs.ClampActuatorModule;
import org.lsst.ccs.subsystems.fcs.common.PieceOfHardware;
import org.lsst.ccs.subsystems.fcs.errors.MismatchMotorConfigurationException;
import org.lsst.ccs.subsystems.fcs.errors.SDORequestError;

/**
 * This deviceErrorFile a model for the clamp actuator in the Single Filter Test.
 * The actuator deviceErrorFile used to unlock or release a clamp.
 * To unlock a clamp, the actuator must be ON and to release the clamp
 * it has to be OFF.
 * There deviceErrorFile 2 clamps actuators on the carousel : 
 * one for side X-, one for side X+
 * For the Single Filter Test the actuator will be Maxon Motor EPOS2 24/5.
 * (cf EPOS2_Firmware_Specification_En.pdf)
 * There deviceErrorFile many operation modes for this actuators. 
 * For the Single Filter Test, we will use VELOCITY mode or CURRENT mode.
 * 
 * 
 * @author virieux
 */
public class CanOpenClampActuatorModule extends ClampActuatorModule implements PieceOfHardware {


    
    public enum ActuatorMode {
	VELOCITY, CURRENT, 
    }
    
    CanOpenProxy tcpProxy;

    /**
     * Node ID : the Can Open node ID this actuator deviceErrorFile expected to have.
     */
    private String nodeID;
    
    /**
     * Serial Number 
     */
    private String serialNB;
    
    
    /* Mode of Operation */
    private ActuatorMode mode;
    
    /*the motor which deviceErrorFile driven by this actuator*/
    private Motor motor;
    
    //Used because we have to wait during the time needed to enable the actuator (very short but not null !)
    //of the clamp
    final Lock lock = new ReentrantLock();
    final Condition enablingCompleted = lock.newCondition();
    
     /* This is used when we update the clamp clampState with the values returned 
     *  by the sensors.
     */
    protected volatile boolean enabling = false;
    
    /**
     * @return the receivedNodeID
     */
    @Override
    public String getNodeID() {
        return nodeID;
    }

    /**
     * @param receivedNodeID the receivedNodeID to set
     */
    public void setNodeID(String nodeID) {
        this.nodeID = nodeID;
    }

    /**
     * @return the serialNB
     */
    @Override
    public String getSerialNB() {
        return serialNB;
    }

    /**
     * @param serialNB the serialNB to set
     */
    public void setSerialNB(String serialNB) {
        this.serialNB = serialNB;
    }

    @Override
    public boolean isConfigOK() {
        return tcpProxy.isConfigOK(this);
    }

    /**
     * @return the motor
     */
    public Motor getMotor() {
        return motor;
    }

    /**
     * @param motor the motor to set
     */
    public void setMotor(Motor motor) {
        this.motor = motor;
    }

    /**
     * @return the mode as a String
     */
    public String getMode() {
        return mode.toString();
    }

    /**
     * @param mode the mode to set
     */
    public void setMode(String modeInString) {       
        this.mode = ActuatorMode.valueOf(modeInString);
    }
    
    /*For Spring*/
    public CanOpenClampActuatorModule() {      
    }
   
    /*Constructor to be used when we'll get rid of Spring*/
    public CanOpenClampActuatorModule(String nodeID, int inputID, String serial, Motor motor) {
        this.tcpProxy = (CanOpenProxy) this.getModule("tcpProxy");
        this.nodeID = nodeID;
        this.serialNB = serial;
        this.mode = ActuatorMode.CURRENT;
        this.motor = motor;
    }

    
   /**
     * @return the readValue on the Actuator
     */
    public void updateReadValue() throws BadCommandException, SDORequestError {
        if (mode.equals(ActuatorMode.VELOCITY)) {
            this.readValue = readVelocity();
        } else if (mode.equals(ActuatorMode.CURRENT)) {
            this.readValue =  readCurrent();
        } else throw new BadCommandException(getName() + "is not in VELOCITY or CURRENT mode");
    } 
    
    

    public void initialize() {
    }
    
    // All the getters following are for the GUI needs
    @Override
    public int getMotor_continuousCurrentLimit() {
        return motor.getContinuousCurrentLimit();
    }

    @Override
    public int getMotor_maxSpeedInCurrentMode() {
        return motor.getMaxSpeedInCurrentMode();
    }

    @Override
    public String getMotor_mode() {
        return this.getMode();
    }

    @Override
    public int getMotor_outputCurrentLimit() {
        return motor.getOutputCurrentLimit();
    }

    @Override
    public int getMotor_polePairNumber() {
        return motor.getPolePairNumber();
    }

    @Override
    public int getMotor_thermalTimeConstantWinding() {
        return motor.getThermalTimeConstantWinding();
    }

    @Override
    public int getMotor_type() {
        return motor.getMotorType();
    }

    @Override
    public int getPositionSensorType() {
        return motor.getPositionSensorType();
    }
    //end 
    

    
    @Override
    public void initModule() {
        tcpProxy = (CanOpenProxy) this.getModule("tcpProxy");
        mode = ActuatorMode.CURRENT;
    }
    
    public boolean isEnable() throws SDORequestError {
        lock.lock();
        
        try {
            while(this.enabling) {
                try {
                    this.enablingCompleted.await();
                } catch (InterruptedException ex) {
                    log.debug(getName() + " INTERRUPTED during waiting for the actuator beeing enable");
                }

            }
        String controlWordInHexa = readControlWord();
        int controlWord = Integer.parseInt(controlWordInHexa, 16);
        return (controlWord == 15); /*F in hexa*/
        } finally {
            lock.unlock();
        }
    }
    
    public void resetFault() throws SDORequestError {
        writeControlWord("80");
    }
  
    /**
     * This methods enable the actuator : i.e. this makes the actuator able to receive commands.
     * @return
     * @throws SDORequestError 
     */
    public String enable () throws SDORequestError {
        lock.lock();
        this.enabling = true;
        
        try {
            disable();
            switchOn();
            return getName() + " ENABLE";
        } finally {
            this.enabling = false;
            this.enablingCompleted.signal();
            lock.unlock();
        }
    }
    
    
    
    public String disable() throws SDORequestError{      
        writeControlWord("6");
        //TODO test the return code for the command writeSDO
        return getName() + " DISABLE";
    }
    
    public void switchOn() throws SDORequestError {
        writeControlWord("F");
    }
    
    /**
     * This method save the parameters in the actuator memory.
     * @return
     * @throws SDORequestError 
     */
    public Object saveParameters() throws SDORequestError {
        return this.writeSDO("1010", "1", "4", "65766173");
    }
    
    public Object restoreParameters() throws SDORequestError {
        return this.writeSDO("1010", "1", "4", "64616F6C");
    }
    
    /**
     * Write a value in hexa in the control world (index=6040, subindex=0,size=2)
     * @param value in hexa
     */
    public void writeControlWord(String value) throws SDORequestError {
        this.writeSDO("6040", "0", "2", value);
    }
    
     /**
     * Read the control world (index=6040, subindex=0,size=2)
     * @return value in hexa
     */
    public String readControlWord() throws SDORequestError {
        return this.readSDO("6040", "0");
    }
    
    /**
    * Read the status world (index=6041, subindex=0,size=2)
    * @return value in hexa
    */
    public String readStatusword() throws SDORequestError {
        return this.readSDO("6041", "0");
    }
    

    

    
    
    public void quickStop() throws SDORequestError {
        if (mode.equals(ActuatorMode.VELOCITY)) {
            writeControlWord("B");
        } else if (mode.equals(ActuatorMode.CURRENT)) {
            writeControlWord("2");
        } //TODO else throw an exception ?
    }
    
    /**
     * This methods enables the actuator and puts the motor on in sending the appropriate current value 
     * to the actuator. 
     * 
     * @return a message 
     */
    @Override
    public String on() throws BadCommandException, ErrorInCommandExecutionException {
        try {
            
            this.enable(); 
            
            if (!isEnable()) throw new BadCommandException("Actuator has to be enable prior ON command.");
            

            if (mode.equals(ActuatorMode.VELOCITY)) {
                    setVelocity();               
            } else if (mode.equals(ActuatorMode.CURRENT)) { 
                    if ( (getMotor().getCurrentValue() == 0)) {
                        throw new BadCommandException("A current value has to be set");
                    }
                    writeCurrent(getMotor().getCurrentValue());
               
            }
            this.on = true;
            this.sentValue = this.getMotor().getCurrentValue();
            this.updateReadValue();
            this.sendToStatus(this.getStatusData());
            return getName() + " ON";
        } catch (SDORequestError ex) {
            log.error(getName() + " Can Open ERROR in sending the command ON.");
            throw new ErrorInCommandExecutionException(ex + "Error in reading ControlWord");
        }

    }
    
    @Override
    public String maintainCurrent() throws BadCommandException, ErrorInCommandExecutionException {
        
        try {
            if (!isEnable()) throw new BadCommandException("Actuator has to be enable prior maintainCurrent command.");
            writeCurrent(this.motor.getCurrentValueToMaintain());
        } catch (SDORequestError ex) {
            log.error(getName() + " Can Open ERROR in sending the command maintainCurrent.");
            throw new ErrorInCommandExecutionException(ex + "Error in reading ControlWord");
        }
        return (getName() + " is maintaining its position");
    }

    /**
     * This method set to zero the current value and stops the motor motion.
     * Then it disables the actuator.
     * @return 
     */
    public String off() throws BadCommandException, ErrorInCommandExecutionException {
        try {
            if (mode.equals(ActuatorMode.VELOCITY)) {
               
                    stopVelocity();
                    disable();
                
            } else if (mode.equals(ActuatorMode.CURRENT)) {

                    stopCurrent();
                    disable();
                
            }
            this.on = false;
            this.sentValue = 0;
            this.updateReadValue();
            this.sendToStatus(this.getStatusData());
            return getName() + " OFF";
        } catch (SDORequestError ex) {
            Logger.getLogger(CanOpenClampActuatorModule.class.getName()).log(Level.SEVERE, null, ex);
            throw new ErrorInCommandExecutionException(ex + "Error in reading SDO request");
        }
    }
    
    
    /**************************************************************************/
    /*Methods to read or display the errors that have occured on the device   */
    /*The errors are stored in 2 places :
    /* error register at index 1001 
    /* error history at index 1003               
    /* The error register and the error history 
    /**************************************************************************/
    
        
    /**
     * At index 1001 there deviceErrorFile an error register for the device. 
     * The device maps internal errors in this byte.
     * @return error code in Hexa
     */
    public String readErrorRegister() throws SDORequestError {
        String error = (String) readSDO("1001", "0");
        return String.format("%02x", Integer.parseInt(error));
    }
    
    public String displayErrorRegister() throws SDORequestError {
        String errorInHexa = readErrorRegister();      
        String errorName = CanOpenErrorsTable.errorRegisterCodes.getProperty(errorInHexa);
        log.debug("error register=" + errorInHexa);
        return errorName;
    }
    
    /**
     * The error history holds errors that have occurred on the device and 
     * have been signalled via the emergency object.
     * This methods reads the Maxon Motor Error History and returns an array of 
     * error code.
     * @return errors : a list of device error code
     */
    public String[] readErrorHistory() throws SDORequestError {              
        int numberOfErrors = readNumberOfErrors();
        String[] errors = new String[numberOfErrors];
        for (int i=0; i < numberOfErrors;i++) {
            String subindex = Integer.toHexString(i+1);
            errors[i] = (String) readSDO("1003", subindex);
        }
        return errors;
    }
    

    public String displayErrorHistory() throws SDORequestError {
        String[] errorHistory = readErrorHistory();
        StringBuilder sb = new StringBuilder("Error history contains " + errorHistory.length + " errors.");
        if (!(errorHistory.length == 0)) {
            sb.append("List of errors in history : "
                    + "the newest is the first, the oldest the last\n");
            for (int ix=0; ix < errorHistory.length; ix++) {
                sb.append("Error code (in hexa)=");sb.append(errorHistory[ix]);
                sb.append("/ error name=");
                sb.append(CanOpenErrorsTable.deviceErrorCodes.getProperty(errorHistory[ix].toUpperCase()));
                sb.append("\n");
            }
        }
        return sb.toString();
    }
    

    
    /**
     * Extract from the Maxon Motor firmware specification :
     * Contains the number of actual errors that are recorded in the array starting 
     * at subindex 1. Writing a “0” deletes the error history (empties the array). 
     * Values higher then “0” (zero) are not allowed to write."
     * 
     * Read and returns the number of errors registred in the Error History 
     */
    public int readNumberOfErrors() throws SDORequestError {
        String errorNBinHEXA = (String) readSDO("1003", "0");
        return Integer.parseInt(errorNBinHEXA, 16);
    }
    
    
    /*********************************/
    /*Methods connected to the motors*/
    /*********************************/
    
    /**
     * Write the values of the motor fields in the actuator memory.
     * @throws SDORequestError 
     */
    public void writeMotorParameters() throws SDORequestError {
        disable();    
        writeMotorType(this.motor.getMotorType());
        
        // mandatory in Current mode
        writeMotorData("01","2", this.motor.getContinuousCurrentLimit());
        writeMotorData("04","4", this.motor.getMaxSpeedInCurrentMode());
        writeMotorData("05","2", this.motor.getThermalTimeConstantWinding());
        //Guillaume wants to set this parameters too :
        writeMotorData("02","2", this.motor.getOutputCurrentLimit());
        writeMotorData("03","1", this.motor.getPolePairNumber());
        writePositionSensorType(this.motor.getPositionSensorType());
              
        //TODO mandatory in velocity mode
        
    }
    
    /**
     * This command set the motor type on the actuator.
     */
    public void writeMotorType(int value) throws SDORequestError {
        String typeInHexa = Integer.toHexString(value);
        writeSDO("6402","0","2",typeInHexa);
    }
    
    /**
     * This command read the motor type on the actuator.
     * @return the motor type in decimal
     */
    public int readMotorType() throws SDORequestError {
        String motorTypeInHexa = readSDO("6402","0");
        return Integer.parseInt(motorTypeInHexa, 16);
    }
    
     /**
     * This command read the position sensor type on the actuator.
     * @return the position sensor type in decimal
     */
    public int readPositionSensorType() throws SDORequestError {
        String positionSensorTypeInHexa = readSDO("2210","02");
        return Integer.parseInt(positionSensorTypeInHexa, 16);
    }
    
    public void writePositionSensorType(int value) throws SDORequestError {
        this.motor.setPositionSensorType(value);
        String valueInHexa = Integer.toHexString(value);
        writeSDO("2210","2","2", valueInHexa);
    }
    
    public void writeMaxSpeed(int value) throws SDORequestError {
        this.motor.setMaxSpeedInCurrentMode(value);
        writeMotorData("4", "4", value );
    }
    
    /**
     * This method writes the parameters for the motor on the actuator.
     * The parameters for the motor data are stored in the index 6410.
     * @param subindex
     * @param sizeInHexa size of the parameter in Can Open
     * @param value FORMAT=int the value of the parameter in decimal format
     */
    protected void writeMotorData(String subindex, String sizeInHexa, int value) throws SDORequestError {
        String valueInHexa = Integer.toHexString(value);
        writeSDO("6410",subindex,sizeInHexa,valueInHexa);
    }
    
    /**
     * This methods reads the parameters of the motor stored in the actuator.
     * (index 6410)
     * @param subindex
     * @return value FORMAT=int the value of the parameter in decimal format
     */
    public int readMotorData(String subindex) throws SDORequestError {
        //TODO this should return a short or a long as it deviceErrorFile coded with 2 or 4 bytes.
        String valueInHexa = readSDO("6410",subindex);
        return Integer.parseInt(valueInHexa, 16);
    }
    
    /**
     * This methods read the parameters of the motor stored in the actuator 
     * (hardware configuration)and compare with the configuration stored in the
     * Configuration Data Base (software configuration).
     * @return true if the hardware and software configuration are the same,
     *         false otherwise.
     * @throws MismatchMotorConfigurationError 
     */
    public boolean checkMotorParameters() throws MismatchMotorConfigurationException, SDORequestError {
        boolean ok = true;
              
        //motor type
        int readMotorType = readMotorType();
        if (!(readMotorType == motor.getMotorType())) ok = false;
        
        //motor type
        int readPositionSensorType = readPositionSensorType();
        if (!(readPositionSensorType == motor.getPositionSensorType())) ok = false;
        
        //motor data
        int readContinuousCurrentLimit = readMotorData("01");
        if (!(readContinuousCurrentLimit == motor.getContinuousCurrentLimit())) ok = false;
        
        int readMaxSpeedInCurrentMode = readMotorData("04");
        if (!(readMaxSpeedInCurrentMode == motor.getMaxSpeedInCurrentMode())) ok = false;
        
        int readThermalTimeConstantWinding = readMotorData("05");
        if (!(readThermalTimeConstantWinding == motor.getThermalTimeConstantWinding())) ok = false;
        
        int readOutputCurrentLimit = readMotorData("02");
        if (!(readOutputCurrentLimit == motor.getOutputCurrentLimit())) ok = false;
        
        int readPolePairNumber = readMotorData("03");
        if (!(readPolePairNumber == motor.getPolePairNumber())) ok = false;
        
        if (!ok) {
            throw new MismatchMotorConfigurationException("ERROR in hardware configuration of the motor",
                    readMotorType,readContinuousCurrentLimit,readMaxSpeedInCurrentMode,
                    readThermalTimeConstantWinding,readOutputCurrentLimit,readPolePairNumber);           
        }
        return ok;
    }
    
    public String displayMotorParameters() throws SDORequestError {
        StringBuilder sb = new StringBuilder("Read decimal values for motor parameters are :");
              
        //motor type
        int readMotorType = readMotorType();
        
        //position sensor type
        int readPositionSensorType = readPositionSensorType();
        
        //motor data
        int readContinuousCurrentLimit = readMotorData("01");      
        int readMaxSpeedInCurrentMode = readMotorData("04");
        int readThermalTimeConstantWinding = readMotorData("05");       
        int readOutputCurrentLimit = readMotorData("02");
        int readPolePairNumber = readMotorData("03");

        sb.append(" motor type ="); sb.append(readMotorType);
        sb.append(" position sensor type ="); sb.append(readPositionSensorType);
        sb.append(" continuousCurrentLimit ="); sb.append(readContinuousCurrentLimit);
        sb.append(" maxSpeedInCurrentMode ="); sb.append(readMaxSpeedInCurrentMode);
        sb.append(" thermalTimeConstantWinding ="); sb.append(readThermalTimeConstantWinding);
        sb.append(" outputCurrentLimit ="); sb.append(readOutputCurrentLimit);
        sb.append(" polePairNumber ="); sb.append(readPolePairNumber);
        return sb.toString();
    }
    
    /*Methods available in CURRENT mode*/ 
    /**
     * In current mode this methods send a current to the motor.
     * This make run the motor.
     * @param aValue UNIT=mA / FORMAT=decimal the value of the current to be sent.
     * @throws BadCommandException 
     * 
     */
    public void writeCurrent(int aValue) throws BadCommandException, SDORequestError {
        if (!isEnable()) 
            throw new BadCommandException(getName() + "is not ENABLE");
        if (!mode.equals(ActuatorMode.CURRENT)) 
            throw new BadCommandException(getName() + "is not in CURRENT mode");       
        if (aValue > motor.getContinuousCurrentLimit())
            throw new IllegalArgumentException(aValue + " is greater than Continuous Current Limit");
        
        motor.setCurrentValue(aValue);
        String currentValueInHexa = Integer.toHexString(aValue);
        writeSDO("2030","0","2",currentValueInHexa);
            //TODO check the return code of writeSDO
    }
    
    /**
     * In Current Mode this methods returns the current actualy received by the motor.
     * @return the current UNIT=mA / FORMAT=decimal 
     * @throws BadCommandException 
     */
    public int readCurrent() throws BadCommandException, SDORequestError {
        if (mode.equals(ActuatorMode.CURRENT)) {
            String currentInHexa = readSDO("6078","0");
            return Integer.parseInt(currentInHexa, 16);
        } else throw new BadCommandException(getName() + "is not in CURRENT mode");
    }
    
    /**
     * In current mode this methods set to zero the value of the current sent to
     * the motor.
     * This stops the motor.
     * @throws BadCommandException 
     */
    public void stopCurrent() throws BadCommandException, SDORequestError {
        if (mode.equals(ActuatorMode.CURRENT)) {
            writeSDO("2030","00","2","0");
        } else throw new BadCommandException(getName() + "is not in CURRENT mode");
        
    }
    
        /*Methods available in VELOCITY mode*/ 
    public void setMaxiVelocity(int velocity) {
        if (velocity > getMotor().maxProfileVelocity) 
            throw new IllegalArgumentException(getName() + ": can't set parameters greater than maxi value");
        motor.maxProfileVelocity = velocity;
        //writeSDO("607F","00",velocity);
    }
    
    public void setMaxiAcceleration(int acceleration) {
        if (acceleration > getMotor().maxAcceleration ) 
            throw new IllegalArgumentException(getName() + ": can't set parameters greater than maxi value");
        motor.maxAcceleration = acceleration;
        //writeMaxonMotor(0x60C5,00,acceleration);
    }
    
    public void setVelocity() throws BadCommandException {
        if (mode.equals(ActuatorMode.VELOCITY)) {
            //writeMaxonMotor(0x206B,00,this.velocityValue);
        } else throw new BadCommandException(getName() + "is not in VELOCITY mode");

    }
    
    public int readVelocity()throws BadCommandException {
        if (mode.equals(ActuatorMode.VELOCITY)) {
            //String velocityInHexa = readSDO(0x2028,00);
            return 0;
        } else throw new BadCommandException(getName() + "is not in VELOCITY mode");
        
    } 
    
    public void stopVelocity() throws BadCommandException {
        if (mode.equals(ActuatorMode.VELOCITY)) {
            //writeMaxonMotor(0x206B,00,0);
        } else throw new BadCommandException(getName() + "is not in VELOCITY mode");
        
    }
    
    /****************************************************************/
    /* The following methods are common to all the Can Open hardware*/
    /****************************************************************/
    
    public String writeSDO(String index, String subindex,String length, String newValue) throws SDORequestError {
        return tcpProxy.writeSDO(this.nodeID,index,subindex,length,newValue);
    }
    
    public String readSDO(String index, String subindex) throws SDORequestError {
        return tcpProxy.readSDO(nodeID, index, subindex);
    }
    
    
    


    
    @Override
    
    public String toString() {
        StringBuilder sb = new StringBuilder(getName());
        sb.append("/");
        sb.append(this.serialNB);
        sb.append("/");
        if (isConfigOK())  {
            sb.append("is booted");
            sb.append("/CanOpen NodeID in hexa= ");
            sb.append(this.nodeID);
            int x = Integer.getInteger(this.nodeID, 16);
            sb.append(" in decimal= ");sb.append(x);
        } else {
            sb.append("NOT DETECTED");
        }
        return sb.toString();
    }

}
