/*
 * To change this template, choose Tools | Templates
 * and open the template in the editor.
 */
package org.lsst.ccs.subsystems.fcs.drivers;

import java.util.concurrent.TimeoutException;
import org.lsst.ccs.framework.Module;
import org.lsst.ccs.subsystems.fcs.common.PieceOfHardware;
import org.lsst.ccs.subsystems.fcs.errors.HardwareNotDetectedException;
import org.lsst.ccs.subsystems.fcs.errors.SDORequestError;

/**
 * 
 * This is the driver class for the ADC we use in the FCS test bench and in SINGLE FILTER TEST
 * CAN-CBX-AI814
 * 
 * @author virieux
 */
public class CanOpenADC extends Module implements PieceOfHardware {
    
    CanOpenProxy tcpProxy;
    
    String serialNB;
    String nodeID;
    
    /**
    * One bit read on the input of the ADC represents 0.3125 mV
    */
    public static final double adcVoltsPerBit = 0.3125;
    public static final double waterFreezeTemp = 273.15;
    
    //FOR SPRING
    public CanOpenADC() {       
    }

    public CanOpenADC(String nodeID, String serial) {
        this.tcpProxy = (CanOpenProxy) this.getModule("tcpProxy");
        this.serialNB = serial;
        this.nodeID = nodeID;       
    }

    public String getSerialNB() {
        return this.serialNB;
    }

    public String getNodeID() {
        return this.nodeID;
    }


    /**
     * @param serialNB the serialNB to set
     */
    public void setSerialNB(String serialNB) {
        this.serialNB = serialNB;
    }

    /**
     * @param nodeID the nodeID to set
     */
    public void setNodeID(String nodeID) {
        this.nodeID = nodeID;
    }

    @Override
    public boolean isConfigOK() {
        return tcpProxy.isConfigOK(this);
    }
    
    @Override
    public void initModule() {
        tcpProxy = (CanOpenProxy) this.getModule("tcpProxy");
    }
    
    /**
     * Read the analog input whom number is given as a parameter
     * @param input : integer from 1 to 8
     * @return the value read in hexa (on 16bits)
     * @throws SDORequestError 
     */
    public String readAnalogInput(int inputNB) throws SDORequestError, HardwareNotDetectedException {

        log.debug(getName() + " IS BOOTED AND CONFIG OK =" + isConfigOK());
        if (this.tcpProxy.isBooted(this)) {
             
            //int inputNB = Integer.parseInt(input, 16);
            if ((inputNB < 1) || (inputNB > 8)) 
                throw new IllegalArgumentException("CAN-CBX-AI814 : input has to be 1..8");
            String input = Integer.toHexString(inputNB);
            return tcpProxy.readSDO(nodeID, "6401", input);
            
        } else {
            String message = String.format("Module %s / nodeID=%s is UNREACHABLE", getName(),nodeID);
            log.error(message);
            //this.getSubsystem().updateState(State.InError, message );
            throw new HardwareNotDetectedException(getName() + message); 
            
        }
    }
    
    public String readInputs() throws TimeoutException {
        return tcpProxy.sendCanOpen("srtr,1401,1402");
    }

    public void initialize() {
        throw new UnsupportedOperationException("Not supported yet.");
    }
    
    public String toString() {
        StringBuilder sb = new StringBuilder(getName());
        sb.append("/");
        sb.append(this.serialNB);
        sb.append("/");
        if (isConfigOK())  {
            sb.append("is booted");
            sb.append("/CanOpen NodeID in hexa= ");
            sb.append(this.nodeID);
            int x = Integer.getInteger(this.nodeID, 16);
            sb.append(" in decimal= ");sb.append(x);
        } else {
            sb.append("NOT DETECTED");
        }
        return sb.toString();
    }
    
    

    
    
    
}
