package org.lsst.ccs.subsystems.fcs.common;

import org.lsst.ccs.bus.BadCommandException;
import org.lsst.ccs.bus.ErrorInCommandExecutionException;
import org.lsst.ccs.subsystems.fcs.CarouselSocket;
import org.lsst.ccs.subsystems.fcs.Filter;
import org.lsst.ccs.subsystems.fcs.errors.HardwareErrorDetectedException;

/**
 * This interface is a model for a carousel.
 * The Filter carousel holds 5 filters. The filters are held in 5
 * sockets around the carousel.
 * <UL>
 *  <LI/>It can rotate and be stopped.
 *  <LI/>It can give its position.
 *  <LI/>It can be locked or unlocked.
 *  <LI/>It can release its clamps at standby position.
 *  <LI/>It can unlock its clamps at standby position.
 *  <LI/>It can return the socket which is at standby position.
 * 
 * </UL>
 * @author virieux
 */
public interface Carousel {
	
    /**
     * The carousel position measures the rotation angle of the carousel within
     * the fixed part of the camera.
     * @return The position of the carousel in degrees.
     * Its values is greater or equal to 0 and less than 360.
     */
    public double getPosition();

    /**
     * The carousel can be locked by a break or prevented to move by the 
     * autochanger fliprail. The carousel is able to move if no piece of 
     * hardware (nor simulated hardware) prevents the carousel to move.
     * @return  true if the carousel is able to move,
     *          false if not.
     */
    public boolean isAbleToMove();

    /**
     * This method is a command that can be sent to the carousel. It make the 
     * carousel rotating for a given angle.
     * @param angle :   given in degree 
     *                  available values for an angle : [-360,+360]
     * @return 
     * @throws IllegalArgumentException if the angle given as parameter 
     * is not in the ranges of available values.
     * @throws BadCommandException if the carousel is already rotating or is 
     * unable to move.
     */
    public String rotate(double angle) throws IllegalArgumentException, BadCommandException;

    /**
     * Stop the carousel if it is rotating.
     * @return 
     */
    public String stop ();

    /**
     * Lock the carousel (or engage the break).
     * @return 
     */
    public String lock();

    /**
     * Unlock the carousel (or disengage the break)
     * @return 
     */
    public String unlock();

    /**
     * This methods returns the socket at standby position or null if there is no
     * socket at standby position. If the carousel is rotating, it returns null.
     * @return the socket which is at standby position or null if there is no socket 
     * at standby position.
     */
    public CarouselSocket getSocketAtStandby();

    /**
     * This is a command that can be sent to the carousel.
     * It releases (unlock it) the clamps at standby position. 
     * (when a clamp is released, it can be locked automaticaly when a filter 
     * comes in the socket)
     * @return
     * @throws BadCommandException
     * @throws ErrorInCommandExecutionException 
     */
    public String releaseClamps() throws BadCommandException, ErrorInCommandExecutionException;

    public String unlockClamps(Filter filter) throws BadCommandException, ErrorInCommandExecutionException, HardwareErrorDetectedException;
    
    /**
     * Before rotating the carousel we have to release the power contact between the
     * moving part and the fixed part.
     */
    public abstract void releaseClampsContact();

    
    /**
     * After a rotation, when the carousel is stopped the power contact can be engaged
     * again.
     */
    public abstract void engageClampsContact();

}
