package org.lsst.ccs.subsystems.fcs;

import java.util.concurrent.locks.Condition;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.lsst.ccs.bus.BadCommandException;
import org.lsst.ccs.bus.ErrorInCommandExecutionException;
import org.lsst.ccs.framework.Module;
import org.lsst.ccs.subsystems.fcs.FcsEnumerations.FilterPresenceStatus;
import org.lsst.ccs.subsystems.fcs.FcsEnumerations.LockStatus;
import org.lsst.ccs.subsystems.fcs.common.Actuator;
import org.lsst.ccs.subsystems.fcs.FcsEnumerations.FilterClampState;
import org.lsst.ccs.subsystems.fcs.common.BridgeToHardware;
import org.lsst.ccs.subsystems.fcs.errors.SensorValueOutOfRangeException;
import org.lsst.ccs.subsystems.fcs.utils.FcsUtils;

/**
 * An abstract class which extends Module to model a clamp that holds a filter on the carousel.
 * It implements pattern clampState.
 * See <a href="CarouselClampState.png"> the clampState machine diagram for a carousel clamp.</a>
 * Each clamp on the carousel is coupled with a sensor which detects the presence of a filter.
 * This abstract class provides a method to lock, unlock or release the clamps, and to compute the clampState 
 * of the clamp. 
 * The clampState of the clamp is computed, updated and published on the status bus each tick of the timer.
 * 
 * 
 * @author virieux
 */
public abstract class FilterClampModule extends Module {


    /**
     * The actuator which controls this clamp.
     */
    ClampActuatorModule actuator;
    
     /**
     * The filterPresenceSensor detects where is the filter within the clamp.
     * @See enum PresenceFilterStateOnClamp
     */
    private Sensor14bits filterPresenceSensor;
    
    
    /**
     * The lock sensor detects if the clamp is locked or not.
     */
    private Sensor14bits lockSensor;

    protected FilterClampState clampState;
    protected FilterPresenceStatus filterPresenceStatus;
    private LockStatus lockStatus;
    
    protected int temperature;
    
    /**
     * If the lock sensor returns a value between 0 and lockSensorValueA:
     * the sensor is unlocked.
     * If the lock sensor returns a value between lockSensorValueA and lockSensorValueB:
     * we don't know.
     * if the sensor returns a value between lockSensorValueB and ValueC:
     * the clamp is locked.
     * If the sensor returns a value greater than valueC the sensor is in ERROR.
     */
    private int lockSensorValueA;
    private int lockSensorValueB;
    private int lockSensorValueC;
    
    /**
     * The min and max value for the sensors.
     * Needed by the GUI.
     */
    private int lockSensorMinValue;
    private int lockSensorMaxValue;
    private int filterPositionMinValue;
    private int filterPositionMaxValue;
    
    
    /**
     * if the filter presence sensor returns a value between 0 and filterPositionValueA : 
     * the sensor is in error.
     */
    //@ConfigChanger( doc= "if the filter presence sensor returns a value between 0 and filterPositionValueA : the sensor is in error.")
    private int filterPositionValueA;
    
    /**
     * if filter presence sensor returns a value between filterPositionValueA and filterPositionValueB : 
     * the filter is engaged and lockable.
     */
    private int filterPositionValueB;
    
     /**
     * if filter presence sensor returns a value between filterPositionValueB and filterPositionValueC : 
     * we don't know. And if the value is greater than valueC, there's no filter.
     */   
    private int filterPositionValueC;
    

    
    
    public String publishedByClampOutputName = "publishedByClamp";
    
    private boolean initialized = false;
    
    
    //Used because we have to wait for the update from the sensors to know the clampState
    //of the clamp
    final Lock lock = new ReentrantLock();
    final Condition stateUpdated = lock.newCondition();
    
     /* This is used when we update the clamp clampState with the values returned 
     *  by the sensors.
     */
    protected volatile boolean updatingState = false;



    
    /**************************************************************************************************/
    /********************** SETTERS AND GETTERS  ******************************************************/
    /**************************************************************************************************/

    public ClampActuatorModule getActuator() {
        return actuator;
    }

    public void setActuator(ClampActuatorModule actuator) {
        this.actuator = actuator;
    }
    
    
    /**
     * @param filterPositionValueA the filterPositionValueA to set
     */
    public void setFilterPositionValueA(int valueA) {
        this.filterPositionValueA = valueA;
    }

    /**
     * @param filterPositionValueB the filterPositionValueB to set
     */
    public void setFilterPositionValueB(int valueB) {
        this.filterPositionValueB = valueB;
    }

    /**
     * @param filterPositionValueC the filterPositionValueC to set
     */
    public void setFilterPositionValueC(int valueC) {
        this.filterPositionValueC = valueC;
    }


    /**
     * Returns the clampState of the clamp. 
     * If the clampState is being updated and waiting 
     * for a response from a sensor, this methods waits until the clampState is updated.
     * If the clampState is not being updated, it returns immediatly the clampState.
     * 
     * @return clampState
     * 
     */

    public FilterClampState getClampState() {
        
        lock.lock();
        try {
            while(updatingState) {
                try {
                    this.stateUpdated.await();
                } catch (InterruptedException ex) {
                    Logger.getLogger(FilterClampModule.class.getName()).log(Level.SEVERE, null, ex);
                }

            }
            return clampState;
            
        } finally {
            lock.unlock();
        }
    }



    /**
     * @return the lockSensorValueA
     */
    public int getLockSensorValueA() {
        return lockSensorValueA;
    }

    /**
     * @param lockSensorValueA the lockSensorValueA to set
     */
    public void setLockSensorValueA(int aValue) {
        this.lockSensorValueA = aValue;
    }

    /**
     * @return the lockSensorValueB
     */
    public int getLockSensorValueB() {
        return lockSensorValueB;
    }

    /**
     * @param lockSensorValueB the lockSensorValueB to set
     */
    public void setLockSensorValueB(int aValue) {
        this.lockSensorValueB = aValue;
    }

    public int getLockSensorValueC() {
        return lockSensorValueC;
    }

    public void setLockSensorValueC(int lockSensorValueC) {
        this.lockSensorValueC = lockSensorValueC;
    }

    public int getFilterPositionMaxValue() {
        return filterPositionMaxValue;
    }

    public void setFilterPositionMaxValue(int filterPositionMaxValue) {
        this.filterPositionMaxValue = filterPositionMaxValue;
    }

    public int getFilterPositionMinValue() {
        return filterPositionMinValue;
    }

    public void setFilterPositionMinValue(int filterPositionMinValue) {
        this.filterPositionMinValue = filterPositionMinValue;
    }

    public int getLockSensorMaxValue() {
        return lockSensorMaxValue;
    }

    public void setLockSensorMaxValue(int lockSensorMaxValue) {
        this.lockSensorMaxValue = lockSensorMaxValue;
    }

    public int getLockSensorMinValue() {
        return lockSensorMinValue;
    }

    public void setLockSensorMinValue(int lockSensorMinValue) {
        this.lockSensorMinValue = lockSensorMinValue;
    }
 
    

    /**
     * @return the filterPositionValueA
     */
    public int getFilterPositionValueA() {
        return filterPositionValueA;
    }

    /**
     * @return the filterPositionValueB
     */
    public int getFilterPositionValueB() {
        return filterPositionValueB;
    }

    /**
     * @return the filterPositionValueC
     */
    public int getFilterPositionValueC() {
        return filterPositionValueC;
    }
    
     /**
     * For encapsulation : we want to access to filterPresenceSensor in sub-packages.
     * @return the filterPresenceSensor
     */
    public Sensor14bits getFilterPresenceSensor() {
        return filterPresenceSensor;
    }

    /**
     * for Spring
     * @param filterPresenceSensor the filterPresenceSensor to set
     */
    public void setFilterPresenceSensor(Sensor14bits filterPresenceSensor) {
        this.filterPresenceSensor = filterPresenceSensor;
    }
    
        /**
     * @forSpring
     * @return the lockSensor
     */
    public Sensor14bits getLockSensor() {
        return lockSensor;
    }
    
        /**
     * @forSpring
     * @param lockSensor the lockSensor to set
     */
    public void setLockSensor(Sensor14bits lockSensor) {
        this.lockSensor = lockSensor;
    }

    /**
     * @return the lockStatus
     */
    public LockStatus getLockStatus() {
        return lockStatus;
    }

    /**
     * @param lockStatus the lockStatus to set
     */
    public void setLockStatus(LockStatus lockStatus) {
        this.lockStatus = lockStatus;
    }

    /**
     * This methods returns true if the clamp is locked.
     * In the simulator, it is overriden.
     * @return 
     */
    public boolean isLocked() {
        return (this.getLockStatus().equals(LockStatus.LOCKED));
    }

    public FilterPresenceStatus getFilterPresenceStatus() {
        return filterPresenceStatus;
    }

    public int getTemperature() {
        return temperature;
    }

    /**************************************************************************************************/
    /********************** END OF SETTERS AND GETTERS  ***********************************************/
    /**************************************************************************************************/
    
    @Override
    public void initModule() {
        this.clampState = FilterClampState.UNDEFINED;
        this.filterPresenceStatus = FilterPresenceStatus.UNKNOWN;
        this.lockStatus = LockStatus.UNKNOWN;
        this.temperature = 0;
    }
    
    abstract public double readTemperature();
    

    public synchronized boolean isFilterEngaged() {
        return (this.filterPresenceStatus.equals(FilterPresenceStatus.ENGAGED));
    }


    /**
     * This method updates the clamp clampState regarding the value returned by the 
     * filter presence sensor and the value returned by the method isLocked().
     * When the update is completed, it sends a signal to threads waiting to get
     * the new value of clampState.
     */
    

    public void updateStateWithSensors() {
        
        lock.lock();
        updatingState = true;   
        
        try {
            //READ filter presence sensor
            this.filterPresenceSensor.updateValue();
            int newFilterPresenceSensorValue = filterPresenceSensor.getValue();
            if (newFilterPresenceSensorValue < this.filterPositionValueA) {
                this.filterPresenceStatus = FilterPresenceStatus.ERROR;
                    this.filterPresenceStatus = FilterPresenceStatus.ERROR;
                    throw new SensorValueOutOfRangeException(this.filterPresenceSensor,(int) newFilterPresenceSensorValue);
                
            } else if (newFilterPresenceSensorValue < this.filterPositionValueB) {
                this.filterPresenceStatus = FilterPresenceStatus.ENGAGED;     
                
            } else if (newFilterPresenceSensorValue < this.filterPositionValueC) {
                this.filterPresenceStatus = FilterPresenceStatus.UNKNOWN;
                
            } else {
                this.filterPresenceStatus = FilterPresenceStatus.NOFILTER;
            } 
            
            
            //READ lock sensor
            this.lockSensor.updateValue();
            
            int newLockSensorValue = lockSensor.getValue();
   
            //log.debug(getName() + "/Filter Presence Sensor value=" + this.filterPresenceSensor.getValue());
            log.debug(getName() + " NEWVALUE FOR LOCK SENSOR=" + newLockSensorValue);
                
            if ( newLockSensorValue < lockSensorValueA) {
                this.lockStatus = LockStatus.UNLOCKED;
                
            } else if (newLockSensorValue <= lockSensorValueB){
                this.lockStatus = LockStatus.UNKNOWN;
                
                
            } else if (newLockSensorValue <= lockSensorValueC) {
                this.lockStatus = LockStatus.LOCKED;
                
            } else {
                this.lockStatus = LockStatus.ERROR;
                    log.error(getName() + " ERROR new value read = " + newLockSensorValue);
                    throw new SensorValueOutOfRangeException(this.lockSensor,(int) newFilterPresenceSensorValue);
            }

            this.clampState = this.computeClampState();
                  
            
        } catch (SensorValueOutOfRangeException ex) {
            log.error(ex.getMessage());
            //getSubsystem().updateState(State.InError, getName() + "ERROR IN READING LOCK SENSOR " + ex.getMessage());
        } finally {
            updatingState = false;
            stateUpdated.signal();
            lock.unlock();
        }
        
        this.sendToStatus(this.getStatusData());

    }
    
    
    
    /**
     * Compute the global state of the clamp given the lock sensor and the presence filter sensor state.
     * This has to be defined overriden for the clamp X-.
     * @return clamp state
     */
    public FilterClampState computeClampState() {
        
        
        if (this.filterPresenceStatus.equals(FilterPresenceStatus.ERROR) 
                || (this.lockStatus.equals(LockStatus.ERROR))) 
            return  FilterClampState.ERROR;

        // a filter is in the socket
        if (this.filterPresenceStatus.equals(FilterPresenceStatus.ENGAGED)) {
            //if we replace isLocked() by this.getLockStatus().equals(LockStatus.LOCKED)
            //the simulator doesn't work anymore 
            //TODO : see if we can change that ?
            if (this.lockStatus.equals(LockStatus.LOCKED)) {
                return FilterClampState.CLAMPEDONFILTER;

            } else if (this.lockStatus.equals(LockStatus.UNLOCKED)) {
                return  FilterClampState.UNCLAMPEDONFILTER;
            } else {
                return FilterClampState.UNDEFINED;
            }
        }

        // no filter
        if (this.filterPresenceStatus.equals(FilterPresenceStatus.NOFILTER)) {
            if (this.lockStatus.equals(LockStatus.LOCKED)) {
                return  FilterClampState.READYTOCLAMP;

            } else if (this.lockStatus.equals(LockStatus.UNLOCKED)) {
                return  FilterClampState.UNCLAMPEDEMPTY;

            } else {
                return  FilterClampState.ERROR;
            }
        }

        // UNKNOWN (UNDEFINED)
        if (this.filterPresenceStatus.equals(FilterPresenceStatus.UNKNOWN)) {
            return FilterClampState.UNDEFINED;
        }
        
        return FilterClampState.UNDEFINED;

    }
    

      
//    @Override
//    public void startTicking() {
//        //TODO when the carousel is not at standby position, it's not necessary to read the sensors.
//        updateStateWithSensors(); // we can't do that before all the hardware has been connected and is OK
//        super.startTicking();
//    }
   

    @Override
    public void tick() {
        if (initialized) {
            //this.updateStateWithSensors();
            //sendToStatus(getStatusData());
        } else {
            if (((BridgeToHardware) this.getModule("bridge")).isHardwareReady()) {
                initialized = true;
            }
        }

    }


    public StatusDataPublishedByClamp getStatusData() {
        StatusDataPublishedByClamp status = FcsUtils.createStatusDataPublishedByClamp(this);
        return status;
    }


    /**
     * The clamps on the carousel are locked automaticaly when the filter comes
     * at the standby position. To be able to lock automaticaly again, it has to be
     * released after each time it has been unlocked.
     *
     * @return
     * @throws BadCommandException
     * @throws ErrorInCommandExecutionException
     */

    public String release() throws BadCommandException, ErrorInCommandExecutionException {
        log.debug("Releasing " + getName() + " on socket at standby position.");
        updateStateWithSensors();

        if (!getClampState().equals(FilterClampState.UNCLAMPEDEMPTY)) 
            throw new BadCommandException("Can't release a clamp if isn't unclamped and empty.");
        
        actuator.off();
        
        updateStateWithSensors();
        
        while(updatingState) {
            try {
                stateUpdated.await();
            } catch (InterruptedException ex) {
                Logger.getLogger(FilterClampModule.class.getName()).log(Level.SEVERE, null, ex);
            }
        }
        
        
        if (getClampState().equals(FilterClampState.READYTOCLAMP)) {
            return getName() + " is released";
        } else {
            throw new ErrorInCommandExecutionException(getName() + ": Could not release clamp.");
        }
     
        
    }

    /**
     * Unlock the clamp when a filter is locked by the clamp.
     * @return
     * @throws BadCommandException
     * @throws ErrorInCommandExecutionException
     */
    public String unlock() 
            throws BadCommandException, ErrorInCommandExecutionException {
        
        log.debug(getName() + ": "
                + "UNLOCK State1 = " + clampState.toString());
        updateStateWithSensors();
        
        if (!getClampState().equals(FilterClampState.CLAMPEDONFILTER)) 
            throw new BadCommandException("Can't unlock a clamp if isn't clamped on filter.");
        
       
        this.actuator.on();
        
        updateStateWithSensors();
        
        while(updatingState) {
            try {
                stateUpdated.await();
            } catch (InterruptedException ex) {
                Logger.getLogger(FilterClampModule.class.getName()).log(Level.SEVERE, null, ex);
            }
        }
               
        
        log.debug(getName() + ":UNLOCK State2 = " + clampState.toString());
        
        if (getClampState().equals(FilterClampState.UNCLAMPEDONFILTER)) {
            this.actuator.maintainCurrent();
            return getName() + " is unlocked";
            
        } else {
            throw new ErrorInCommandExecutionException(getName() + ": Could not unlock clamp.");
        }
        
    }    
    
	
}
