
package org.lsst.ccs.subsystems.fcs;

import java.util.concurrent.locks.Condition;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.lsst.ccs.bus.BadCommandException;
import org.lsst.ccs.bus.ErrorInCommandExecutionException;
import org.lsst.ccs.subsystems.fcs.common.Actuator;
import org.lsst.ccs.subsystems.fcs.FcsEnumerations.FilterClampState;
import org.lsst.ccs.subsystems.fcs.common.FilterLocation;
import org.lsst.ccs.subsystems.fcs.errors.HardwareErrorDetectedException;


/**
 * This is a socket on the carousel : there is 5 sockets on a carousel.
 * When a filter is on the carousel, it is attached at a socket.
 * A socket has a clamp on each side : 
 * - clampX- (clampXminus)
 * - clampX+ (clampXplus)
 * clampX+ and clampX- are the same except for the method isLocked()
 * See FilterClampXminusModule and FilterClampXplusModule
 * 
 * The main job of this class CarouselSocket is to synchronize the actions on the 2 clamps.
 * The state of the CarouselSocket is the state of the clamps when both clamps state are 
 * identical. 
 * To synchonize an action (updateSateWithSensor, unlock or release) on both clamps, we use a CyclicBarrier :
 * we start 2 threads and they wait for each other action completion with a CyclicBarrier.
 * 
 * During an action on both clamps, we want to lock the CarouselSocket so no other thread can 
 * try to access this object before the completion of action. We do that with a Lock and Condition.
 * (see java.util.concurrent)
 * 
 * @author virieux
 *
 */
public class CarouselSocket {


    private FilterClampModule clampXminus;

    private FilterClampModule clampXplus;

    /**
     * socket position (angle) in degrees on the carousel
     * 
     */
    private double position;

    /**
     * carousel position (angle) when this socket is at standby position
     * should be 360 - position, but could be different
     */
    private double standbyPosition;

    /**
     * When the socket holds a filter, this field is initialized whith this filter.
     * When the socket is empty, this field is null.
     */
    private Filter filter;
    
    volatile boolean updatingClamps = false;
    volatile boolean releasingClamps = false;
    volatile boolean unlockingClamps = false;
    private volatile FilterClampState clampsState = FilterClampState.CLAMPEDONFILTER;
    
    //When we update the 2 clamps, we want to take a lock on the socket until
    //the end of the update, so any other thread has to wait for this update.
    final Lock lock = new ReentrantLock();
    final Condition stateUpdated = lock.newCondition();
    final Condition releaseCompleted = lock.newCondition();
    final Condition unlockCompleted = lock.newCondition();


    public CarouselSocket() {}

    /**
     * This constructor is for tests.
     * @param position
     */
    public CarouselSocket(double position) {
            this.position = position;
            if (position == 0) {
                    standbyPosition = 0;
                    } else {
                            standbyPosition = 360 - position;
                    }
    }


    
    /**************************************************************************************************/
    /********************** SETTERS AND GETTERS  ******************************************************/
    /**************************************************************************************************/
    

    public FilterClampModule getClampXminus() {
        return clampXminus;
    }

    public void setClampXminus(FilterClampModule clampXminus) {
        this.clampXminus = clampXminus;
    }

    public FilterClampModule getClampXplus() {
        return clampXplus;
    }

    public void setClampXplus(FilterClampModule clampXplus) {
        this.clampXplus = clampXplus;
    }


    public double getPosition() {
            return position;
    }
    
    ////FOR SPRING
    public void setPosition(double position) {
            this.position = position;
    }

    public Filter getFilter() {
            return filter;
    }

    protected void setFilter(Filter filter) {
            this.filter = filter;
    }

    public double getStandbyPosition() {
            return standbyPosition;
    }

    //FOR SPRING has to be public but it's stupid
    public void setStandbyPosition(double standbyPosition) {
            this.standbyPosition = standbyPosition;
    }
        /**
     * Returns the state of the clamp. 
     * If the state is being updated and waiting 
     * for a response from a sensor, this methods waits until the state is updated.
     * If the state is not being updated, it returns immediatly the state.
     * 
     * @return state
     * 
     */
    
    /**************************************************************************************************/
    /********************** END OF SETTERS AND GETTERS  ***********************************************/
    /**************************************************************************************************/

    public FilterClampState getState() {
        
        lock.lock();
        try {
            while(updatingClamps) {
                try {
                    this.stateUpdated.await();
                } catch (InterruptedException ex) {
                    Logger.getLogger(FilterClampModule.class.getName()).log(Level.SEVERE, null, ex);
                }

            }
            return clampsState;
            
        } finally {
            lock.unlock();
        }
    }
    
    
    
    /**
     * This method updates the state of the 2 clamps in reading the values sent by the sensors.
     * The 2 threads wait for each other until the 2 updates are completed.
     * When the 2 updates are completed, the method clampsUpdateCompleted() is executed.
     * 
     * @throws HardwareErrorDetectedException 
     */
    public void updateClampsStateWithSensors() throws HardwareErrorDetectedException {

        
        lock.lock();
        this.updatingClamps = true;
        try {
            clampXminus.updateStateWithSensors();
            clampXplus.updateStateWithSensors();
            
            if (clampXminus.getClampState().equals(clampXplus.getClampState()))  {
                    this.clampsState = clampXminus.getClampState();
                    System.out.println("XXXXXXXXSocket at standby : clamps are updated");

            } else throw new HardwareErrorDetectedException("Error in filter presence detection at standby position : "
                        + "the clamps don't agree.");
            
            
        } finally {
            this.updatingClamps = false;
            this.stateUpdated.signal();         
            lock.unlock();
        }
    }
    


    /*
     * The carousel socket is empty if there is no filter in the socket.
     * That appends when clamps of both side of the socket are empty.
     * A clamp is empty when no filter is engaged in the clamp.
     * @throws HardwareErrorDetectedException
     * If the 2 clamps are not in the same state this method throws
     * a HardwareErrorDetectedException : this should not append except if 
     * the sensors are broken.
     */
    public boolean isEmpty() throws HardwareErrorDetectedException {
            return (getState().equals(FilterClampState.READYTOCLAMP)) 
                    || (getState().equals(FilterClampState.UNCLAMPEDEMPTY));      
    }
   

    public boolean isClampedOnFilter() {
        return getState().equals(FilterClampState.CLAMPEDONFILTER);
    }
    
    public boolean isUnclampedOnFilter() {
        return getState().equals(FilterClampState.UNCLAMPEDONFILTER);
    }

    public boolean isUnclampedEmpty() {
        return getState().equals(FilterClampState.UNCLAMPEDEMPTY);
    }
    
    public boolean isReadyToClamp() {
        return getState().equals(FilterClampState.READYTOCLAMP);
    }
    
    
    
    /**
     * Releases the 2 clamps of the socket.
     * @param actuatorXminus
     * @param actuatorXplus
     * @throws BadCommandException
     * @throws ErrorInCommandExecutionException
     * @throws HardwareErrorDetectedException 
     */
    public void releaseClamps() throws BadCommandException, ErrorInCommandExecutionException, HardwareErrorDetectedException {
           
        lock.lock();
        this.releasingClamps = true;

        try {
            clampXminus.release();
            clampXminus.release();
            
            if (clampXminus.getClampState().equals(FilterClampState.READYTOCLAMP) & clampXplus.getClampState().equals(FilterClampState.READYTOCLAMP))  {
                this.clampsState = FilterClampState.READYTOCLAMP;
                System.out.println("XXXXXXXXSocket at standby : clamps are released");
                //this.stateUpdated.signal();
                this.releaseCompleted.signal();

            } else if (!clampXminus.getClampState().equals(clampXplus.getClampState())) {
                throw new HardwareErrorDetectedException("Error in filter presence detection at standby position : "
                    + "the clamps don't agree.");
            } else if (!clampXminus.getClampState().equals(FilterClampState.READYTOCLAMP)) {
                throw new ErrorInCommandExecutionException("Could not release clampX-");
            } else if (!clampXplus.getClampState().equals(FilterClampState.READYTOCLAMP)) {
                throw new ErrorInCommandExecutionException("Could not release clampX+");
            }
            
        } finally {
            this.releasingClamps = false;
            this.releaseCompleted.signal();
            lock.unlock();
        }

    }
    
    /**
     * Unlock the 2 clamps 
     * @param actuatorXminus
     * @param actuatorXplus
     * @throws BadCommandException
     * @throws ErrorInCommandExecutionException 
     */
    public void unlockClamps() throws BadCommandException, ErrorInCommandExecutionException, HardwareErrorDetectedException {

        
        lock.lock();
        this.unlockingClamps = true;

        try {
            clampXminus.unlock();
            clampXminus.unlock();
            
            if (clampXminus.getClampState().equals(FilterClampState.UNCLAMPEDONFILTER) & clampXplus.getClampState().equals(FilterClampState.UNCLAMPEDONFILTER))  {
                this.clampsState = FilterClampState.UNCLAMPEDONFILTER;
                System.out.println("XXXXXXXXSocket at standby : clamps are unlocked");


            } else if (!clampXminus.getClampState().equals(clampXplus.getClampState())) {
                throw new HardwareErrorDetectedException("Error in filter presence detection at standby position : "
                    + "the clamps don't agree.");
            } else if (!clampXminus.getClampState().equals(FilterClampState.UNCLAMPEDONFILTER)) {
                throw new ErrorInCommandExecutionException("Could not unlock clampX-");
            } else if (!clampXplus.getClampState().equals(FilterClampState.UNCLAMPEDONFILTER)) {
                throw new ErrorInCommandExecutionException("Could not unlock clampX+");
            }
            
        } finally {
            this.unlockingClamps = false;
            this.unlockCompleted.signal();
            lock.unlock();
        }

    }
    
    
    public void removeFilter() throws BadCommandException {
        if (this.filter == null) {
                throw new BadCommandException("Carousel socket : there is no filter to remove at standby position");
        }
        this.filter = null;
    }

    public synchronized void putFilterOnSocket(Filter filter) {
        setFilter(filter);
        filter.setFilterLocation(FilterLocation.ONCAROUSEL);
    }

    @Override
    public synchronized String toString() {
        StringBuilder sb = new StringBuilder("Filter in socket : ");
        if (getFilter() == null) {
            sb.append(" NO FILTER").append("\n");
        } else {
            sb.append(getFilter().getName()).append("\n");
        }
        return sb.toString();
    }


}
