
/*
 * To change this template, choose Tools | Templates
 * and open the template in the editor.
 */

package org.lsst.ccs.subsystems.fcs;

import java.util.Observable;
import java.util.concurrent.locks.Condition;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;
import org.lsst.ccs.bus.BadCommandException;
import org.lsst.ccs.bus.ErrorInCommandExecutionException;
import org.lsst.ccs.framework.Module;
import org.lsst.ccs.subsystems.fcs.FcsEnumerations.AutoChangerTrucksLocation;
import org.lsst.ccs.subsystems.fcs.FcsEnumerations.LockStatus;
import org.lsst.ccs.subsystems.fcs.common.AutoChanger;
import org.lsst.ccs.subsystems.fcs.common.FilterLocation;
import org.lsst.ccs.subsystems.fcs.common.ModuleState;
import org.lsst.ccs.subsystems.fcs.errors.CanOpenError;
import org.lsst.ccs.subsystems.fcs.errors.HardwareErrorDetectedException;
import org.lsst.ccs.subsystems.fcs.utils.FcsUtils;

/**
 * An abstract changer module without a fliprail, without online clamp, 
 * and without a motor.
 * This Module is in charge of the 2 latches which hold a filter in the autochanger.
 * This basic autochanger is used in single filter test and will be used with the final 
 * hardware too.
 * @author virieux
 */
public abstract class BasicAutoChangerModule extends Module implements AutoChanger {
    
    private boolean trucksEmpty;
    Filter filterOnTrucks;
    Filter filterToGrab;
    
    
    /**
    * Position of the trucks of the autochanger. It is monitored by the 4 rail sensors.
    * If the sensors detects a value different for the 2 sides, the trucksLocation is ERROR.
    */
    AutoChangerTrucksLocation trucksLocation;

    /**
     * A latch which hold a filter on the trucks at position X-
     */
    private FilterLatchModule latchXminus;

    /**
     * A latch which hold a filter on the trucks at position X+
     */
    private FilterLatchModule latchXplus;
    
    /**
     * The truck on side X-
     */
    protected TruckModule truckXminus;
    
    /**
     * The truck on side X+
     */
    protected TruckModule truckXplus;
    
    CompactIOModule railsSensorsDIO;
    CompactIOModule filterSensorsDIO;
     

    public static String publishedByAutoChangerOutputName = "publishedByAutoChanger";
    protected volatile ModuleState state = ModuleState.HALTED;
    private volatile LockStatus latchesState;
    
    private boolean initialized = false;
    
     /* FOR THE GUI 
     * 
     */
    volatile boolean movingToStandby;
    volatile boolean movingToStandback;
    
    /*******************************************************************************************/
    /********************** Tools for the synchronization **************************************/
    /*******************************************************************************************/
    protected volatile boolean updatingLatches = false;
    protected volatile boolean lockingLatches = false;
    protected volatile boolean unlockingLatches = false;
    
    private volatile boolean updatingTrucksLocation = false;
    //When we update the 2 latches, we want to take a lock on the autochanger until
    //the end of the update, so any other thread has to wait for this update.
    final Lock lock = new ReentrantLock();
    final Condition stateUpdated = lock.newCondition();
    final Condition lockCompleted = lock.newCondition();
    final Condition unlockCompleted = lock.newCondition();
    
    //When we update the 2 trucks position reading the sensors, 
    //we want to take a lock on the autochanger until
    //the end of the update, so any other thread has to wait for this update.
    final Condition trucksLocationUpdated = lock.newCondition();
    
    /*******************************************************************************************/
    /********************** End of Tools for the synchronization********************************/
    /*******************************************************************************************/

    
    
    public BasicAutoChangerModule() {
    }

    public CompactIOModule getRailsSensorsDIO() {
        return railsSensorsDIO;
    }

    public void setRailsSensorsDIO(CompactIOModule railsSensorsDIO) {
        this.railsSensorsDIO = railsSensorsDIO;
    }

    public CompactIOModule getFilterSensorsDIO() {
        return filterSensorsDIO;
    }

    public void setFilterSensorsDIO(CompactIOModule filterSensorsDIO) {
        this.filterSensorsDIO = filterSensorsDIO;
    }
    
    
    /**************************************************************************************************/
    /********************** SETTERS AND GETTERS  ******************************************************/
    /**************************************************************************************************/

    
    
    @Override
    public Filter getFilterOnTrucks() {
        return filterOnTrucks;
    }
    
        /**
     * @return the state
     */
    public ModuleState getState() {
        return state;
    }

    /**
     * @param state the state to set
     */
    public void setState(ModuleState state) {
        this.state = state;
    }

    public FilterLatchModule getLatchXminus() {
        return latchXminus;
    }

    public void setLatchXminus(FilterLatchModule latchXminus) {
        this.latchXminus = latchXminus;
    }

    public FilterLatchModule getLatchXplus() {
        return latchXplus;
    }

    public void setLatchXplus(FilterLatchModule latchXplus) {
        this.latchXplus = latchXplus;
    }


    public TruckModule getTruckXminus() {
        return truckXminus;
    }

    public void setTruckXminus(TruckModule truckXminus) {
        this.truckXminus = truckXminus;
    }

    public TruckModule getTruckXplus() {
        return truckXplus;
    }

    public void setTruckXplus(TruckModule truckXplus) {
        this.truckXplus = truckXplus;
    }
    
    public boolean isTrucksEmpty() {
        return trucksEmpty;
    }
    
    public void setTrucksEmpty(boolean isEmpty) {
        this.trucksEmpty = isEmpty;
    }

    public void setFilterOnTrucks(Filter filterOnTrucks) {
        this.filterOnTrucks = filterOnTrucks;
    }
    
    public String getFilterOnTrucksName() {
        if (this.filterOnTrucks == null) {
            return "none";
        } else {
            return this.filterOnTrucks.getName();
        }
    }

    public boolean isMovingToStandback() {
        return movingToStandback;
    }

    public void setMovingToStandback(boolean movingToStandback) {
        this.movingToStandback = movingToStandback;
    }

    public boolean isMovingToStandby() {
        return movingToStandby;
    }

    public void setMovingToStandby(boolean movingToStandby) {
        this.movingToStandby = movingToStandby;
    }

 
    
    
    
    /*****************************************************************************************************/
    /********************** END OF SETTERS AND GETTERS  **************************************************/
    /*****************************************************************************************************/
    
    /*****************************************************************************************************/
    /********************** Overriden methods from Modules or generic methodes ***************************/
    /*****************************************************************************************************/
    
    
    @Override
    public void initModule() {
        log.info("[AutoChangerModule] Initializing the Auto Changer module ");
        trucksLocation = AutoChangerTrucksLocation.UNKNOWN;
        latchesState = LockStatus.UNKNOWN;
    }

    /**
     * What has to be done for each tick of the timer.
     * We have to publish on the status bus. 
     */
//    @Override
//    public void tick() {      
//        sendToStatus(getStatusData());
//    }
    
//   @Override // this one doesn't work because tick doesn't throws any exception.
//    public void tick() {
//        if (initialized) {
//            try {
//                this.updateLatchesStateWithSensors();
//            } catch (HardwareErrorDetectedException ex) {
//                Logger.getLogger(BasicAutoChangerModule.class.getName()).log(Level.SEVERE, null, ex);
//            } catch (ErrorInCommandExecutionException ex) {
//                Logger.getLogger(BasicAutoChangerModule.class.getName()).log(Level.SEVERE, null, ex);
//            }
//            sendToStatus(getStatusData());
//        } else {
//            if (((BridgeToHardware) this.getModule("bridge")).isHardwareReady()) {
//                initialized = true;
//            }
//        }
//
//    }
    
        /*
     * what to do when the Modules we observe change their values.
     */
    @Override
    public void processUpdate(Observable source, ValueUpdate v) {
        this.publishDataAndNotifyObservers();
    }

    
    public StatusDataPublishedByBasicAutoChanger getStatusData() {
        return FcsUtils.createStatusDataPublishedByBasicAutoChanger(this);
    }

    public void publishDataAndNotifyObservers() {
        StatusDataPublishedByBasicAutoChanger status = getStatusData();
        sendToStatus(status);
        setChanged();
        notifyObservers(new ValueUpdate(publishedByAutoChangerOutputName, status));
    }

    @Override
    public String toString() {
        StringBuilder sb = new StringBuilder(getName());
        sb.append("\n");
        sb.append("Trucks position = ");
        sb.append("Filter on trucks:");
        if (this.filterOnTrucks == null) {
            sb.append(" NONE").append("\n");
        } else {
            sb.append(this.filterOnTrucks.getName()).append("\n");
        }
        return sb.toString();
    }
    
    /*****************************************************************************************************/
    /********************** END of Overriden methods from Modules  ***************************************/
    /*****************************************************************************************************/

    /*****************************************************************************************************/
    /********************** Methods for the motion of the trucks *****************************************/
    /*****************************************************************************************************/
    
    public abstract boolean isMoving();

    public abstract String goToStandby() throws BadCommandException, ErrorInCommandExecutionException, HardwareErrorDetectedException, CanOpenError;
    
    public abstract String goToStandback() throws BadCommandException, ErrorInCommandExecutionException, HardwareErrorDetectedException, CanOpenError ;

    //public abstract String moveFilterToOnline(Filter aFilter) throws BadCommandException, ErrorInCommandExecutionException;

    public abstract String moveFilterToStandby(Filter aFilter) throws BadCommandException, ErrorInCommandExecutionException, HardwareErrorDetectedException, CanOpenError;

    public abstract String moveFilterToStandback(Filter aFilter) throws BadCommandException, ErrorInCommandExecutionException, HardwareErrorDetectedException, CanOpenError;
    

    /**
     * This method checks if the preconditions are ok before a motion of the trucks can be started.
     * If the preconditions are not ok, it throws a BadCommandException.
     * A motion of the trucks can be started : 
     * if the latches are not in error and 
     * if the truks are empty or (the trucks are loaded and the latches are locked).
     * @throws BadCommandException 
     */
    public void checkPreConditionsForMotion() throws BadCommandException, HardwareErrorDetectedException, ErrorInCommandExecutionException {
        log.debug("Checking pre-conditions for motion");
        updateLatchesStateWithSensors();
        log.debug(getName() + " LATCHES STATUS= " + this.latchesState.toString());
        log.debug("LATCHX- is LOCKED=" + this.latchXminus.isLocked());
        log.debug("LATCHX+ is LOCKED=" + this.latchXplus.isLocked());
        log.debug("LATCHX- is UNLOCKED=" + this.latchXminus.isUnlocked());
        log.debug("LATCHX+ is UNLOCKED=" + this.latchXplus.isUnlocked());
        log.debug("LATCHX- status=" + latchXminus.getLockStatus());
        log.debug("LATCHX+ status=" + latchXminus.getLockStatus());
        if (latchesState == LockStatus.ERROR)  
            throw new BadCommandException(getName() + "Trucks motion forbidden when latches are in ERROR.");
        if (latchesState == LockStatus.UNKNOWN)
            throw new BadCommandException(getName() + "Trucks motion forbidden when latches status is in UNKNOWN.");
        if (!trucksEmpty && latchesState == LockStatus.UNLOCKED) {
            
            throw new BadCommandException(getName() + "Trucks motion forbidden when trucks are loaded and latches are UNLOCKED."); 
        }
        log.debug("===== Preconditions for motion are checked.");
    }
    

    @Override
    public String grabFilterAtStandby(Filter filter) throws BadCommandException, ErrorInCommandExecutionException, HardwareErrorDetectedException, CanOpenError {
        if (this.filterOnTrucks != null) {
            throw new BadCommandException(getName() + ": can't grabbe a filter " + "when a filter is already loaded in trucks ");
        }
        log.debug(getName() + ": grabbing " + filter.getName() + " at standby position.");
        goToStandby();
        lockLatchesAtStandby(filter);      
        String ack = getName() + ": " + filter.getName() + " is grabbed on autochanger";
        log.info(ack);
        return ack;
    }
        
    @Override
    public String ungrabFilterAtStandby() throws ErrorInCommandExecutionException, BadCommandException, HardwareErrorDetectedException, CanOpenError {
        //TODO to check : carousel clamp must be locked
        //
        log.info(getName() + ": ungrabbing filter at standby position.");
        unlockLatchesAtStandby();
        //eject filter from autochanger
        synchronized (this) {
            this.filterOnTrucks = null;
            setTrucksEmpty(true);
            this.publishDataAndNotifyObservers();
        }
        //when a filter is at standby position the trucks goes empty to SWAPOUT position
        log.info(getName() + " trucks going empty to SWAPOUT position");
        //goToPosition(this.getTrucksPositionSwapout());
        goToStandback();
        
        String ack = getName() + ": trucks are empty at SWAPOUT position ";
        return ack;
    }
    
    /*****************************************************************************************************/
    /********************** END of Methods for the motion of the trucks **********************************/
    /*****************************************************************************************************/

    /*****************************************************************************************************/
    /********************** Methods related to the latches ***********************************************/
    /*****************************************************************************************************/
    @Override
    public String unlockLatchesAtStandby() throws BadCommandException, ErrorInCommandExecutionException, HardwareErrorDetectedException {
        
        
        lock.lock();
        this.unlockingLatches = true;
        try {
            this.latchXminus.open();
            this.latchXplus.open();
            
            if ((latchXminus.getLockStatus().equals(LockStatus.UNLOCKED)) && 
                (latchXplus.getLockStatus().equals(LockStatus.UNLOCKED)))  {

                this.latchesState = latchXminus.getLockStatus();
                System.out.println("XXXXXXX AUTOCHANGER : latches are unlocked");

                
            } else if (!latchXminus.getLockStatus().equals(latchXplus.getLockStatus())) {
                    throw new HardwareErrorDetectedException(getName() + "Error in latches lock status : "
                        + "the latches don't agree.");
            } else if (!latchXminus.getLockStatus().equals(LockStatus.UNLOCKED)) {
                    throw new ErrorInCommandExecutionException("Could not unlock latchX-");
            } else if (!latchXplus.getLockStatus().equals(LockStatus.UNLOCKED)) {
                    throw new ErrorInCommandExecutionException("Could not unlock latchX+");
            }
            
            
        } finally {
            this.unlockingLatches = false;
            this.unlockCompleted.signal();
            lock.unlock();
        }
        publishDataAndNotifyObservers();
        return "Autochanger latches are unlocked";
    }

   /**
     * This methods locks the trucks latches at standby position and updates the 
     * field filterOnTrucks, the field filterLocation of the object Filter and updates
     * the data that we publish on the status bus.
     * It sends the command "lock" to the hardware of the latches and if this command is 
     * executed with success it goes on with the updates otherwise it returns an exception.
     * @return
     * @throws ErrorInCommandExecutionException 
     */
    @Override
    public String lockLatchesAtStandby(Filter aFilter) throws BadCommandException, 
                                    ErrorInCommandExecutionException, HardwareErrorDetectedException {
        //TODO: have to test if autochanger truck is HALTED, otherwise throw an exception
        //TODO: check if a filter is detected at standby 

        lock.lock();
        this.lockingLatches = true;
        this.filterToGrab = aFilter;
        try {

            this.latchXminus.close();
            this.latchXplus.close();
            
            if ((latchXminus.getLockStatus().equals(LockStatus.LOCKED)) && 
                (latchXplus.getLockStatus().equals(LockStatus.LOCKED)))  {

                this.latchesState = latchXminus.getLockStatus();
                System.out.println("XXXXXXX AUTOCHANGER : latches are locked");
                this.filterOnTrucks = this.filterToGrab;
                setTrucksEmpty(false);
                this.filterOnTrucks.setFilterLocation(FilterLocation.ONAUTOCHANGER);


            } else if (!latchXminus.getLockStatus().equals(latchXplus.getLockStatus())) {
                throw new HardwareErrorDetectedException(getName() + "Error in latches lock status : "
                 + "the latches don't agree.");
            } else if (!latchXminus.getLockStatus().equals(LockStatus.LOCKED)) {
                    throw new ErrorInCommandExecutionException("Could not lock latchX-");
            } else if (!latchXplus.getLockStatus().equals(LockStatus.LOCKED)) {
                    throw new ErrorInCommandExecutionException("Could not lock latchX+");
            }

            
        } finally {
            this.lockingLatches = false;
            this.lockCompleted.signal();
            lock.unlock();
        }
        
        publishDataAndNotifyObservers();
        return "Autochanger latches are locked";
    }
    
    /**
     * This methods updates the latches status in reading the sensors.
     */
    public void updateLatchesStateWithSensors() throws HardwareErrorDetectedException, ErrorInCommandExecutionException {
        lock.lock();
        this.updatingLatches = true;
        
        try {
            
            this.filterSensorsDIO.updateValue();
            
            String filterSensorsHexaValue = this.filterSensorsDIO.getHexaValue();
            this.latchXminus.updateState(filterSensorsHexaValue);
            this.latchXplus.updateState(filterSensorsHexaValue);
            
            if (latchXminus.getLockStatus().equals(latchXplus.getLockStatus()))  {
                    this.latchesState = latchXminus.getLockStatus();
                    System.out.println("XXXXXXX AUTOCHANGER : latches are updated");

            } else {
            throw new HardwareErrorDetectedException(getName() + "Error in latches lock status : "
             + "the latches don't agree.");
            }

        } finally {
            this.updatingLatches = false;
            this.stateUpdated.signal();
            lock.unlock();
        }
        publishDataAndNotifyObservers();
    }

    
    /*****************************************************************************************************/
    /********************** END of Methods related to the latches ****************************************/
    /*****************************************************************************************************/
    
    /*****************************************************************************************************/
    /********************** Methods to retrieve the trucks location **************************************/
    /*****************************************************************************************************/
    
     /**
     * This method returns the truckLocation if it is not been updated otherwise it waits until the end of 
     * the update.
     * @return the trucksLocation
     */
    public AutoChangerTrucksLocation getTrucksLocation() {
        lock.lock();
        try {
            while (updatingTrucksLocation) {
                try {
                    trucksLocationUpdated.await();
                } catch (InterruptedException ex) {
                    log.error(ex);
                }         
            }
            return trucksLocation;
        } finally {
            lock.unlock();
        }
    }
    
    
        /**
     * 
     * @return true if the trucks are at standby position, otherwise false
     */
    @Override
    public boolean isAtStandby() {
        return getTrucksLocation() == AutoChangerTrucksLocation.STANDBY;
    }
    
     /**
     * 
     * @return true if the trucks are at standback position, otherwise false
     */

    public boolean isAtStandback() {
        return getTrucksLocation() == AutoChangerTrucksLocation.STANDBACK;
    }
        
     /**
     * This methods updates the trucks location in reading the sensors.
     */
    public void updateTrucksLocationWithSensors() throws ErrorInCommandExecutionException {
        lock.lock();
        updatingTrucksLocation = false;
        try {
            this.railsSensorsDIO.updateValue();

            String dioHexaValue = this.railsSensorsDIO.getHexaValue();
            log.debug(this.railsSensorsDIO.getName() + " HEXA VALUE=" + dioHexaValue);
            this.truckXminus.updateLocation(dioHexaValue);
            this.truckXplus.updateLocation(dioHexaValue);

            if (truckXminus.getTruckLocation() == truckXplus.getTruckLocation()) {
                this.trucksLocation = truckXminus.getTruckLocation();

            } else {
                String msg = getName() + "Trucks are not at the same location";
                log.error(msg);
                throw new ErrorInCommandExecutionException(msg);
            }
            
       } finally {
           this.updatingTrucksLocation = false;
           this.trucksLocationUpdated.signal();
                           
           lock.unlock();
       } 
       publishDataAndNotifyObservers();

    }
    
    /*****************************************************************************************************/
    /********************** END of Methods to retrieve the trucks location *******************************/
    /*****************************************************************************************************/



    
  


    
}
