/*
 * To change this template, choose Tools | Templates
 * and open the template in the editor.
 */
package org.lsst.ccs.subsystems.fcs.testbench;

import java.util.concurrent.TimeoutException;
import org.lsst.ccs.framework.Module;
import org.lsst.ccs.state.State;
import org.lsst.ccs.subsystem.proxies.HardwareControlNotReady;
import org.lsst.ccs.subsystems.fcs.common.PieceOfHardware;
import org.lsst.ccs.subsystems.fcs.drivers.CanOpenADC;
import org.lsst.ccs.subsystems.fcs.drivers.CanOpenDAC;
import org.lsst.ccs.subsystems.fcs.drivers.CanOpenProxy;
import org.lsst.ccs.subsystems.fcs.drivers.CanOpenErrorsTable;
import org.lsst.ccs.subsystems.fcs.errors.DeploymentError;
import org.lsst.ccs.subsystems.fcs.errors.HardwareNotDetectedException;
import org.lsst.ccs.subsystems.fcs.errors.SDORequestError;
import org.lsst.ccs.subsystems.fcs.errors.ReadingADCError;

/**
 *This is the main module for the control software of the APC test bench.
 * 
 * When the TestBenchMainModule starts, it starts a a TCP-IP server.
 * the can open wrapper software will connect. 
 * Then the TestBenchProxy and the Wrapper will communicate over this TCP connection 
 * with a protocol to be define.
 * In a first version :
 * 1- The TCP server is started and waits for a client connection,
 * 2- When the client connects, it sends first a key word "TestBench".
 * 3- When the server receives the key word "TestBench", it's ready to boot the hardware 
 * and receive commands.
 * @author virieux
 */
public class APCTestBenchMainModule extends Module {
    
    private CanOpenProxy tcpProxy;
        
    /*The ADC that we have to control on the APC test bench.*/
    private CanOpenADC adc;
    
    /*The DAC that we have to control on the APC test bench.*/
    private CanOpenDAC dac;
    

    
    /**
    * One bit read on the input of the ADC represents 0.3125 mV
    */
    private static double adcVoltsPerBit = 0.3125;
    private static double waterFreezeTemp = 273.15;
    
    /**
     * @return the myTcpProxy
     */
    public CanOpenProxy getTcpProxy() {
        return tcpProxy;
    }

    /**
     * @param myTcpProxy the myTcpProxy to set
     */
    public void setTcpProxy(CanOpenProxy myTcpProxy) {
        this.tcpProxy = myTcpProxy;
    }

    /**
     * @return the nodeNB
     */
    public int getNodeNB() {
        return tcpProxy.getExpectedNodesNB();
    }

    /**
     * @return the myClientName
     */
    public String getMyClientName() {
        return tcpProxy.getMyClientName();
    } 
    
     /**
     * This method returns true if the C-Wrapper is connected to the tcp proxy.
     * The C-Wrapper has to send the tcpProxy clientName as a keyword before
     * any communication can begin. So if the C-Wrapper doesn't send the good
     * keyword or if it doesn't connect to the tcp proxy, this method returns 
     * false.
     */
    public boolean isHardwareConnected() {
        return tcpProxy.isReady(tcpProxy.getMyClientName());
    }

    /**
     * @return the adc
     */
    public CanOpenADC getAdc() {
        return adc;
    }

    /**
     * @param adc the adc to set
     */
    public void setAdc(CanOpenADC adc) {
        this.adc = adc;
    }

    /**
     * @return the dac
     */
    public CanOpenDAC getDac() {
        return dac;
    }

    /**
     * @param dac the dac to set
     */
    public void setDac(CanOpenDAC dac) {
        this.dac = dac;
    }
    
    
    /**
     * returns true if the hardware is booted, identified and initialized.
     */
    public boolean isHardwareReady() {
        return tcpProxy.isHardwareReady();
    }
    
    
    @Override
    public void initModule() {
        PieceOfHardware[] hardwareList1 = {adc, dac};
        tcpProxy.hardwareList = hardwareList1;
        log.debug("Number of pieces of hardware=" + hardwareList1.length);
        try {
            CanOpenErrorsTable.loadDeviceErrorTable();
            CanOpenErrorsTable.loadErrorRegisterTable();
            CanOpenErrorsTable.loadCommunicationErrorTable();
        } catch (DeploymentError ex) {
            log.error(ex.getMessage());
            this.getSubsystem().updateState(State.InError, ex.getMessage());
        }
    }
    

    
    public void shutdown() {
        tcpProxy.stop();
        this.getSubsystem().shutdown();
    }
    
    
    /**
     * For engineering mode, this method can be used to send Can Open commands
     * to the Wrapper. 
     * @param command A Can Open command that the Wrapper should understand.
     * @return the response from the Wrapper
     */
    public Object sendCanOpen(String command) throws InterruptedException, TimeoutException {
        return tcpProxy.sendCanOpen(command);
    }
    
        /**
     * Returns the value read on the socket for the input of the ADC given as an argument.
     * @param responseLine A string read on the socket.
     * @param inputNB  The numero if the input on the ADC. (1<=inputNB<=8)
     * @return the value (int) given 
     * @throws ReadingADCError 
     */
    public int getReturnValue(String responseLine, int inputNB) throws ReadingADCError {
            if (inputNB <1 || inputNB >8) throw new IllegalArgumentException();
            String[] words = ((String) responseLine).split(",");
            String command = words[0];
            log.debug("Value in HEXA:" + words[inputNB]);
            int value = Integer.parseInt(words[inputNB],16);
            log.debug("Value in decimal:" + words[inputNB]);
            return value;
    }
    
    /**
     * Read the temperature : in the test bench, a thermometer is plugged on the input number 1
     * of the ADC.
     * @return 
     * @throws ReadingADCError 
     */
    public double readTemperature() throws SDORequestError, HardwareNotDetectedException {
        String valueInHexa = adc.readAnalogInput(1);
        int value = Integer.parseInt(valueInHexa, 16);
        log.debug("Input Value read on the ADC:");
        double voltage = value * APCTestBenchMainModule.adcVoltsPerBit;
        log.debug("Voltage = " + voltage + " volts");
        double temperature = (voltage / 10) - APCTestBenchMainModule.waterFreezeTemp;
        return temperature;
    }
    
    /**
     * Read the value of the first channel of the ADC.
     * It just send the command rsdo,NODEid,6401,01 to the Can Open Wrapper 
     * @return
     * @throws ReadingADCError 
     */
    public String readADC() throws SDORequestError, TimeoutException {
            log.debug("Client name = " + getMyClientName());
            if (!tcpProxy.isHardwareReady()) throw new HardwareControlNotReady(getMyClientName()); 
            //response = getTcpProxy().call(getMyClientName(), "adc,");
            //return tcpProxy.readSDO(adc.getNodeID(), "6401", "01");
            return adc.readInputs();
        
    }
    
    
    public void turnOnLED() throws InterruptedException, SDORequestError, HardwareNotDetectedException {
        if (!tcpProxy.isHardwareIdentified()) throw new HardwareControlNotReady(getMyClientName());
        //getTcpProxy().call(getMyClientName(), "dac,3000,0000,0000,0000");
        //tcpProxy.writeSDO(dac.getNodeID(), "6411", "01", "2", "3000");
        dac.writeAnalogOutput(1, 14000);
    }
    
    public void turnOffLED() throws InterruptedException, SDORequestError {
        if (!tcpProxy.isHardwareIdentified()) throw new HardwareControlNotReady(getMyClientName());
        tcpProxy.writeSDO(dac.getNodeID(), "6411", "01", "2", "0");
    }
    
        /**
     * Initialize the ADC and configure the way it will send its data : 
     * we want it to send data each time it receives syncNB SYNC.
     * if syncNB = 1, it will send data every SYNC,
     * if syncNB = 2, it will send data every 2 SYNC, etc...
     * @param syncNB 
     */
    private void initADC(int syncNB) throws ReadingADCError {
        try {
            //if (!hardwareBooted) throw new HardwareNotBootedError();
            //for the first 4 INPUT channels from 1 to 4
            Object ack = getTcpProxy().call(getMyClientName(), "wsdo,1,1801,2,1,1");
            if (ack instanceof String) {           
                    log.debug("initADC-step1 = " + ack); 
                    Object ack2 = getTcpProxy().call(getMyClientName(), "wsdo,1,1802,2,1,1");
                    log.debug("initADC-step2 = " + ack2); 
            } else throw new ReadingADCError(ack);
        } catch (TimeoutException ex) {
            log.error(ex.getMessage());
            throw new ReadingADCError(ex);
        }
        
    }

    
}
