/*
 * To change this template, choose Tools | Templates
 * and open the template in the editor.
 */
package org.lsst.ccs.subsystems.fcs.drivers;

import java.util.concurrent.TimeoutException;
import org.lsst.ccs.bus.Alarm;
import org.lsst.ccs.bus.BadCommandException;
import org.lsst.ccs.state.State;
import org.lsst.ccs.subsystems.fcs.common.PieceOfHardware;
import org.lsst.ccs.subsystems.fcs.errors.CanOpenCommunicationError;
import org.lsst.ccs.subsystems.fcs.errors.CanOpenDeviceError;
import org.lsst.ccs.subsystems.fcs.errors.CanOpenError;
import org.lsst.ccs.subsystems.fcs.errors.CanOpenGenericError;
import org.lsst.ccs.subsystems.fcs.errors.CanOpenMotionError;
import org.lsst.ccs.subsystems.fcs.errors.CanOpenTemperatureError;
import org.lsst.ccs.subsystems.fcs.errors.CanOpenVoltageError;
import org.lsst.ccs.subsystems.fcs.errors.NodeIDMismatchException;
import org.lsst.ccs.subsystems.fcs.errors.ErrorInBootingHardwareProcess;
import org.lsst.ccs.subsystems.fcs.errors.HardwareNotDetectedException;
import org.lsst.ccs.subsystems.fcs.errors.SDORequestError;
import org.lsst.ccs.subsystems.fcs.errors.UnexpectedBootMessageReceived;
import org.lsst.ccs.subsystems.fcs.FcsAlarm;

/**
 * This Module starts a tcpip server, waits for the connection of a client whose 
 * name is the value of the field myClientName.
 * This client is supposed to be a CanOpen C-wrapper.
 * When the client starts it scans the can open network and send to the server
 * some information on the can open nodes living on the network.
 * 
 * @author virieux
 */
public class CanOpenProxy extends FcsTcpProxy {
    
    
    
    //private ApcTcpProxy tcpProxy;
    /*This is the key word that has to be sent by the client when it connects to the tcp proxy. */
    private String myClientName = "CanOpenProxy";
    
    /*number of can open nodes expected on the network */
    private int expectedNodesNB;
    
    /*number of booted nodes*/
    private int bootedNodesNB = 0;
    
   
    
    /*An array of booted hardware*/
    private CanOpenNode[] nodes = new CanOpenNode[getExpectedNodesNB()];
    
    private boolean hardwareBootProcessEnded = false;
    private boolean hardwareIdentified = false;
    
    /**
     * A timeout for the booting process : 
     * we don't want to wait longer that this elapse of time.
     * This has to be tuned during the test bench.
     * UNIT = milliseconds
     */
    private long hardwareBootTimeout = 2000;
    
    private long hardwareBootProcessBeginTime;
    
    /*An array of hardware controled by this main Module*/
    //PieceOfHardware[] hardwareList = {actuatorA, actuatorB, actuatorC};
    public PieceOfHardware[] hardwareList;
    
    private boolean canOpenNodeNumbersOK = false;
    private boolean canOpenNodesChecked = false;

        
    /**************************************************************************************************/
    /********************** SETTERS AND GETTERS  ******************************************************/
    /**************************************************************************************************/
    
    
    /**************************************************************************************************/
    /********************** END OF SETTERS AND GETTERS  ***********************************************/
    /**************************************************************************************************/
    /**
     * @return the expectedNodesNB
     */
    public int getExpectedNodesNB() {
        return expectedNodesNB;
    }

    /**
     * @param expectedNodesNB the expectedNodesNB to set
     */
    public void setExpectedNodesNB(int nodeNB) {
        this.expectedNodesNB = nodeNB;
    }
    
     /**
     * @return the bootedNodesNB
     */
    public int getBootedNodesNB() {
        return bootedNodesNB;
    }


   

    /**
     * @return the myClientName
     */
    public String getMyClientName() {
        return myClientName;
    }

    /**
     * @param myClientName the myClientName to set
     */
    public void setMyClientName(String myClientName) {
        this.myClientName = myClientName;
    }
    
    @Override
    public void initModule() {
       super.initModule();
       nodes = new CanOpenNode[getExpectedNodesNB()];
       hardwareBootProcessBeginTime = 0;
    }
    
    public boolean isHardwareIdentified() {
        return hardwareIdentified;
    }
    
    
    
    
    @Override
    public void tick() {
        //INITIALISATION  
        //TODO : All the initialisations have to go in the startTicking method
        
        // we can start the hardware booting process when the tcp client is ready. 
        if ((hardwareBootProcessBeginTime == 0) && isReady(myClientName)) {
            hardwareBootProcessBeginTime = System.currentTimeMillis();
        }
        
        //we want to check if the hardware is booted.
        if ((hardwareBootProcessBeginTime > 0) && !hardwareBootProcessEnded) { 
            long hardwareBootDuration = System.currentTimeMillis() - hardwareBootProcessBeginTime;
            if (this.bootedNodesNB == this.getExpectedNodesNB()) {
                hardwareBootProcessEnded = true;
            } else if (hardwareBootDuration > hardwareBootTimeout) {
                hardwareBootProcessEnded = true;
                log.error("Timeout expired during hardware booting process ");
                this.getSubsystem().updateState(State.InError, 
                        "Timeout expired during hardware booting process : number of booted is not was expected"); 
            }
                
        }
        
        // and retrieve some information in the hardware
        if ((hardwareBootProcessEnded) && !hardwareIdentified) {          
            try {
                identifieHardware();
                
            } catch (TimeoutException ex) {
                log.error("Timeout expired during hardware identifie process ");
                this.getSubsystem().updateState(State.InError,"Timeout expired during hardware identifie process." + ex.getMessage());
            }
            hardwareIdentified = true;
                       
        }
        
        if (this.hardwareIdentified && !canOpenNodesChecked) {
            try {
                canOpenNodesChecked = checkCanOpenNodes();               
                log.debug("Hardware configuration is checked");
            } catch (NodeIDMismatchException ex) {
                canOpenNodesChecked = false;
                log.error("ERROR in configuration for a nodeID" + ex.getMessage());
                Alarm alarm = new FcsAlarm(ex.toString());
                this.sendToStatus(alarm);
                //this.getSubsystem().updateState(State.InError, "ERROR in configuration for a nodeID" + ex.toString());
            } catch (HardwareNotDetectedException ex) {
                canOpenNodesChecked = false;
                log.error("HARDWARE NOT DETECTED : POWER FAILURE ?" + ex.getMessage());
                Alarm alarm = new FcsAlarm(ex.toString());
                this.sendToStatus(alarm);
                //this.getSubsystem().updateState(State.InError, "HARDWARE NOT DETECTED : POWER FAILURE ?" + ex.getMessage());
            } catch (ErrorInBootingHardwareProcess ex) {
                canOpenNodesChecked = false;
                log.error("HARDWARE NOT DETECTED : POWER FAILURE ?" + ex.getMessage());
                Alarm alarm = new FcsAlarm(ex.toString());
                this.sendToStatus(alarm);
                //this.getSubsystem().updateState(State.InError, "HARDWARE NOT DETECTED : POWER FAILURE ?" + ex.getMessage());
            } catch (BadCommandException ex) {
                log.error(ex.getMessage());
                Alarm alarm = new FcsAlarm(ex.toString());
                this.sendToStatus(alarm);
            }
        }
        
//        //send a sync command so all the devices will update and send their new values
//        if (hardwareBootProcessEnded && hardwareIdentified && canOpenNodesChecked) {
//            try {
//                sendCanOpen("sync,");
//            } catch (TimeoutException ex) {
//                Logger.getLogger(CanOpenProxy.class.getName()).log(Level.SEVERE, null, ex);
//            }
//            
//        }

    }
    
    
    
    /*Check if a piece of hardware with nodeID aNodeID is already booted or not.
     * If it is booted it is on the nodes table.    
     * It means that this node ID is on the can open bus.
     */
    public boolean isBooted(String aNodeID) {
        
        if (bootedNodesNB == 0) return false;
        boolean nodeFound = false;
        for (int i=0; i < this.bootedNodesNB; i++ ) {
            if (!(nodes[i] == null) &&  (nodes[i].getNodeID().equals(aNodeID))) {
                nodeFound = true;
            }
        }
        return nodeFound;
    }
    
    /* Check if a piece of hardware is already booted or not.
     * If it is booted its node is on the nodes table. 
     * This means that this node ID is on the can bus.
     */
    public boolean isBooted(PieceOfHardware piece) {
        
        if (bootedNodesNB == 0) return false;
        boolean pieceFound = false;
        for (int i=0; i < this.bootedNodesNB; i++ ) {
            
            if (!(nodes[i] == null) &&  (nodes[i].getNodeID().equals(piece.getNodeID()))) {
                pieceFound = true;
            }
        }
        return pieceFound;
    }
    
     /**
     * Check if a piece of hardware is booted 
     * and if its can open node ID is correct.
     */
    public boolean isConfigOK(PieceOfHardware piece) {
        
        if (bootedNodesNB == 0) return false;
        boolean pieceFound = false;
        for (int i=0; i < this.bootedNodesNB; i++ ) {
            
            if (!(nodes[i] == null) &&  (nodes[i].getNodeID().equals(piece.getNodeID()))
                    && (nodes[i].getSerialNB().equals(piece.getSerialNB()))) {
                pieceFound = true;
            }
        }
        return pieceFound;
    }

    /**
     * List the can open nodes which are in the nodes table.
     * @return the list of can open nodes and the information stored in this.nodes.
     */
    public String listNodes() {
        
        StringBuilder sb = new StringBuilder("Nodes LIST = (values are in HEXA)");
        for (int i=0; i < this.bootedNodesNB; i++) {
            sb.append("\n");
            if (nodes[i] == null) {
                sb.append("null node");
            } else sb.append(nodes[i].toString()) ;
        }
        return sb.toString();

    }
    
    
    /**
     * For engineering mode, this method can be used to send Can Open commands
     * to the Wrapper. 
     * Test if the command is valid before sending it to the can open stack.
     * @param command A Can Open command that the Wrapper should understand.
     * @return the response from the Wrapper
     * 
     */
    public String sendCanOpen(String command) throws TimeoutException  {
//        try {
          String[] words = command.split(",");
          String keyWord = words[0]; 
          if (words.length > 1) {
              if (!(keyWord.equals("rsdo") 
                      || keyWord.equals("wsdo") 
                      || keyWord.equals("info")
                      || keyWord.equals("rsync")
                      || keyWord.equals("srtr"))) 
                throw new IllegalArgumentException(command);
          } else {
              if (!(keyWord.equals("sync") 
                      || keyWord.equals("rpdo") 
                      || keyWord.equals("quit"))) 
                throw new IllegalArgumentException(command);
          }
          return (String) call(getMyClientName(), command);
//        } catch (TimeoutException ex) {
//            log.error("no response from can open stack for command: " + command);
//        }
//        return null;  
    }
      
    
    /**
     * build a Can Open wsdo Command that can understand the C wrapper.
     * exemple : wsdo,2,6411,01,2,3000
     */
    private static String buildWsdoCommand(String nodeID, String index, String subindex, String size, String data) {
        char sep = ',';
        String command = String.format("wsdo%1$s%2$s%1$s%3$s%1$s%4$s%1$s%5$s%1$s%6$s",sep,nodeID,index,subindex,size,data);
        System.out.println("Command built = " + command);
        return command;
    }
    
    /**
     * build a Can Open rsdo Command that can understand the C wrapper.
     * exemple : rsdo,1,1018,0
     */
    private static String buildRsdoCommand(String nodeID, String index, String subindex) {
        char sep = ',';
        String command = String.format("rsdo%1$s%2$s%1$s%3$s%1$s%4$s",sep,nodeID,index,subindex);
        return command;
    }

    /**
     * Write a SDO request and send it to the can open stack, then analyses the response
     * or throws an exception if the request failed.
     * The parameters are given in HEXA.
     * @param nodeID FORMAT=HEXA
     * @param index FORMAT=HEXA
     * @param subindex FORMAT=HEXA
     * @param size FORMAT=HEXA
     * @param value FORMAT=HEXA
     * @return "writeSDO request OK" if OK
     */
    public String writeSDO(String nodeID, String index, String subindex, String size, String value) throws SDORequestError  {
        try {
            String sdoResponseLine = sendCanOpen(buildWsdoCommand(nodeID,index,subindex,size,value));
            String[] words = sdoResponseLine.split(",");
            String command = words[0];
            String receivedNodeID = words[1];
            String errorCode = words[2];

            if (errorCode.equals("0")) {
                return "writeSDO request OK";
            } else if  (errorCode.equals("-1")) {
                log.error("Wrong SDO command : " + sdoResponseLine);
                throw new SDORequestError("Wrong SDO command : " + sdoResponseLine);
            } else {
                log.error("SDO request was going wrong");
                String error = String.format("%08d", Integer.parseInt(errorCode));
                log.debug("ERROR=" + error);
                String errorName = CanOpenErrorsTable.commErrorCodes.getProperty(error);
                throw new SDORequestError("SDO request was going wrong", errorCode, nodeID, index, subindex, errorName);
            }
        } catch (TimeoutException ex) {
            SDORequestError err = new SDORequestError("write SDO request timeout expired : ", nodeID,"timeout expired", index, subindex, ex.getMessage());
            log.error(String.format("%s %s is %s stopped ?", "SDO request timeout expired : ",ex.getMessage(),nodeID));
            throw err;
        }
       
    }
    
     /**
     * Command to be used by the end user at the console.
     * Write a SDO message and send it to the can open stack.
     * The parameters are given in the following format:
     * @param nodeID FORMAT=decimal
     * @param index FORMAT=HEXA
     * @param subindex FORMAT=HEXA
     * @param size FORMAT=decimal
     * @param value FORMAT=decimal
     * @return 
     */
    public String writeSDO(int nodeID, String index, String subindex, int size, int value) throws SDORequestError {
        if (nodeID <0 || nodeID > 127) 
            throw new IllegalArgumentException("nodeID must be > 0 and < 128");
        if (size <0 || size > 4) 
            throw new IllegalArgumentException("size must be > 0 and < 4");
        return writeSDO(Integer.toHexString(nodeID),index,subindex,Integer.toHexString(size),Integer.toHexString(value));
    }
    
    
     /**
     * Read a SDO with the given index and subindex
     * RETURNS the value read in hexa (String) if no error occured
     * or returns the error code if a error was detected.
     *
     */
    public String readSDO(String nodeID, String index, String subindex) throws SDORequestError  {
        try {
            String sdoLine = sendCanOpen(buildRsdoCommand(nodeID,index,subindex));
            String[] words = sdoLine.split(",");
            int responseLength = words.length;
            
            String command = words[0];
            String receivedNodeID = words[1];
            String errorCode = words[2];
           
            if (errorCode.equals("0") && (words.length > 3)) {
                return words[3];
            } else if  (errorCode.equals("-1")) {
                log.error("Wrong SDO command : " + sdoLine);
                throw new SDORequestError("Wrong SDO command : " + sdoLine);
            } else {
                log.error("SDO request was going wrong, received response=" + sdoLine);
                String error = String.format("%08d", Integer.parseInt(errorCode));
                log.debug("ERROR=" + error);
                String errorName = CanOpenErrorsTable.commErrorCodes.getProperty(error);
                throw new SDORequestError("SDO request was going wrong", nodeID, errorCode, index, subindex, errorName);
            }
        } catch (TimeoutException ex) {
            SDORequestError err = new SDORequestError("read SDO request timeout expired : ", nodeID, "timeout expired",index, subindex, ex.getMessage());
            log.error(String.format("%s %s is %s stopped ?", "SDO request timeout expired : ",ex.getMessage(),nodeID));
            throw err;
        }
       
    }
    
    /**
     * Command to be used by the end user at the console in engineering mode.
     * @param nodeID FORMAT=decimal
     * @param index FORMAT=hexa
     * @param subindex FORMAT=hexa
     * @return the value sent by the can open stack if the read SDO request succeded.
     * @throws SDORequestError 
     */
    public String readSDO(int nodeID, String index, String subindex) throws SDORequestError  {
        if (nodeID <0 || nodeID > 127) 
            throw new IllegalArgumentException("nodeID must be > 0 and < 128");
        return readSDO(Integer.toHexString(nodeID),index,subindex);
    }

    /**
     * Identification of the hardware : we want to retrieve the information 
     * stored in the hardware of the booted nodes and update the array of nodes
     * with this information.
     * The Can Open node ID is a main information to retrieve.
     */
    public void identifieHardware() throws TimeoutException {
        log.debug("Identification of the hardware");
        for (int i=0; i < this.bootedNodesNB; i++ ) {
            if (!(nodes[i] == null)) {
                log.debug("Sending to can open command : info," + nodes[i].getNodeID());
                Object result = sendCanOpen("info," + nodes[i].getNodeID());
                log.debug("Received on socket command = " + result);
                String infoLine = (String) result;
                String[] words = infoLine.split(",");
                String command = words[0];
                String nodeID = words[1];            
                if (command.equals("info") && nodeID.equals(nodes[i].getNodeID())) { 
                    log.debug("updating Node Info for node " + nodeID);
                    String type = words[2];
                    String vendor = words[3];
                    String productCode = words[4];
                    String revision = words[5];
                    String serialNB = words[6];
                    nodes[i].setNodeInfo(type, vendor, productCode, revision, serialNB);
                } else {
                    log.debug("ERROR for command = " + command + "NodeID = " + nodeID);
                    //TODO throw an exception
                }
            }
        }
        log.debug("Hardware is identified");
        log.debug(listNodes());
    }

    /**
     * This method returns true if : all the hardware items are booted and identified 
     * and the hardware have the node ID expected within the configuration.
     */
    public boolean isHardwareReady() {
        log.debug("hardwareBootProcessEnded=" + hardwareBootProcessEnded);
        log.debug("hardwareIdentified=" + hardwareIdentified);
        log.debug("canOpenNodeNumbersOK=" + this.canOpenNodeNumbersOK);
        log.debug("canOpenNodesChecked=" + this.canOpenNodesChecked);
        return this.hardwareBootProcessEnded && this.hardwareIdentified 
                && this.canOpenNodeNumbersOK && this.canOpenNodesChecked;
    }

    /**
     * This method returns the can open detected on the can open network 
     * for a piece of hardware identified by its serial number.
     * It can return NULL if the Serial Number is not on the can open network.
     * @param serialNB
     * @return 
     */    
    public CanOpenNode getDetectedNodeForSerialNumber(String aSerialNB) {
        CanOpenNode detectedNode = null;
        for (int i=0; i < this.bootedNodesNB; i++ ) {
            if (nodes[i].getSerialNB().equals(aSerialNB)) {
                detectedNode = nodes[i];
            }
        }
        return detectedNode;
    }

    @Override
    void processBootMessage(String nodeID)  {
        if (isBooted(nodeID)) {
            if (hardwareBootProcessEnded) {
                    /*The node is already in the table of nodes : something weird has just happened*/ 
                    String msg = "An asynchronus boot message is arrived for node ID = " + nodeID;
                    log.error(msg);
                    UnexpectedBootMessageReceived ex = new UnexpectedBootMessageReceived(msg,nodeID);
                    Alarm alarm = new FcsAlarm(ex.toString());
                    this.sendToStatus(alarm);
//                    this.getSubsystem().updateState(State.InError,
//                            " Possible power failure for node : " + nodeID);
              
            }
            
        } else {
            //TODO : if the number of nodes we found is greater than the expectedNodesNB, throw an exception.
            /*the node is not already in the table of nodes : we have to add it*/
            nodes[bootedNodesNB] = new CanOpenNode(nodeID);
            bootedNodesNB++;
            log.debug("Node " + nodeID + " added");
            log.debug("Hardware booted nb=" + bootedNodesNB);
        }
    }
    
    @Override
    void processEmcyMessage(String message) {
        //TODO put the corresponding node InERROR state ?
            String[] words = message.split(",");
            String nodeID = words[1];
            String errCode = words[2];
            String errReg = words[3];
            log.error("Received EMERGENCY message for nodeID: " + nodeID);
            log.error("message=" + message);
            String errorName = CanOpenErrorsTable.errorRegisterCodes.getProperty(errReg);
            log.debug("errorName=" + errorName);
            
        try {
                     
            if (errReg.equals("01")) throw new CanOpenGenericError(errorName,nodeID, errReg);
            else if (errReg.equals("04")) throw new CanOpenVoltageError(errorName,nodeID, errReg);
            else if (errReg.equals("08")) throw new CanOpenTemperatureError(errorName,nodeID, errReg);
            else if (errReg.equals("10")) throw new CanOpenCommunicationError(errorName,nodeID, errReg);
            else if (errReg.equals("20")) throw new CanOpenDeviceError(errorName,nodeID, errReg);
            else if (errReg.equals("80")) throw new CanOpenMotionError(errorName,nodeID, errReg);
            else throw new CanOpenError("Received EMERGENCY message (UNKNOWN Cnn Open Error) for nodeID: " + nodeID);
                
            

        } catch (CanOpenError ex) {
            
            log.error(ex.getMessage());
            Alarm alarm = new FcsAlarm(ex.toString());
            this.sendToStatus(alarm);
            //TODO in normal mode : put the subsystem in ERROR
            String err = String.format("NodeID %s (%02d in decimal) : %s ",nodeID,Integer.parseInt(nodeID, 16), ex.getMessage());
            //this.getSubsystem().updateState(State.InError, err);
        }
    }
    

    
    @Override
    void processUnknownCommand(String message) {
        String[] words = message.split(",");
        String command = words[1];
        String errorMessage = "Unknown command:" + command;
        log.error(errorMessage);
    }
    
    public boolean checkCanOpenNodes() throws BadCommandException, ErrorInBootingHardwareProcess, NodeIDMismatchException, HardwareNotDetectedException {
        if (!(isReady(this.myClientName) && isHardwareIdentified()))
                    throw new BadCommandException("Hardware not booted or not identified");
        
        log.debug("CHECKING HARDWARE CONFIGURATION ");
        if (this.bootedNodesNB == 0) {
            log.error("NO HARDWARE DETECTED : POWER FAILURE ?");
            
            throw new ErrorInBootingHardwareProcess("NO HARDWARE DETECTED : POWER FAILURE ?", this.hardwareBootTimeout, this.bootedNodesNB, this.expectedNodesNB);       
        }
        if (this.bootedNodesNB != this.expectedNodesNB) {
            log.error("SOME HARDWARE IS MISSING : POWER FAILURE ?");
            
            throw new ErrorInBootingHardwareProcess("SOME HARDWARE IS MISSING : POWER FAILURE ?", this.hardwareBootTimeout,this.bootedNodesNB, this.expectedNodesNB);       
        }
        this.canOpenNodeNumbersOK = true;
        boolean configNodesID = true;
        for (PieceOfHardware hardware : this.hardwareList)  {
            log.debug("ABOUT TO CHECK: " + hardware.getName() + " NODE_ID=" + hardware.getNodeID());
            boolean configOK = checkCanOpenNodeConfiguration(hardware);
            configNodesID = configNodesID && configOK;
            log.info(hardware.getName() + " CONFIG OK=" + configOK);
        }
       
        return configNodesID;
        
    }
    

    
        
     /**
     * When the hardware is booted and we have retrieve the information, 
     * we want to check for each piece of hardware that the node ID stored 
     * in the Configuration data base is the same than the node ID which is stored 
     * on the hardware with a switch.
     * @return 
     */
    public boolean checkCanOpenNodeConfiguration(PieceOfHardware pieceOfHardware) throws BadCommandException, NodeIDMismatchException, HardwareNotDetectedException  {
        
        if (!(this.hardwareBootProcessEnded && this.hardwareIdentified))
                    throw new BadCommandException("Hardware not booted or not identified");
        
        log.debug("Checking configuration for " + pieceOfHardware.getName());
        CanOpenNode canOpenNode = getDetectedNodeForSerialNumber(pieceOfHardware.getSerialNB());
        //if (canOpenNode == null) {
        if (this.isBooted(pieceOfHardware)) {

            String detectedNodeID = canOpenNode.getNodeID();
            if (detectedNodeID.equals(pieceOfHardware.getNodeID()))  {
                log.debug(pieceOfHardware.getName() + " Serial Number= " + pieceOfHardware.getSerialNB() +
                        " is node ID=" + detectedNodeID);
                
            } else {
                
//                this.getSubsystem().updateState(State.InError,
//                    " Possible power failure for node ID: " + pieceOfHardware.getNodeID());
                throw new NodeIDMismatchException("ERROR IN CAN OPEN NODE ID CONFIGURATION", canOpenNode,pieceOfHardware);
            }

        } else {

            throw new HardwareNotDetectedException("HARDWARE NOT DETECTED", pieceOfHardware.getName(),pieceOfHardware.getNodeID(),pieceOfHardware.getSerialNB());
        }
 
        return pieceOfHardware.isConfigOK();
    }
    
    
    public String listHardware() {
        StringBuilder sb = new StringBuilder();
        for (PieceOfHardware pieceOfHardware : this.hardwareList)  {
            sb.append(pieceOfHardware.toString());
            sb.append("**");
        }
        return sb.toString();      
    }
    
    public void initializeHardware() {
        for (PieceOfHardware pieceOfHardware : this.hardwareList)  {
            pieceOfHardware.initialize();
        }
    }
    
    /**
     * Configure a node as a hearbeat producer.
     * Every heartbeat_time milliseconds this node will produce a message like:
     * 0x700+nodeID: 05 - 1340268404.980000 
     * @param nodeID node ID in hexa
     * @param heartbeat_time FORMAT=hexa UNIT=milliseconds
     */
    public String configAsHeartbeatProducer(String nodeID, String heartbeat_time) throws SDORequestError, TimeoutException {
        return writeSDO(nodeID,"1017","0","2",heartbeat_time);
    }
    
    /**
     * Command to be used by the end users.
     * Configure a node as a heartbeat producer.
     * Parameters given in decimal format.
     * @param nodeID FORMAT=decimal
     * @param heartbeat_time FORMAT=decimal UNIT=milliseconds
     */
    public String configAsHeartbeatProducer(int nodeID, int heartbeat_time) throws SDORequestError, TimeoutException {
        
        if (heartbeat_time < 0 || heartbeat_time > 65025)
            throw new IllegalArgumentException("heartbeat_time is coded on 2 bytes"
                    + " can't be > 65535");
        return configAsHeartbeatProducer(Integer.toHexString(nodeID), Integer.toHexString(heartbeat_time));
    }
    
     /**
     * Configure a node as a hearbeat consumer.
     * Every heartbeat_time milliseconds this node will produce a message like:
     * 0x700+nodeID: 05 - 1340268404.980000 
     * @param nodeID node ID in decimal
     * @param heartbeat_time FORMAT=decimal UNIT=milliseconds
     */
    public String configAsHeartbeatConsumer(int nodeID, int producerNodeID, int heartbeat_time) throws SDORequestError, TimeoutException {
        if (nodeID <0 || nodeID > 127) 
            throw new IllegalArgumentException("nodeID must be > 0 and < 128");
        if (producerNodeID <0 || producerNodeID > 127) 
            throw new IllegalArgumentException("producerNodeID must be > 0 and < 128");
        if (heartbeat_time < 0 || heartbeat_time > 65025)
            throw new IllegalArgumentException("heartbeat_time is coded on 2 bytes"
                    + " can't be > 65535");
        String value = String.format("%04x",producerNodeID) + String.format("%04x",heartbeat_time);
        return writeSDO(Integer.toHexString(nodeID),"1016","1","4",value);
    }


    
    



    
   
    
}
