/*
 * To change this template, choose Tools | Templates
 * and open the template in the editor.
 */
package org.lsst.ccs.subsystems.fcs.drivers;

import org.lsst.ccs.state.State;
import org.lsst.ccs.subsystems.fcs.CompactIOModule;
import org.lsst.ccs.subsystems.fcs.common.PieceOfHardware;
import org.lsst.ccs.subsystems.fcs.errors.HardwareNotDetectedException;
import org.lsst.ccs.subsystems.fcs.errors.SDORequestError;
import org.lsst.ccs.subsystems.fcs.FcsAlarm;

/**
 * This is the driver class for the Digital-Input-Output we use in the FCS: CAN-CBX-DIO8.
 * A CAN-CBX-DIO8 has 8 inputs or outputs. At the initialization, it has to be configured to say which are 
 * the inputs and which are the ouputs.
 * The outputs for the standby or standback motors of the autochanger are on this device.
 * The sensors to know if a latch on the autochanger is closed or open are also connected to a CAN-CBX-DIO8.
 * @author virieux
 */
public class CanOpenDIO  extends CompactIOModule implements PieceOfHardware {

    CanOpenProxy tcpProxy;
    
    String serialNB;
    String nodeID;
    
    //FOR SPRING
    public CanOpenDIO() {       
    }

    public CanOpenDIO(String nodeID, String serial) {
        this.tcpProxy = (CanOpenProxy) this.getModule("tcpProxy");
        this.serialNB = serial;
        this.nodeID = nodeID;       
    }
    
        
    @Override
    public void initModule() {
        tcpProxy = (CanOpenProxy) this.getModule("tcpProxy");
    }
            
    /**************************************************************************************************/
    /********************** SETTERS AND GETTERS  ******************************************************/
    /**************************************************************************************************/
    
    @Override
    public String getSerialNB() {
        return this.serialNB;
    }

    @Override
    public String getNodeID() {
        return this.nodeID;
    }


    /**
     * @param serialNB the serialNB to set
     */
    public void setSerialNB(String serialNB) {
        this.serialNB = serialNB;
    }

    /**
     * @param nodeID the nodeID to set
     */
    public void setNodeID(String nodeID) {
        this.nodeID = nodeID;
    }

    @Override
    public boolean isConfigOK() {
        return tcpProxy.isConfigOK(this);
    }      
    /**************************************************************************************************/
    /********************** SETTERS AND GETTERS  ******************************************************/
    /**************************************************************************************************/
        
        
     /**
     * Write a value to the output of the DAC
     * @param outputNB in 1..8
     * @param value given in decimal format
     * TODO chnage this : it's not so simple !
     */
    public String writeDigitalOutput(int outputNB, int value) throws SDORequestError, HardwareNotDetectedException {
        
        if (!this.tcpProxy.isHardwareReady()) {
            String message = String.format("DAC %s / nodeID=%s is NOT BOOTED", getName(),this.nodeID);
            log.error(message);
            //this.getSubsystem().updateState(State.InError, message );
            throw new HardwareNotDetectedException("a CanOpenDIO is missing",getName(),this.nodeID, this.serialNB);
        }
        
        if ((outputNB < 1) || (outputNB > 8)) 
            throw new IllegalArgumentException("CAN-CBX-DIO8 : input has to be 1..8");
        String valueInHexa = Integer.toHexString(value);
        String inputNBInHexa = Integer.toHexString(outputNB);
        return tcpProxy.writeSDO(nodeID, "6200", inputNBInHexa, "2", valueInHexa);
    }
    
    /**
     * returns the value in hexa of all the sensors connected on the inputs channels of
     * the device. cf CAN-CBX-DIO8
     * examples : 
     * if returns "80" means that the input 8 is set to 1, the others to 0
     * if returns "A0" means that the inputs 7 and 5 are set to 1, the others to 0
     * if returns "FF" all the inputs are set to 1
     * if returns "00" all the inputs are set to 0
     * 
     */   
    @Override
    public String readNewHexaValue() {
        try {
            return tcpProxy.readSDO(this.nodeID, "6000", "1");
        } catch (SDORequestError ex) {
            String message = getName() + ": ERROR IN READING SENSOR";
            log.error(message);
            FcsAlarm alarm = new FcsAlarm(ex.toString());
            sendToStatus(alarm);
            //this.getSubsystem().updateState(State.InError,getName() + ":CanOpen Error in reading sensor.");
            return null;
        }
    }
    
    /**
     * This methods initialize the CAN-CBX-DIO8 with the mask (value 3) to configure the 
     * 2 first channels of the device to be output channel.
     * (3 = 00000011)
     * @return mask Value as a String (hexa)
     * @throws SDORequestError 
     */
    public String writeMask() throws SDORequestError {
        return tcpProxy.writeSDO(nodeID, "2250", "1", "1", "3");
    }
    
    /**
     * This methods checks if the mask of the device is the good one.
     * The good one is when the first 2 channels are output channels, that means that 
     * the value of the mask is 3 (3= 00000011)
     * @return true if the mask is OK, false otherwise
     * @throws SDORequestError 
     */
    public boolean checkMask() throws SDORequestError {
        String maskValue = tcpProxy.readSDO(nodeID,"6000","1");
        return maskValue.equals("3");
    }



    @Override
    public void initialize() {
        throw new UnsupportedOperationException("Not supported yet.");
    }

    @Override
    public int readNewValue() {
        throw new UnsupportedOperationException("Not supported yet.");
    }



    
}
