/*
 * To change this template, choose Tools | Templates
 * and open the template in the editor.
 */
package org.lsst.ccs.subsystems.fcs.drivers;

import org.lsst.ccs.framework.Module;
import org.lsst.ccs.subsystems.fcs.common.PieceOfHardware;
import org.lsst.ccs.subsystems.fcs.errors.HardwareNotDetectedException;
import org.lsst.ccs.subsystems.fcs.errors.SDORequestError;

/**
 * This is the driver class for the DAC we use in the FCS test bench: 
 * CAN-CBX-AO412
 * @author virieux
 */
public class CanOpenDAC extends Module implements PieceOfHardware {
    
    /**
    * In order to have 1 Volt send to the output we have to write 3210.27287 in decimal.
    * 
    */
    public static final double bitOverVolts = 3210.27287; 

    CanOpenProxy tcpProxy;
    
    String serialNB;
    String nodeID;
    
    //FOR SPRING
    public CanOpenDAC() {       
    }

    public CanOpenDAC(String nodeID, String serial) {
        this.tcpProxy = (CanOpenProxy) this.getModule("tcpProxy");
        this.serialNB = serial;
        this.nodeID = nodeID;       
    }

        
    /**************************************************************************************************/
    /********************** SETTERS AND GETTERS  ******************************************************/
    /**************************************************************************************************/
    
    public String getSerialNB() {
        return this.serialNB;
    }

    public String getNodeID() {
        return this.nodeID;
    }


    /**
     * @param serialNB the serialNB to set
     */
    public void setSerialNB(String serialNB) {
        this.serialNB = serialNB;
    }

    /**
     * @param nodeID the nodeID to set
     */
    public void setNodeID(String nodeID) {
        this.nodeID = nodeID;
    }

    public boolean isConfigOK() {
        return tcpProxy.isConfigOK(this);
    }
    /**************************************************************************************************/
    /********************** END OF SETTERS AND GETTERS  ***********************************************/
    /**************************************************************************************************/

    
    @Override
    public void initModule() {
        tcpProxy = (CanOpenProxy) this.getModule("tcpProxy");
    }
    
    /**
     * Write a value to the output of the DAC
     * @param inputNB in 0..4
     * @param value given in decimal format
     */
    public void writeAnalogOutput(int inputNB, int value) throws SDORequestError, HardwareNotDetectedException {
        
        if (!this.tcpProxy.isBooted(this)) {
            String message = String.format("DAC %s / nodeID=%s is NOT BOOTED", getName(),nodeID);
            log.error(message);
            //this.getSubsystem().updateState(State.InError, message );
            
            throw new HardwareNotDetectedException("CanOpenDAC is MISSING", getName(),this.nodeID, this.serialNB);
        }
        
        if ((inputNB < 1) || (inputNB > 4)) 
            throw new IllegalArgumentException("CAN-CBX-AO412 : input has to be 1..4");
        //TODO test the value < 65535
        String valueInHexa = Integer.toHexString(value);
        String inputNBInHexa = Integer.toHexString(inputNB);
        tcpProxy.writeSDO(nodeID, "6411", inputNBInHexa, "2", valueInHexa);
    }

    public void initialize() {
        throw new UnsupportedOperationException("Not supported yet.");
    }
    
    public String toString() {
        StringBuilder sb = new StringBuilder(getName());
        sb.append("/");
        sb.append(this.serialNB);
        sb.append("/");
        // because the configuration system can call toString before the object is completely built
        // for tracing purpose.
        if (tcpProxy == null ) return sb.toString();
        if (isConfigOK())  {
            sb.append("is booted");
            sb.append("/CanOpen NodeID in hexa= ");
            sb.append(this.nodeID);
            int x = Integer.getInteger(this.nodeID, 16);
            sb.append(" in decimal= ");sb.append(x);
        } else {
            sb.append("NOT DETECTED");
        }
        return sb.toString();
    }
    
    /****************************************************************/
    /* The following methods are common to all the Can Open hardware*/
    /****************************************************************/
    
    public String writeSDO(String index, String subindex,String length, String newValue) throws SDORequestError {
        return tcpProxy.writeSDO(this.nodeID,index,subindex,length,newValue);
    }
    
    public String readSDO(String index, String subindex) throws SDORequestError {
        return tcpProxy.readSDO(this.nodeID, index, subindex);
    }

    
}
