/*
 * To change this template, choose Tools | Templates
 * and open the template in the editor.
 */
package org.lsst.ccs.subsystems.fcs;

import java.util.concurrent.locks.Condition;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;
import org.lsst.ccs.framework.Module;
import org.lsst.ccs.subsystems.fcs.common.Sensor;
import org.lsst.ccs.subsystems.fcs.errors.HardwareError;
import org.lsst.ccs.subsystems.fcs.errors.SensorValueOutOfRangeException;

/**
 * This class represents an ADC which is a  12bits sensor : 
 * the value returned by the sensor is 
 * coded with 14 bits, so its value is between 0 and 32760.
 * @author virieux
 */
public abstract class Sensor14bits extends Module implements Sensor {
    
    
   protected int value;   
   int minValue = 0;
   int maxValue = 32768; //because it's on 14 bits
   
   //updatingValue is true while we are reading the value sent by the Sensor
   //this could take some time
   volatile boolean updatingValue = false;
   final Lock lock = new ReentrantLock();
   final Condition valueUpdated = lock.newCondition();
   
//   //constrcuteur pour String 
//    public Sensor14bits() {}

 //Constructeur if we don't have Spring.
   public Sensor14bits() {
       this.updatingValue = false;
       this.value = 0;
       this.minValue = 0;
       this.maxValue = 32768;          
   }
   
    @Override
   public void initModule() {
       this.value=minValue;
       this.minValue = 0;
       this.maxValue = 32768;
   }
   
  /**
   * This method returns value if the sensor is not updating otherwise it waits
   * until the update is completed and returns the new read value from the hardware.
   * @return value  
   */
    @Override
   public int getValue() {
       lock.lock();
       try {
           while (updatingValue) {
                try {
                    //TODO put a timeout
                    valueUpdated.await();
                } catch (InterruptedException ex) {
                    log.error(getName() + ex.getMessage());
      
                }
           }
           return value;
       } finally {
           lock.unlock();
       }
   }
   
   //for Spring
   public void setValue(int aValue) {
       this.value = aValue;
   }

    /**
     * @return the maxValue
     */
    public int getMaxValue() {
        return maxValue;
    }

    public int getMinValue() {
        return minValue;
    }

    //for SPRING
    public void setMinValue(int minValue) {
        this.minValue = minValue;
    }
    
    
       
   /**
   * This method updates the sensor value : first it reads a new value from the 
     * hardware or from the simulated sensor,
     * then it checks if the value is in the range between minimal value and 
     * maximal value. If it's ok it updates the sensor value with the new value, 
     * otherwise it throws an exception.
   * 
   */
    @Override
   public void updateValue() throws SensorValueOutOfRangeException, HardwareError {

       lock.lock();
       
       updatingValue = true;
       try {
           int newValue = readNewValue();
           if ((newValue < minValue) || (newValue > maxValue)) {
                throw new SensorValueOutOfRangeException("ERROR in Sensor14bits during updateValue for ", this.getName(), this.minValue, this.maxValue, newValue);
           }
           this.value = newValue;

       } finally {
           updatingValue = false;
           valueUpdated.signal();
           lock.unlock();
       }
   }

    @Override
    public String toString() {
        StringBuilder sb = new StringBuilder(getName());
        sb.append("/returned value=");
        sb.append(Integer.toString(value));
        return sb.toString();
    }

    
                 
}
