package org.lsst.ccs.subsystems.fcs;


import java.util.logging.Level;
import java.util.logging.Logger;
import org.lsst.ccs.bus.CommandReply;
import org.lsst.ccs.bus.CommandReply.CommandStatus;
import org.lsst.ccs.subsystems.fcs.common.Actuator;
import org.lsst.ccs.subsystems.fcs.common.FlipRail;
import org.lsst.ccs.subsystems.fcs.common.FlipRailState;
import org.lsst.ccs.bus.BadCommandException;
import org.lsst.ccs.bus.ErrorInCommandExecutionException;
import org.lsst.ccs.framework.Module;

/**
 * The fliprail is a rail for the auto-changer.
 * The auto-changer trucks can go along this fliprail when it's closed (aligned)
 * with the rail of the auto-changer. It should disapear in the finale version.
 * @author virieux
 */
public class FlipRailModule extends Module implements FlipRail {
	

	
	/**
	 * 
	 */
	private static final long serialVersionUID = 5211766913111710943L;
	FlipRailState state;
	Actuator actuator;

 
	
	String stateOutputName = "Flip Rail State";
	
	
	/**
	 * minimal autochanger trucks position to permit to fliprail to on.
	 */
	private double trucksPositionMin = 30;
	

	public FlipRailModule() {
	}

  
	
	/**
	 * close fliprail (align fliprail with autochanger rail)
	 * The fliprail has to be closed to be able to move autochanger.
	 */
	public synchronized String close() throws ErrorInCommandExecutionException {
		log.info(getName() + ": closing ");
        try {
            actuator.on();
        } catch (BadCommandException ex) {
            Logger.getLogger(FlipRailModule.class.getName()).log(Level.SEVERE, null, ex);
        }
		
		//TODO in real life : check if flip rail is really closed
		// if it is not closed, throw an exception
		setState(FlipRailState.CLOSED);
		
		if (!(state.equals(FlipRailState.CLOSED))) 
			throw new ErrorInCommandExecutionException("Error in execution of command close for " + getName());
		

		this.sendToReply(new CommandReply("Fliprail is closed", CommandStatus.OK));
		return getName() + " closed";
	}

	/**
	 * on fliprail (it goes out carousel way.)
	 * Fliprail has to be on to be able to rotate carousel.
	 * autochanger trucks position must be greater than SWAPOUT position ?
	 * @throws BadCommandException 
	 */
	public synchronized String open() throws ErrorInCommandExecutionException, BadCommandException {
		
		if (getAutochangerTrucksPosition() < trucksPositionMin)
			throw new BadCommandException(getName() + ": unable to open because autochanger truck is on the way.");
		log.info(getName() + ": opening ");
		actuator.off();
		// to do in real life : check if flip rail is really closed
		// if it is not closed throw an exception
		setState(FlipRailState.OPEN);

		if (!(state.equals(FlipRailState.OPEN))) 
			throw new ErrorInCommandExecutionException("Error in execution of command open for " + getName());
		

		this.sendToReply(new CommandReply("Fliprail is open", CommandStatus.OK));
		return getName() + " open";
	}
	
	public void goHome()  {
		//open();		
	}

	@Override
	public void initModule () 
	{
		log.info("[FlipRailModule] Initializing the Flip Rail  module ");
		try {
			
			// if state = unknown go to home position
			goHome();
			//TODO in real life : check flip rail state
			setState(FlipRailState.OPEN);


		} catch (final RuntimeException e) {
			log.fatal("[Flip RAil Module] ERROR: " + this.name +  
					" Initialization failed for Flip Rail module : " + e.getMessage());
			throw  e;
			
		}
	}
	
	/**
	 * What has to be done for each tick of the timer.
	 * We have to notify our observers we have changed 
	 * and give them the new values of the update data. 
	 */

	@Override
	public void tick() {
		try {
			setChanged();
			notifyObservers(new ValueUpdate(stateOutputName, this.getState()));
	
		} catch (Exception e) {
			log.error(e);
		}

	}
	
	
	public String stop() {
		return getName() + ": stopped";
		
	}
	
	private double getAutochangerTrucksPosition() {
            AutoChangerModule autochanger =  (AutoChangerModule) this.getModule("autochanger");
            return autochanger.getTrucksPosition();
        }

//setters and getters
	
	public FlipRailState getState() {
		return this.state;
	}


	public void setState(FlipRailState state) {
		this.state = state;
	}

	public Actuator getActuator() {
		return actuator;
	}

	public void setActuator(Actuator actuator) {
		this.actuator = actuator;
	}






}
