/*
 * To change this template, choose Tools | Templates
 * and on the template in the editor.
 */
package org.lsst.ccs.subsystems.fcs;

import java.util.concurrent.locks.Condition;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.lsst.ccs.bus.BadCommandException;
import org.lsst.ccs.bus.ErrorInCommandExecutionException;
import org.lsst.ccs.framework.Module;
import org.lsst.ccs.subsystems.fcs.FcsEnumerations.FilterPresenceInLatchStatus;
import org.lsst.ccs.subsystems.fcs.FcsEnumerations.LockStatus;
import org.lsst.ccs.subsystems.fcs.errors.HardwareError;
import org.lsst.ccs.subsystems.fcs.utils.FcsUtils;

/**
 * This is a model for a latch. 
 * A latch is what holds a filter in the trucks of the autochanger.
 * There is 2 latches : one on each side of the filter.
 * The state of the  latches are monitoring by digital sensors :
 * one sensor to know if a filter is in the latch or not and 
 * 2 status sensors to know if the latch is locked or unlocked.
 * If the 2 status sensors are inconsistent, the latch is in error.
 * @author virieux
 */
public class FilterLatchModule extends Module {


    private static class LatchError extends Exception {
        FilterLatchModule latch;
               
        public LatchError(String string, FilterLatchModule aThis) {
            super(string);
            latch = aThis;
        }
    }


    private boolean filterEngaged;
    private boolean locked;
    private boolean unlocked;
    private boolean inError;
    
    private FilterPresenceInLatchStatus presenceStatus;
    private LockStatus lockStatus;
    
    LatchActuatorModule latchActuator;
    protected NumericSensor filterPresenceSensor;
    protected NumericSensor lockSensor;
    protected NumericSensor unlockSensor;
    CompactIOModule filterSensorsDIO;
    
    
    //Used because we have to wait for the update from the sensors to know the state
    //of the latch
    final Lock lock = new ReentrantLock();
    final Condition stateUpdated = lock.newCondition();
    
     /**
     * This is used when we update the latch state with the values returned 
     *  by the sensors.
     */
    protected volatile boolean updatingState = false;



 
    
    
    
    /**************************************************************************************************/
    /********************** SETTERS AND GETTERS  ******************************************************/
    /**************************************************************************************************/
    
    public LatchActuatorModule getLatchActuator() {
        return latchActuator;
    }

    public void setLatchActuator(LatchActuatorModule latchActuator) {
        this.latchActuator = latchActuator;
    }

    public NumericSensor getFilterPresenceSensor() {
        return filterPresenceSensor;
    }

    public void setFilterPresenceSensor(NumericSensor filterPresenceSensor) {
        this.filterPresenceSensor = filterPresenceSensor;
    }

    public NumericSensor getLockSensor() {
        return lockSensor;
    }

    public void setLockSensor(NumericSensor lockSensor) {
        this.lockSensor = lockSensor;
    }

    public NumericSensor getUnlockSensor() {
        return unlockSensor;
    }

    public void setUnlockSensor(NumericSensor unlockSensor) {
        this.unlockSensor = unlockSensor;
    }

    public CompactIOModule getFilterSensorsDIO() {
        return filterSensorsDIO;
    }

    public void setFilterSensorsDIO(CompactIOModule filterSensorsDIO) {
        this.filterSensorsDIO = filterSensorsDIO;
    }
    
    


    /**************************************************************************************************/
    /********************** END OF SETTERS AND GETTERS  ***********************************************/
    /**************************************************************************************************/

    @Override
    public void initModule() {
        this.lockStatus = LockStatus.UNKNOWN;
        this.presenceStatus = FilterPresenceInLatchStatus.UNKNOWN;
        this.filterEngaged = false;
    }

    public LockStatus getLockStatus() {
        
        lock.lock();
        try {
            while(updatingState) {
                try {
                    this.stateUpdated.await();
                } catch (InterruptedException ex) {
                    Logger.getLogger(FilterClampModule.class.getName()).log(Level.SEVERE, null, ex);
                }

            }
            return lockStatus;
            
        } finally {
            lock.unlock();
        }
        
    }

    public void setLockStatus(LockStatus lockStatus) {
        this.lockStatus = lockStatus;
    }
    
    public FilterPresenceInLatchStatus getPresenceStatus() {
        
        lock.lock();
        try {
            while(updatingState) {
                try {
                    this.stateUpdated.await();
                } catch (InterruptedException ex) {
                    Logger.getLogger(FilterClampModule.class.getName()).log(Level.SEVERE, null, ex);
                }

            }
            return presenceStatus;
            
        } finally {
            lock.unlock();
        }
        
    }

    
    public StatusDataPublishedByLatch getStatusData() {
        StatusDataPublishedByLatch status = FcsUtils.createStatusDataPublishedByLatch(this);
        return status;
    }

    
    public String open()  throws BadCommandException, ErrorInCommandExecutionException, HardwareError {
        //TODO : check conditions
        
        log.debug(getName() + " IS OPENING");
        latchActuator.open();
        try {
            log.debug(getName() + ": Waiting for the latch to open");
            Thread.sleep(4000);
        } catch (InterruptedException ex) {
            log.error(ex);
        }

        filterSensorsDIO.updateValue();
        
        String filterSensorsHexaValue = filterSensorsDIO.getHexaValue();

        this.updateState(filterSensorsHexaValue);
        //In engineering mode we want to update one latch at a time.
        //autochanger.updateLatchesStateWithSensors();

        while(updatingState) {
            try {
                log.debug(getName() + " : Waiting for state to be updated");
                stateUpdated.await();               
            } catch (InterruptedException ex) {
                Logger.getLogger(FilterClampModule.class.getName()).log(Level.SEVERE, null, ex);
            }
        }
               
        if (this.isLocked() || this.isInError()) {
            String message;
            if (this.isInError()) {
                message = " unlockSensor and lockSensor disagree";
            } else {               
                message = " unable to unlock latch";
            } 
            throw new ErrorInCommandExecutionException(getName() + message);
        }
        
        if (this.isUnlocked()) {
            latchActuator.powerOff();
            return getName() + ": Unlocked";
        } 
        
        log.debug(getName() + " IS OPEN");
        return getName() + " IS OPEN";      
    }


    public String close() throws BadCommandException, ErrorInCommandExecutionException, HardwareError {
        //TODO : check conditions
        if (!this.isUnlocked()) {
            throw new BadCommandException(getName() + "Can't lock a latch which is already locked");
        }
        log.debug(getName() + " LOCKING");
        latchActuator.close();
        try {
            log.debug(getName() + ": Waiting for the latch to close");
            Thread.sleep(4000);
        } catch (InterruptedException ex) {
            log.error(ex);
        }

        filterSensorsDIO.updateValue();
 
        String filterSensorsHexaValue = filterSensorsDIO.getHexaValue();
        this.updateState(filterSensorsHexaValue);

        
        //In engineering mode we want to update one latch at a time.
        //autochanger.updateLatchesStateWithSensors();       
        
        while(updatingState) {
            try {
                stateUpdated.await();
            } catch (InterruptedException ex) {
                log.error(getName() + ": Interruption while waiting for the state update. ");
            }
        } 
        
        //TODO some refactoring to be sure to test all the cases
        if (this.isInError()) {
            throw new ErrorInCommandExecutionException(getName() + " unlockSensor and lockSensor disagree");
        }
               
        if (this.isUnlocked()) {
            throw new ErrorInCommandExecutionException(getName() + " unable to lock latch");
        }
        
        if (this.isLocked()) {
            latchActuator.powerOff();
        } 
        
        log.debug(getName() + " IS CLOSED");
        return getName() + " IS CLOSED";
        
              
    }

    //TODO : wait for the update of the state
    public boolean isLocked() {       
        return locked;
    }


    public boolean isUnlocked() {
        return unlocked;
    }
    

    public boolean isInError() {
        return inError;
    }

    
    
    public void updateState(String hexaValue) {
        lock.lock();
        try {
            updatingState = true; 
            this.filterPresenceSensor.updateValue(hexaValue);
            this.lockSensor.updateValue(hexaValue);
            this.unlockSensor.updateValue(hexaValue);
            
            if (this.filterPresenceSensor.getDigitalValue() == 1) {
                filterEngaged = true;
                this.presenceStatus = FilterPresenceInLatchStatus.ENGAGED;
            } else {
                filterEngaged = false;
                this.presenceStatus = FilterPresenceInLatchStatus.NOFILTER;
            }
            if (this.lockSensor.getDigitalValue() == 1) {
                locked = true;
            } else {
                locked = false;
            }
            if (this.unlockSensor.getDigitalValue() == 1) {
                unlocked = true;
            } else {
                unlocked = false;
            }
            if (locked && unlocked) {
                inError = true;
            } else {
                inError = false;
            }
            
            if (inError) {
                lockStatus = LockStatus.ERROR;
            } else if (locked) {
                lockStatus = LockStatus.LOCKED;
            } else if (unlocked) {
                lockStatus = LockStatus.UNLOCKED;
            } else {
                lockStatus = LockStatus.UNKNOWN;
            }

            

            
        } finally {
            updatingState = false;
            stateUpdated.signal();
            lock.unlock();
        }
        this.sendToStatus(this.getStatusData());        

    }


    
}
