package org.lsst.ccs.subsystems.fcs;


import org.lsst.ccs.subsystems.fcs.errors.HardwareError;
import java.util.HashMap;
import java.util.Map;
import java.util.Observable;
import org.lsst.ccs.subsystems.fcs.common.FilterLocation;
import org.lsst.ccs.bus.BadCommandException;
import org.lsst.ccs.bus.ErrorInCommandExecutionException;
import org.lsst.ccs.framework.Module;
import org.lsst.ccs.subsystems.fcs.errors.CanOpenError;

/**
 * The main module in FCS.
 * Commands can be sent to the FcsMainModule from the CCS Master Control Module (MCM)
 * in normal operation mode or from a Console (org.lsst.ccs.subsystem.console.TestConsole).
 * 
 * <P/>Example of commands that can be sent to the FcsMainModule :
 * <BR/>
 * <TT>
 * Console> invoke fcs-test MoveFilterToOnline <i>filtername</i>
 * </TT>
 * <P/>The main goals of the FcsMainModule is :
 * <UL>
 *  <LI>Receive commands from MCM for the whole subsystem.</LI>
 *  <LI>Execute commands received from MCM in invoking methods on the carousel Module 
 * and on the autochanger Module.
 *  </LI>
 *  <LI>Handle the logic of the whole subsystem.</LI>
 *  <LI>Publish on the status bus the status of the whole subsystem.</LI>
 * </UL>
 * @author virieux
 *
 */
public class FcsMainModule extends Module {
	
	/**
	 * 
	 */
	private static final long serialVersionUID = 7669526660659959402L;
	
	private CarouselModule carousel;
	private AutoChangerModule autochanger;
	
	private Double standbyPosition;

	/**
	 * A map to store the filters that FCS manages, by name.
	 * String = filter name
	 */
	private Map<String, Filter> filters = new HashMap<String, Filter>();
	
	private static final String ack = "[FCSMainModule]";	
	
	/**
         * In the initialisation, we want to do some software initialization,
         * and some hardware initialization.
         * We publish the initial data for each piece of hardware : carousel
         * and autochanger.
         */
	public void initModule()  {
		log.info("[FcsMainModule] Initializing the FCS Main module ");
		try {
			carousel = (CarouselModule) this.getModule("carousel");
			autochanger = (AutoChangerModule) this.getModule("autochanger");
			
			// TODO in real life : check if every piece of hardware : carousel, fliprail, autochanger, latch, clamps are OK
			// when every hardware pieces are OK, FCS is READY
			// check where are the filters
			this.locateFilters();
                        
			autochanger.locateTrucks();
                        
                        log.info("SUBSYSTEM NAME= " + this.getSubsystem().getName());

                        publishData(CarouselModule.publishedByCarouselOutputName, carousel.getStatusData());
                        publishData(AutoChangerModule.publishedByAutoChangerOutputName, autochanger.getStatusData());

		} catch (final RuntimeException e) {
			log.fatal("[FcsMainModule] ERROR: " + this.name +  " Initialization failed for FCS main module : " + e.getMessage());
			throw e;
			
		}

	}
	

	/**
	 * What to do when the Modules we observe (carousel and autochanger) update
         * their data to publish : we publish on status bus.
	 */
	public void processUpdate(Observable source, ValueUpdate v) {
            
                publishData(v.getName(), v.getValue());
	}
	
	//for tests purpose mainly
        public String printFilters() {
	
            String message = ack + " printFilters \n ";
            for(String key : filters.keySet()) {
		message = message + filters.get(key).getName() + " " + filters.get(key).getFilterLocation();	
		if (filters.get(key).isOnCarousel()) {
			message = message + " position: " + this.carousel.getFilterPosition(filters.get(key));
			message = message + " socket number: " + this.carousel.getSocketNumber(filters.get(key));
			message = message + " standby position: " + this.carousel.getStandbyPositionForFilter(filters.get(key));
		}
		message = message + " \n ";
            }
	
            log.info(message);
            return message;
	
        }

        /**
         * This method can be invoked from the Console to display at the Console
         * status data of the whole subsystem.
         * @return 
         */
        public String displayData() {
            StringBuilder sb = new StringBuilder("Data published by ");
            sb.append(getContext().getSubsystem().getName()).append("\n  ");
            sb.append(CarouselModule.publishedByCarouselOutputName);
            sb.append(carousel.getStatusData().toString()).append("\n  ");
            sb.append(AutoChangerModule.publishedByAutoChangerOutputName);
            sb.append(autochanger.getStatusData().toString());
            return sb.toString();
        }

    @Override
        public String toString() {
            StringBuilder sb = new StringBuilder(getName());
            sb.append(carousel.toString());
            sb.append(autochanger.toString());
            return sb.toString();
        }

        // Commands that can be sent to FcsMainModule from the console
	
	//Fixme
	public void shutdown() {		
		getContext().getSubsystem().shutdown();		
	}
	
	/**
	 * This method stops the FCS subsystem given as an argument.
	 * @param moduleName cannot be null / must be : carousel, autochanger or fliprail
	 * @throws IllegalArgumentException
	 */
	public String stop(String moduleName) throws IllegalArgumentException {
		if (moduleName == null) {
			throw new IllegalArgumentException("Please enter the FCS subsystem you want to stop : carousel, autochanger or fliprail");
		}
		if (moduleName.equals("carousel")) {
			return carousel.stop();
		} else if (moduleName.equals("autochanger")) {
			return autochanger.stop();
		} else if (moduleName.equals("fliprail")) {
			return autochanger.getFliprail().stop();
		} else {
			throw new IllegalArgumentException("Bad module name : must be carousel or autochanger or fliprail ");
		}
	}
	

	/**
	 * Move a filter to online position.
	 * The filter must be in the camera. 
	 * It can be on the carousel or on the autochanger.
	 * This method can be called by the end-user at the console.
	 * @param filterName name of the filter to move
	 * @return message for the user
	 * @throws InterruptedException, IllegalArgumentException
	 * @throws BadCommandException 
	 * @throws ErrorInCommandExecutionException 
	 */
	public String moveFilterToOnline(Object filterName) 
		throws InterruptedException, IllegalArgumentException, BadCommandException, ErrorInCommandExecutionException, HardwareError, CanOpenError {
	
		controlFilterName(filterName);
		
		// the entered filter name is correct
		log.info("Filter to move : " + filterName);
		log.info("Filter location :" + getFilterByName((String) filterName).getFilterLocation());
		return ack + moveFilterToOnline(getFilterByName((String) filterName));

	}
	
	/**
	 * Move a filter to standby position.
	 * The filter must be in the camera. 
	 * The filter can be on carousel or on autochanger.
	 * This method can be called by the end-user at the console.
	 * @param filterName name of the filter to move
	 * @return message for the user
	 * @throws InterruptedException, IllegalArgumentException
	 * @throws BadCommandException 
	 * @throws ErrorInCommandExecutionException 
	 */	
	public String moveFilterToStandby (Object filterName) 
		throws InterruptedException, IllegalArgumentException, BadCommandException, ErrorInCommandExecutionException {
		
		controlFilterName(filterName);
		
		// the entered filter name is correct
		log.info("Filter to move : " + filterName);
		log.info("Filter location : " + getFilterByName((String) filterName).getFilterLocation());				
		return ack + moveFilterToStandby(getFilterByName((String) filterName));
			
	}

//TODO IN REAL LIFE : rotate carousel, identify each filter on each socket
	private void locateFilters() {

                carousel.getSockets()[0].putFilterOnSocket(filters.get("filterU"));
                carousel.getSockets()[1].putFilterOnSocket(filters.get("filterG"));
                carousel.getSockets()[2].putFilterOnSocket(filters.get("filterR"));
                carousel.getSockets()[3].putFilterOnSocket(filters.get("filterI"));
                carousel.getSockets()[4].putFilterOnSocket(filters.get("filterZ"));

                carousel.engageClampsContact();

                filters.get("filterY").setFilterLocation(FilterLocation.OUT);

	}

	
	/**
	 * A method to move a filter in the camera to online position. 
         * It envolves the carousel Module and the autochanger Module.
         * If the filter we want to move is on the autochanger, the autochanger has to move it to online position.
         * If the filter is on the carousel, and the autochanger truck is loaded with another filter, the autochanger has 
         * to move back the other filter on the carousel at standby position. 
         * If the filter is on the carousel and the autochanger is empty, we have to rotate the carousel 
         * in order to put the required filter at standby position, then the autochanger grabbes the filter on the carousel 
         * and moves it to online position.
	 * @param filter filter to move
	 * @return message for the user
	 * @throws InterruptedException 
	 * @throws InterruptedException 
	 * @throws BadCommandException if the carousel is already rotating or the autochanger is already moving when 
         * the command is sent : we want to receive a command stop before executing a new command.
	 */
	private String moveFilterToOnline(Filter filter) throws InterruptedException, BadCommandException, ErrorInCommandExecutionException, HardwareError, CanOpenError {

                //The carousel and the autochanger must not move, otherwise they have to be stopped before a new command.
                if (carousel.isRotating() || autochanger.isMoving()) throw new BadCommandException(getName()
                        + " Please wait the end of the last command or send a stop command.");


		//The filter we want to move is on autochanger.
		if (filter == autochanger.getFilterOnTrucks()) {
			return autochanger.moveFilterToOnline(filter);
		} 

                //The filter we want to move is not on the autochanger.

                //The filter we want to move is on carousel.
		if (filter.isOnCarousel()) {

                        //a filter is held by the the autochanger
                        //we have to put it back on the carousel before moving the carousel
			if (!autochanger.isTrucksEmpty()) {
				Filter filterOnTrucks = autochanger.getFilterOnTrucks();
				// another filter in on the autochanger : we have to put it on carousel
                                // the standby position on carousel must be empty
                                if (!carousel.isReadyToGrabbeAFilterAtStandby())
                                    throw new ErrorInCommandExecutionException(getName() + "/Carousel is not ready to grabbe a filter at standby position");
				autochanger.moveFilterToStandby(filterOnTrucks);
				carousel.grabbeFilterAtStandby(filterOnTrucks);
				autochanger.ungrabFilterAtStandby();
			}
									
			carousel.moveFilterToStandby(filter);
			autochanger.grabFilterAtStandby(filter);
			carousel.ungrabbeFilterAtStandby(filter);
			autochanger.moveFilterToOnline(filter);
                        carousel.releaseClamps();
			autochanger.closeOnlineClamp();
                        return(getName() + ": command moveFilterToOnline completed / filter on line: " + filter.getName());
		}
		
		//assert : we should not go there
		String message = "Panic : filter is neither on carousel, neither on autochanger";
		log.error(message);
		return message;
	}

	/**
	 * Move a filter in the camera to standby positionon the carousel.
         * If the filter is on the carousel, the command is sent to the carousel,
         * if the filter is on the autochanger the command is sent to the autochanger.
         * 
	 * @param filter : filter to move. It can be on the carousel or on the autochanger.
	 * @return message
	 * @throws BadCommandException
	 * @throws InterruptedException
	 * @throws ErrorInCommandExecutionException 
	 */
	private String moveFilterToStandby(Filter filter) throws BadCommandException, InterruptedException, ErrorInCommandExecutionException {
		String message = null;
		
		if (filter.isOnCarousel()) {						
			message = carousel.moveFilterToStandby(filter);
			
		} else if (filter.isOnAutoChanger()) {
			message = autochanger.moveFilterToStandby(filter);;
		
		}
		return message;		
	}



	public Filter getFilterAtStandby() {
		return carousel.getFilterAtStandby();
	}
		
	/**
         * This method controls if the filter name given at the console by an operator is correct.
         * 
         * @param o the object we receive from the command line at the Console.
         * @throws IllegalArgumentException in this case :
         *   <UL>
         *      <LI>the name is not a String,</LI>
         *      <LI>the name is unknown in the map filters,</LI>
         *      <LI>the name is known in the map filters, but the filter location is unknown,</LI>
         *      <LI>the name is known in the map filters, but the filter is out of the camera.</LI>
         *   </UL>
         */
	protected void controlFilterName(Object o) throws IllegalArgumentException {
					
			if (!(o instanceof String)) {
				throw new IllegalArgumentException("Incorrect filter name : " + o);
			}
		
			String filterName = (String) o;
			
			if (!(filters.containsKey(filterName))) {
				throw new IllegalArgumentException( "Unknown filter name : " + o);
			}
						
			if (getFilterByName(filterName).isUnknown()) { 				
				throw new IllegalArgumentException("Filter: " + o + " is at unknown location.");			
			} 
			
			if (this.getFilterByName(filterName).isOut()) {
				throw new IllegalArgumentException( "Filter: " + o + " is out of the camera.");
			}
	}


	public Filter getFilterByName(String filterName) {
		return filters.get(filterName);
	}

        /**
         * 
         * @return true if the socket at standby position on the carousel is empty (contains no filter)
         */
	public boolean isEmptyStandBy() {
		return false;
	}

        /**
         * 
         * @return true if there is no filter at online position. This can be because :
         * <UL>
         *  <LI>the autochanger truck is not at online position.</LI>
         *  <LI>or the autochanger truck is at online position but it's empty.</LI>
         * </UL>
         */
	public boolean isEmptyOnline() {
		return true;
	}
	


	protected Filter getFilterOnline() {
		return autochanger.getFilterOnline();
	}


	public Map<String, Filter> getFilters() {
		return filters;
	}

	public void setFilters(Map<String, Filter> filters) {
		this.filters = filters;
	}

	public Double getStandbyPosition() {
		return standbyPosition;
	}

	public void setStandbyPosition(Double standbyPosition) {
		this.standbyPosition = standbyPosition;
	}


}
