package org.lsst.ccs.subsystems.fcs;

import org.lsst.ccs.subsystems.fcs.errors.HardwareError;
import java.util.Observable;
import org.lsst.ccs.bus.CommandReply;
import org.lsst.ccs.bus.CommandReply.CommandStatus;
import org.lsst.ccs.subsystems.fcs.common.Carousel;
import org.lsst.ccs.subsystems.fcs.common.FlipRailState;
import org.lsst.ccs.subsystems.fcs.common.Motor;
import org.lsst.ccs.subsystems.fcs.common.EngineState;
import org.lsst.ccs.subsystems.fcs.common.ModuleState;
import org.lsst.ccs.subsystems.fcs.common.RunningWay;
import org.lsst.ccs.bus.BadCommandException;
import org.lsst.ccs.bus.ErrorInCommandExecutionException;
import org.lsst.ccs.framework.Module;
import org.lsst.ccs.subsystems.fcs.FcsEnumerations.FilterClampState;
import org.lsst.ccs.subsystems.fcs.common.Actuator;
import org.lsst.ccs.subsystems.fcs.common.GenericLatch;
import org.lsst.ccs.subsystems.fcs.utils.FcsUtils;

/**
 * This is a representation of the hardware of the carousel.
 * It receives commands from the FCSMAinModule and send back an anknowledge.
 * It publishes data on the status bus.
 * In engineering mode it can receive commands from the engineering console.
 * It has a latch, an engine which are connected to real or simulated hardware. 
 * @author virieux
 *
 */
public class CarouselModule extends Module implements Carousel {

    /**
     * 
     */
    private static final long serialVersionUID = -2376279469784152348L;
    private Motor carouselMotor;
    /*This actuator opens the clamp Xminus when the carousel is halted at standby position.*/
    private Actuator clampActuatorXminus;
    /*This actuator opens the clamp Xplus when the carousel is halted at standby position.*/
    private Actuator clampActuatorXplus;
    private int nbSockets;
    private CarouselSocket[] sockets = new CarouselSocket[nbSockets];
    /**
     * This latch is to hold the carousel in position.
     * It prevents it from rotating. It should be a brake.
     */
    private GenericLatch brake;
    public static String publishedByCarouselOutputName = "publishedByCarousel";
    private volatile ModuleState state = ModuleState.HALTED;
    private boolean stopped = false;
    private boolean initialized = false;

    public CarouselModule() {
    }

    //SETTERS and GETTERS
    public CarouselSocket[] getSockets() {
        return sockets;
    }

    public void setSockets(CarouselSocket[] sockets) {
        this.sockets = sockets;
    }

    @Override
    public double getPosition() {
        return carouselMotor.getPosition();
    }

    public Motor getCarouselMotor() {
        return carouselMotor;
    }

    public void setCarouselMotor(Motor engine) {
        this.carouselMotor = engine;
    }

    public GenericLatch getBrake() {
        return brake;

    }

    public void setBrake(GenericLatch latch) {
        this.brake = latch;
    }

    public synchronized ModuleState getState() {
        return state;
    }

    public synchronized void setState(ModuleState state) {
        this.state = state;
    }

    /**
     * @return the nbSockets
     */
    public int getNbSockets() {
        return nbSockets;
    }

    /**
     * @param nbSockets the nbSockets to set
     */
    public void setNbSockets(int nbSockets) {
        this.nbSockets = nbSockets;
    }

    /**
     * @return the clampActuatorXminus
     */
    public Actuator getClampActuatorXminus() {
        return clampActuatorXminus;
    }

    /**
     * @param clampActuatorXminus the clampActuatorXminus to set
     */
    public void setClampActuatorXminus(Actuator clampActuator) {
        this.clampActuatorXminus = clampActuator;
    }

    /**
     * @return the clampActuatorXminus
     */
    public Actuator getClampActuatorXplus() {
        return clampActuatorXplus;
    }

    /**
     * @param clampActuatorXminus the clampActuatorXminus to set
     */
    public void setClampActuatorXplus(Actuator clampActuator) {
        this.clampActuatorXplus = clampActuator;
    }

    /**
     * This method returns the clampX- which is at standby position.
     * It can returns null if there is no socketAtStandby halted at standby position.
     * 
     * @return 
     */
    public FilterClampModule getClampXminus() {
        CarouselSocket socketAtStandby = getSocketAtStandby();
        if (socketAtStandby == null) {
            return null;
        } else {
            return (FilterClampModule) socketAtStandby.getClampXminus();
        }
    }

    /**
     * This method returns the clampX+ which is at standby position.
     * It can returns null if there is no socketAtStandby halted at standby position.
     * 
     * @return 
     */
    public FilterClampModule getClampXplus() {
        CarouselSocket socketAtStandby = getSocketAtStandby();
        if (socketAtStandby == null) {
            return null;
        } else {
            return (FilterClampModule) socketAtStandby.getClampXplus();
        }
    }

    @Override
    public void initModule() {
        log.info("[CarouselModule] Initializing the carousel module ");
        setState(ModuleState.HALTED);
    }

    public StatusDataPublishedByCarousel getStatusData() {
        return FcsUtils.createStatusDataPublishedByCarousel(this);
    }

    public String toString() {
        StringBuilder sb = new StringBuilder(getName());
        sb.append("\n Numbers of sockets: ");
        sb.append(nbSockets);
        for (int i = 0; i < nbSockets; i++) {
            sb.append("\n Socket number ");
            sb.append(i);
            sb.append(" / ");
            sb.append(sockets[i].toString());
        }
        sb.append("\nClamps actuator  ").append(clampActuatorXminus.toString()).append("\n");
        return sb.toString();
    }

    /**
     * What to do when the Modules we observe send there new values : we notify our own observers
     */
    @Override
    public void processUpdate(Observable source, ValueUpdate v) {
        StatusDataPublishedByCarousel status = getStatusData();
        sendToStatus(status);
        setChanged();
        notifyObservers(new ValueUpdate(publishedByCarouselOutputName, status));
        log.info(getName() + ": Filter at standby position : " + getFilterInStandbyName());

    }

    @Override
    public synchronized CarouselSocket getSocketAtStandby() {
        if (this.isRotating()) {
            return null;
        } else {
            for (CarouselSocket socket : sockets) {
                if (this.getPosition() == socket.getStandbyPosition()) {
                    return socket;
                }
            }
        }
        return null;
    }

    /**
     * This methods return the filter which is in the socketAtStandby at standby position.
     * When there is no filter at standby position it returns null.
     * 
     * @return filter at standby position or null if there is no filter at standby position.
     */
    public synchronized Filter getFilterAtStandby() {

        CarouselSocket socketAtStandby = getSocketAtStandby();
        if (socketAtStandby == null) {
            return null;
        } else {

            return socketAtStandby.getFilter();

        }
    }

    public synchronized String getFilterInStandbyName() {

        Filter filterAtStandby = getFilterAtStandby();
        if (filterAtStandby == null) {
            return "none";
        } else {
            return filterAtStandby.getName();
        }

    }

    /**
     * This method let us know if the carousel is ready to receive a filter at
     * standby position : 
     *  - the carousel must not rotate
     *  - an empty socketAtStandby is at standby position.
     * @return true if the position of the carousel matches
     *              the position when one of its sockets is at standby position
     *              and this socketAtStandby is empty.
     *         false
     */
    public boolean isReadyToGrabbeAFilterAtStandby() throws HardwareError {

        if (this.isRotating()) {
            return false;
        }
        CarouselSocket socketAtStandby = getSocketAtStandby();
        if (socketAtStandby == null) {
            return false;
        }
        if (socketAtStandby.isEmpty() && socketAtStandby.isReadyToClamp()) {
            return true;
        }
        return false;
    }
    
    
    public boolean isHoldingFilterAtStandby() throws HardwareError {
        if (this.isRotating()) {
            return false;
        }
        CarouselSocket socketAtStandby = getSocketAtStandby();
        if (socketAtStandby == null) {
            return false;
        }
        if (socketAtStandby.isEmpty()) {
            return false;
        }
        if (socketAtStandby.isClampedOnFilter()) {
            return true;
        }
        return false;
        
        
    }

    public boolean isRotating() {
        return carouselMotor.getEngineState().equals(EngineState.RUNNING);
    }

    @Override
    public boolean isAbleToMove() {
        FlipRailModule fliprail = (FlipRailModule) this.getModule("fliprail");
        return !brake.isLocked() && (fliprail.state.equals(FlipRailState.OPEN));
        //return !latch.isLocked();
    }

    public boolean isLocked() {
        return brake.isLocked();
    }

    @Override
    public String stop() {
        String message;
        if (isRotating()) {
            stopped = true;
            message = "Stopping carousel";
            carouselMotor.stop();
            setState(ModuleState.HALTED);
        } else {
            message = "Carousel is already stopped";
        }

        return message;
    }

    @Override
    public String lock() {
        brake.lock();
        return getName() + " locked";

    }

    public String unlock() {
        brake.unlock();
        return getName() + " unlocked";
    }
    
    public void updateClampsStateWithSensors() throws BadCommandException, HardwareError {
        if (this.getSocketAtStandby() == null) 
            throw new BadCommandException(getName() 
                    + " can't update Clamps State because there is no socket at standby.");
        this.getSocketAtStandby().updateClampsStateWithSensors();
    }

    /**
     * Rotates carousel for an angle given as a parameter.
     * This method can be launched by the user from the TestConsole.
     * It checks the value of the rotation angle entered by the user,
     * converts it in a double,
     * and send the result of the method rotate(double angle).
     * @param angleEntered : rotation angle
     * 			The rotation angle can be positive or negative.
     * 			It must be a double and have a value between -360.0 and 360.0
     * @return the result of the command rotate(double angle)
     * @throws IllegalArgumentException, BadCommandException 
     */
    public String rotate(double angle) throws IllegalArgumentException, BadCommandException {


        //availables values for an angle : [-360,+360]
        if (angle < -360 || angle > 360) {
            throw new IllegalArgumentException("Please enter an angle between -360 and +360");
        }

        RunningWay way = null;
        double absAngle;
        if (angle >= 0) {
            absAngle = angle;
            way = RunningWay.POSITIVE;
        } else {
            absAngle = -angle;
            way = RunningWay.NEGATIVE;
        }
        double requiredPosition = addAngle(getPosition(), angle);
        return rotate(absAngle, way, requiredPosition);
    }

    /**
     * Rotate the carousel to a final position.
     * 
     * @param angle of the rotation : 0 <= angle <= 360
     * @param the final position has to be : 0 <= finalPosition <= 360
     * @param runningWay rotation way (enum)
     * @param finalPosition is the position the carousel must have
     * 			when the command is completed
     * The method rotate is delegated to the engine, which is an interface.
     * An engine can be a simulated engine or a real hardware driver.
     * @throws BadCommandException, IllegalArgumentException
     *
     */
    private String rotate(double angle, RunningWay runningWay, double finalPosition)
            throws BadCommandException, IllegalArgumentException {
        //assertion 0 <= angle <= 360
        //assertion 0 <= finalPosition <= 360
        if (angle < 0 || angle > 360) {
            throw new IllegalArgumentException(getName()
                    + " command rotate accepts angle only between 0 and 360)");
        }
        if (finalPosition < 0 || finalPosition > 360) {
            throw new IllegalArgumentException(getName()
                    + " command rotate accepts finalPosition only between 0 and 360)");
        }

        String message = null;

        if (isRotating()) {
            // Thread-safe because of encapsulation for carousel.state		
            //if (getClampsState() == ModuleState.RUNNING) {		
            message = "Carousel is rotating, stop it before sending a new rotate command.";
            log.error(message);
            throw new BadCommandException(message);
        }

        if (!this.isAbleToMove()) {
            message = "Carousel is unable to move. Check lock or flip rail.";
            log.error(message);
            throw new BadCommandException(message);
        }


        stopped = false;
        setState(ModuleState.RUNNING);

        getCarouselMotor().setRequiredPosition(finalPosition);
        this.sendToReply(new CommandReply("Please wait during carousel rotation...", CommandStatus.OK));
        carouselMotor.move(angle, runningWay);
        message = "Rotating to required position: " + finalPosition;
        log.info(message);

        while (isRotating()) {
            try {
                log.info("...carousel rotating, please wait.....");
                log.info(getName() + " WAITING TO BE AT REQUIRED POSITION=" + finalPosition);
                log.info(state.toString());
                Thread.sleep(tickMillis);
            } catch (InterruptedException e) {
                return "Sleeping interrupted";
            }
        }
        if (!isRotating()) {
            log.info("THREAD=" + Thread.currentThread().getName());
            setState(ModuleState.HALTED);
            log.info("carousel is now stopped.");
            //log.info("filter in standby position: " + getFilterInStandbyName());
            if (getPosition() == finalPosition) {
                message = "Command completed";
                log.info(message);
            } else {
                message = "Command non completed: carousel is not stopped at the required position";
                log.info(message);
            }
        }

        if (stopped) {
            message = "Command rotate carousel interrupted by command stop";
            log.info(message);
        }

        if (!stopped && (!(getPosition() == finalPosition))) {
            message = "Command non completed: carousel is not stopped at the required position";
            log.error(message);
            //TODO
            //throw new ErrorCommandException(message);
        }

        return message;

    }

    /**
     * What has to be done for each tick of the timer.
     * We have to publish on the status bus. 
     */
    @Override
    public void tick() {
        //sendToStatus(getStatusData());
    }

    /**
     * This method has no action on hardware.
     * This method updates the socketAtStandby at standby position when a filter is put on the carousel
     * at standby position.
     * A filter must be detected in the socketAtStandby at standby position (socketAtStandby not empty).
     * 
     * @param filter
     * @throws ErrorInCommandExecutionException 
     * @throws BadCommandException 
     */
    private synchronized void putFilterOnCarousel(Filter filter) throws ErrorInCommandExecutionException, BadCommandException, HardwareError {

        CarouselSocket socketAtStandby = getSocketAtStandby();

        if (socketAtStandby == null) {
            throw new BadCommandException(getName() + ": there is no socket at standby position to put a filter in.");
        }


        socketAtStandby.putFilterOnSocket(filter);
        StatusDataPublishedByCarousel status = getStatusData();
        sendToStatus(status);
        setChanged();
        notifyObservers(new ValueUpdate(publishedByCarouselOutputName, status));

    }


    /*
     * Removes a filter from carousel. This method has no action on hardware.
     * It updates the objects Socket and Filter.
     * A filter can be removed from Carousel only when the socketAtStandby 
     * is at standby position.
     */
    public synchronized void removeFilterFromCarousel(Filter filter) throws BadCommandException, HardwareError, ErrorInCommandExecutionException {

        if (filter == null) {
            throw new IllegalArgumentException(getName()
                    + " : removeFilterFromCarousel must have an argument not null");
        }
        CarouselSocket socketAtStandby = getSocketAtStandby();

        if (socketAtStandby == null) {
            throw new BadCommandException(getName() + " : "
                    + " : removeFilterFromCarousel can't remove the filter because there is no socket at standby position");
        }

        if (socketAtStandby.getFilter() == null) {
            throw new BadCommandException(getName() + " : "
                    + " : removeFilterFromCarousel can't remove the filter because there is no filter at standby position");
        }

        if (!socketAtStandby.getFilter().equals(filter)) {
            throw new BadCommandException(getName() + " : can't remove a filter which is not at standby position");
        }

        socketAtStandby.removeFilter();
        StatusDataPublishedByCarousel status = getStatusData();
        sendToStatus(status);
        setChanged();
        notifyObservers(new ValueUpdate(publishedByCarouselOutputName, status));

    }

    //TODO exception if filter is not on Carousel ?
    public double getFilterPosition(Filter filter) {
        double filterPosition = 0;
        for (int ix = 0; ix < sockets.length; ix++) {
            if (sockets[ix].getFilter() == filter) {
                filterPosition = sockets[ix].getPosition();
            }
        }
        return filterPosition;
    }

    //TODO throw an exception if filter is not on carousel.
    public double getStandbyPositionForFilter(Filter filter) {
        double position = 0;
        for (CarouselSocket socket : sockets) {
            if (socket.getFilter() == filter) {
                position = socket.getStandbyPosition();
                return position;
            }

        }
        return position;
    }

    //TODO exception if filter is not on Carousel ?
    /**
     *
     * @param filter
     * @return the position of the filter in the sockets array
     * I'm not sure this method is useful actually.
     * Only used is FcsMainModule in method printFilters.
     */
    public int getSocketNumber(Filter filter) {
        int socketNumber = 0;
        for (int ix = 0; ix < sockets.length; ix++) {
            if (sockets[ix].getFilter() == filter) {
                socketNumber = ix;
            }
        }
        return socketNumber;
    }

    /**
     * This method rotates the carousel to put the given filter in front of the autochanger
     * (standby position).
     * This method can't be launched by the user from the console.
     * It computes the shortest way for the carousel. So the running way can be positive or
     * negative.
     * This could be more complicated if the carousel was not able to rotate
     * more than TBD turns in one way.
     * @params filter : the filter can not be null and must be on the carousel
     * @throws BadCommandException
     */
    protected String moveFilterToStandby(Filter filter) throws InterruptedException, BadCommandException {

        //assert filter is on carousel
        if (!filter.isOnCarousel()) {
            throw new IllegalArgumentException("filter: " + filter.getName()
                    + " is not on carousel.");
        }
        String message = " [CarouselModule]";
        double requiredPosition = getStandbyPositionForFilter(filter);


        if (state == ModuleState.RUNNING) {
            throw new BadCommandException("Carousel is running, stop it before sending a new command.");
        }

        if (getPosition() == requiredPosition) {
            log.info(filter.getName() + " is at STANDBY position on carousel.");
            return (filter.getName() + " is at STANDBY position on carousel.");
        }

        //if carousel is locked, we have to unlock it before running.		
        if (getBrake().isLocked()) {
            log.info("Carousel is locked. Unlocking carousel");
            unlock();
        }

        double angle = addAngle(getPosition(), getFilterPosition(filter));

        //
        RunningWay runningWay;
        if (angle < 180) {
            runningWay = RunningWay.NEGATIVE;
        } else {
            angle = 360 - angle;
            runningWay = RunningWay.POSITIVE;
        }
        //Release The clamps contact before the rotation
        releaseClampsContact();
        //rotate the carousel
        rotate(angle, runningWay, requiredPosition);
        //engage the clamps contact when it's stopped at standby position
        engageClampsContact();
        //lock the carousel to hold it in position
        return message += lock();
    }

    /**
     * Print for every socket on carousel, the name of the filter it contains.
     */
    public void printSockets() {

        for (int ix = 0; ix < sockets.length; ix++) {
            log.info("Socket number: " + ix + " socket position: " + sockets[ix].getPosition()
                    + " contains filter: " + sockets[ix].getFilter().getName());

        }
    }

    public String grabbeFilterAtStandby(Object filterName) throws BadCommandException, ErrorInCommandExecutionException, HardwareError {

        FcsMainModule fcsMainModule = (FcsMainModule) this.getModule("main");
        fcsMainModule.controlFilterName(filterName);

        Filter filterToGrabbe = fcsMainModule.getFilterByName((String) filterName);

        // the entered filter name is correct
        log.info("Filter to move : " + filterName);
        log.info("Filter location : " + filterToGrabbe.getFilterLocation());
        return getName() + grabbeFilterAtStandby(filterToGrabbe);
    }

    /**
     * This method has to be executed when a filter has just been moved on the carousel 
     * at standby position by the autochanger. It updates the carousel socketAtStandby at standby or throw 
     * an exception.
     * 
     * This method tests if the clamps are locked on a filter at standby position 
     * The clamps lock automaticaly, and the filter presence sensor tells us if the filter is here or not.
     * If the clamps are locked on a filter at standby position, this method updates the socketAtStandby 
     * at standby position : it puts the filter on the socketAtStandby.
     * @param filter
     * @return
     * @throws BadCommandException
     * @throws ErrorInCommandExecutionException
     * @throws HardwareError 
     */
    public synchronized String grabbeFilterAtStandby(Filter filter) throws BadCommandException, ErrorInCommandExecutionException, HardwareError {
        if (filter == null) {
            throw new IllegalArgumentException(getName() + ": grabbeFilterAtStandby must receive a non null argument as a filter");
        }
        if (this.isRotating()) {
            throw new BadCommandException(getName() + ": grabbeFilterAtStandby can't grabbe a filter while rotating");
        }
        CarouselSocket socketAtStandby = getSocketAtStandby();

        if (socketAtStandby == null) {
            throw new BadCommandException(getName() + ": grabbeFilterAtStandby can't be executed because there is no socket at standby position.");
        }

        socketAtStandby.updateClampsStateWithSensors();

//        log.debug("ClampX- / filter presence sensor = " + getClampXminus().isFilterEngaged());
//        log.debug("ClampX+ / filter presence sensor = " + getClampXplus().isFilterEngaged());

        if ((socketAtStandby.isEmpty())) {
            throw new BadCommandException(getName() + ": grabbeFilterAtStandby can't be executed because no filter is detected in socket at standby position.");
        }


        if ((socketAtStandby.isClampedOnFilter())) {
            log.info(getName() + ": grabbing " + filter.getName() + " at standby position.");
            putFilterOnCarousel(filter);
            log.debug(getName() + ": " + filter.getName() + " is grabbed on carousel.");
            String ack = getName() + ": " + filter.getName() + " is grabbed at standby position";
            return ack;
        } else {
            throw new BadCommandException(getName() + ": grabbeFilterAtStandby can't be executed because the clamps are not CLAMPED ON FILTER.");
        }





    }

    public void ungrabbeFilterAtStandby(Filter filter) throws BadCommandException, ErrorInCommandExecutionException, HardwareError {

        if (filter == null) {
            throw new IllegalArgumentException("Filter to ungrabbe should not be null");
        }
        //            if (!autochangerLatchesAreLocked())
        //                throw new BadCommandException("Carousel can't ungrabbe filter if it's not held by autochanger");
        CarouselSocket socketAtStandby = getSocketAtStandby();

        if (socketAtStandby == null) {
            throw new BadCommandException(getName() + ":A socket has to be halted at standby position for this operation.");
        }


        if (!isOnStandby(filter)) {
            throw new BadCommandException(getName() + ": filter has to be at standby position to be ejected");
        }

        if (!socketAtStandby.isClampedOnFilter()) {
            throw new BadCommandException(getName() + ": can't ungrabbe filter which is already unclamped.");
        }

        log.debug(getName() + ": ungrabbing " + filter.getName() + " at standby position.");

        //TODO test if the filter is held by the autochanger trucks.

        // The clamps can be opened.
        unlockClamps(filter);
        log.info(getName() + ": socket at standby state : " + socketAtStandby.toString());


        log.debug(getName() + ": " + filter.getName() + " is ungrabbed from carousel.");
    }

    @Override
    public String releaseClamps() throws BadCommandException, ErrorInCommandExecutionException, HardwareError {

            CarouselSocket socketAtStandby = getSocketAtStandby();
            if (socketAtStandby == null) {
                throw new BadCommandException(getName() + 
                        ":Can't release clamps while a socket is not halted at standby position.");
            }

            socketAtStandby.updateClampsStateWithSensors();
            if (!socketAtStandby.isEmpty()) {
                throw new BadCommandException(getName() + 
                        ":Can't release clamps while "
                        + "a filter is in the socket at standby.");
            }

            if (socketAtStandby.isUnclampedEmpty()) {
                socketAtStandby.releaseClamps();
            } else {
                throw new BadCommandException(getName() + ":Can't release clamps when both clamps are not in state UNCLAMPEDEMPTY.");
            }

            socketAtStandby.updateClampsStateWithSensors();

            if (socketAtStandby.isReadyToClamp()) {
                return "Clamps are released and ready to clamp again.";
            } else {
                throw new ErrorInCommandExecutionException("Could not release clamps.");
            }
    }

    @Override
    public synchronized String unlockClamps(Filter filter)
            throws BadCommandException, ErrorInCommandExecutionException, HardwareError {

        CarouselSocket socketAtStandby = getSocketAtStandby();
        if (socketAtStandby == null) {
            throw new BadCommandException("Can't unlock clamps while a socket is not halted at standby position.");
        }

        socketAtStandby.updateClampsStateWithSensors();

        if (socketAtStandby.isClampedOnFilter()) {
            log.debug("Unlocking clamps at standby.");
            socketAtStandby.unlockClamps();
        } else {
            throw new BadCommandException("Can't unlock clamps when a filter is not clamped at standby.");
        }

        socketAtStandby.updateClampsStateWithSensors();


        //After the execution of the unlockClamps method in case of success.
        if (socketAtStandby.isUnclampedOnFilter()) {
            log.debug("Just about to remove filter from carousel");
            removeFilterFromCarousel(filter);
            log.debug("Command unlockClamps completed");
            return "Clamps unlocked";

        } else {
            log.debug(state);
            throw new ErrorInCommandExecutionException(getName() + ":Could not unlock filter at standby");

        }
    }
    
    //for ENGINEERING MODE
    public  String unlockClamps() throws BadCommandException, HardwareError, ErrorInCommandExecutionException  {
        return unlockClamps(this.getFilterAtStandby());
    }

    public synchronized boolean isOnStandby(Filter filter) {
        return this.getPosition() == this.getStandbyPositionForFilter(filter);
    }

    /**
     * Add 2 angles : an angle is a double which value is between -360 and +360
     * @param angle1
     * @param angle2
     * @return result of the angles addition. Always > 0
     */
    public static double addAngle(double angle1, double angle2) {
        //if ( abs(angle1) > 360 || abs(angle2) > 360) throw new AssertionError("Adding ");

        double angle = angle1 + angle2;
        if (angle >= 360) {
            angle -= 360;
        }
        if (angle < 0) {
            angle += 360;
        }
        return angle;
    }

    public void releaseClampsContact() {
        //TODO;
    }

    public void engageClampsContact() {
        //TODO ;
    }
}
