
package org.lsst.ccs.subsystems.fcs.ui.commons;

import java.awt.GridBagConstraints;
import java.util.HashMap;
import java.util.Map;
import javax.swing.BorderFactory;
import javax.swing.JFrame;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.JTable;
import static javax.swing.JTable.AUTO_RESIZE_ALL_COLUMNS;
import javax.swing.SwingUtilities;
import javax.swing.table.DefaultTableModel;
import javax.swing.table.TableColumn;
import org.lsst.ccs.subsystems.fcs.EPOSEnumerations.EposParameter;
import static org.lsst.ccs.subsystems.fcs.ui.commons.Tools.BIG_FONT;
import static org.lsst.ccs.subsystems.fcs.ui.commons.Tools.insets_std;
import static org.lsst.ccs.subsystems.fcs.ui.commons.Tools.NICE_PINK;
import static org.lsst.ccs.subsystems.fcs.ui.commons.Tools.resetTable;

/**
 * To display the parameters of an EPOS controller.
 *
 * @author virieux
 */
public class EPOSParametersPanel extends JPanel {
    private static final long serialVersionUID = 7733122808854397608L;

    private JTable paramsTable;
    private JScrollPane jScrollPane;
    private final JLabel mLabel = new JLabel("Parameters defined for mode: ");
    private final JLabel modeLabel = new JLabel("Unknown Mode");
    private final String controllerName;

    /**
     * A Runnable class to initialize this panel from a map of parameters.
     */
    private class GuiInitialization implements Runnable {

        private final String modeInString;
        private final Map<String, String> paramsMap;

        GuiInitialization(String mode, Map<String, String> paramsMap) {
            this.modeInString = mode;
            this.paramsMap = paramsMap;
        }

        @Override
        public void run() {
            modeLabel.setText(modeInString);
            DefaultTableModel paramsTableModel = (DefaultTableModel) paramsTable.getModel();

            /* if the table counts already some rows, we have to reset the panels to get rid of old data.*/
            if (paramsTableModel.getRowCount() > 0) {
                resetPanel();
            }
            if (!(paramsMap == null)) {
                for (Map.Entry<String, String> entry : paramsMap.entrySet()) {
                    String paramName = entry.getKey();
                    String configValue = entry.getValue();
                    EposParameter param = EposParameter.valueOf(paramName);
                    String indexHexa = "0x" + Integer.toHexString(param.getIndex());
                    String subindexHexa = "0x" + Integer.toHexString(param.getSubindex());
                    Object[] tab = {paramName, indexHexa, subindexHexa,
                        param.getSize(), param.isSigned(), configValue};
                    paramsTableModel.addRow(tab);
                }
            }
            paramsTableModel.fireTableDataChanged();
        }
    }


    /**
     * Creates new form EPOSControllerPanel
     * @param ctlName
     */
    public EPOSParametersPanel(String ctlName) {
        controllerName = ctlName;
        initComponents();
    }

    /**
     * Initializes or updates GUI with configuration INFO.
     * @param mode
     * @param paramsMap
     */
    public void initializeGUI(String mode, Map<String, String> paramsMap) {
        SwingUtilities.invokeLater(new GuiInitialization(mode, paramsMap));
    }


    private void initComponents() {

        /**
         ******************************************************************************
         ** Parameters Table
         * *****************************************************************************
         */
        paramsTable = new JTable();

        paramsTable.setModel(new DefaultTableModel(
                new Object[][]{},
                new String[]{
                    "Parameter Name", "Index", "Subindex", "Size", "Signed", "Decimal Value"
                }
        ) {
            private static final long serialVersionUID = -6158042587124966771L;
            private final Class<?>[] types = new Class[]{
                String.class, String.class, String.class, String.class, Boolean.class, Integer.class
            };
            private final boolean[] canEdit = new boolean[]{
                false, false, false, false, false, true
            };

            @Override
            public Class<?> getColumnClass(int columnIndex) {
                return types[columnIndex];
            }

            @Override
            public boolean isCellEditable(int rowIndex, int columnIndex) {
                return canEdit[columnIndex];
            }
        });

        TableColumn column;
        for (int i = 0; i < 6; i++) {
            column = paramsTable.getColumnModel().getColumn(i);
            if (i == 0) {
                column.setPreferredWidth(250); //first column is bigger
            } else if (i == 5) {
                column.setPreferredWidth(120); //6th column is bigger
            }
        }
        paramsTable.setName("paramsTable_" + controllerName);
        //paramsTable.setPreferredScrollableViewportSize(new Dimension(1000,200));
        paramsTable.setAutoResizeMode(AUTO_RESIZE_ALL_COLUMNS);
        jScrollPane = new JScrollPane(paramsTable);
        jScrollPane.setMinimumSize(new java.awt.Dimension(400, 50));
        jScrollPane.setPreferredSize(new java.awt.Dimension(500, 100));
        paramsTable.setFillsViewportHeight(true);

        /**
         ******************************************************************************
         ** end ofParameters Table
         * *****************************************************************************
         */
        /**
         ******************************************************************************
         ** Whole Panel
         * *****************************************************************************
         */
        setBorder(BorderFactory.createLineBorder(NICE_PINK));
        //setForeground(new java.awt.Color(204, 204, 255));
        setLayout(new java.awt.GridBagLayout());
        GridBagConstraints gbc = new GridBagConstraints();
        gbc.insets = insets_std;
        gbc.anchor = GridBagConstraints.NORTHWEST;
        gbc.gridx = 0;
        gbc.gridy = 0;
        add(mLabel, gbc);
        gbc.gridx++;
        modeLabel.setFont(BIG_FONT);
        add(modeLabel, gbc);
        gbc.gridx--;
        gbc.gridy++;
        gbc.gridwidth = 4;
        add(jScrollPane, gbc);
        /**
         ******************************************************************************
         ** end of Whole Panel
         * *****************************************************************************
         * *
         */
    }

    /**
     * Delete params table rows.
     * Used when GUI is disconnecting to reset params panel to empty values.
     */
    public void resetPanel() {
        resetTable(paramsTable);
    }

    public static void main(String[] argv) {

        EPOSParametersPanel p = new EPOSParametersPanel("mon_contrôleur");
        Map<String, String> paramsMap = new HashMap<>();
        paramsMap.put("ContinuousCurrentLimit", "3440");
        paramsMap.put("OutputCurrentLimit", "6880");
        paramsMap.put("MaxSpeedInCurrentMode", "2000");
        p.initializeGUI("CURRENT", paramsMap);

        JFrame frame = new JFrame("EPOS Parameters");
        frame.setContentPane(p);
        frame.pack();
        frame.setVisible(true);
        frame.setDefaultCloseOperation(JFrame.EXIT_ON_CLOSE);
    }

}
