
package org.lsst.ccs.subsystems.fcs.ui.commons;

import java.awt.Color;
import java.awt.Dimension;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.util.logging.Logger;
import javax.swing.BorderFactory;
import javax.swing.JFrame;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.SwingUtilities;
import static org.lsst.ccs.subsystems.fcs.FCSCst.UNKNOWN;
import org.lsst.ccs.subsystems.fcs.StatusDataPublishedByEPOSController;
import static org.lsst.ccs.subsystems.fcs.ui.commons.Tools.UNKNOWN_STATE;
import static org.lsst.ccs.subsystems.fcs.ui.commons.Tools.insets_std;

/**
 * A panel to display state of controller.
 *
 * @author virieux
 */
public class ControllerStatePanel extends JPanel {
    private static final Logger FCSLOG = Logger.getLogger(ControllerStatePanel.class.getName());
    private final int WIDTH = 180;
    private final int HEIGHT = 250;

    private final String controllerName;

    private final DigitalSwitch controllerSwitch = new DigitalSwitch();
    private final JLabel eposStateLabel = new JLabel();
    private final JLabel modeLabel = new JLabel();
    private final JLabel currentjLabel = new JLabel("0");
    private final JLabel positionjLabel = new JLabel("0");
    private final JLabel velocityjLabel = new JLabel("0");

    private boolean displayPosition = true;

    /**
     * Create a ControllerStatePanel
     *
     * @param ctlName controller name
     * @param displayPosition true if we want to display position (when
     * controller has a position encoder).
     */
    public ControllerStatePanel(String ctlName, boolean displayPosition) {
        this.controllerName = ctlName;
        this.displayPosition = displayPosition;
        initComponents();
    }

    public void update(StatusDataPublishedByEPOSController s) {
        FCSLOG.fine(() -> "ControllerStatePanel updating from controller data of " + controllerName);
        SwingUtilities.invokeLater(new UpdateController(s));
    }

    /**
     * A Runnable to update controller panel with data published by a
     * controller.
     */
    public class UpdateController implements Runnable {

        private final StatusDataPublishedByEPOSController s;

        private UpdateController(StatusDataPublishedByEPOSController s) {
            this.s = s;
        }

        @Override
        public void run() {
            if (s.isBooted() && s.getState() != null) {
                eposStateLabel.setText(s.getState().name());
                controllerSwitch.setColor(s.getState().getColor());
            } else if (!s.isBooted()) {
                eposStateLabel.setText("not booted");
                controllerSwitch.setColor(Color.RED);
            }

            if (s.isBooted()) {
                if (displayPosition) {
                    positionjLabel.setText(Integer.toString(s.getPosition()));
                }
                currentjLabel.setText(Integer.toString(s.getCurrent()));
                velocityjLabel.setText(Integer.toString(s.getVelocity()));
            }
            modeLabel.setText(s.getMode() == null ? UNKNOWN : s.getMode().toString());
        }
    }

    // <editor-fold defaultstate="collapsed" desc="Form Description">
    private void initComponents() {
        setDefaultValues();
        setBorder(BorderFactory.createTitledBorder(controllerName));
        setLayout(new GridBagLayout());
        this.setPreferredSize(new Dimension(WIDTH, HEIGHT));
        GridBagConstraints gbc = new GridBagConstraints();
        gbc.insets = insets_std;
        gbc.anchor = GridBagConstraints.CENTER;
        gbc.gridx = 0;
        gbc.gridy = 0;
        gbc.gridwidth = 2;
        controllerSwitch.setSize(40);
        add(controllerSwitch, gbc);
        gbc.gridy++;
        add(eposStateLabel, gbc);
        gbc.gridy++;
        add(modeLabel, gbc);

        /* current, position, velocity */
        gbc.anchor = GridBagConstraints.WEST;
        gbc.gridwidth = 1;
        gbc.gridy = 4;
        add(new JLabel("<HTML>Current : <BR>(in mA) </HTML>"), gbc);
        gbc.gridy++;
        add(new JLabel("<HTML>Velocity : <BR>(in rpm) </HTML>"), gbc);
        if (displayPosition) {
            gbc.gridy++;
            add(new JLabel("<HTML>Position : <BR>(in um)</HTML>"), gbc);
        }

        gbc.gridx++;
        gbc.gridy = 4;
        gbc.anchor = GridBagConstraints.EAST;
        add(currentjLabel, gbc);
        gbc.gridy++;
        add(velocityjLabel, gbc);
        if (displayPosition) {
            gbc.gridy++;
            add(positionjLabel, gbc);
        }
    }

    private void setDefaultValues() {
        controllerSwitch.setColor(Color.gray);
        eposStateLabel.setText(UNKNOWN_STATE);
        modeLabel.setText(UNKNOWN);
        positionjLabel.setText("0");
        currentjLabel.setText("0");
        velocityjLabel.setText("0");
    }

    /**
     * Reset Panel to default values when subsystem is shutdown.
     */
    public void resetPanel() {
        setDefaultValues();
    }

    @Override
    public String toString() {
        return "Controller State View";
    }

    /**
     * Just to test the Panel.
     *
     * @param argv
     */
    public static void main(String[] argv) {

        ControllerStatePanel d = new ControllerStatePanel("controller", true);
        JFrame frame = new JFrame("Test");
        frame.setContentPane(d);
        frame.pack();
        frame.setVisible(true);
        frame.setDefaultCloseOperation(JFrame.EXIT_ON_CLOSE);

    }

}
