package org.lsst.ccs.subsystems.fcs.carousel.ui;

import java.io.Serializable;
import org.lsst.ccs.subsystems.fcs.ui.commons.HYTTC580Panel;
import java.util.List;
import java.util.Map;
import java.util.TreeMap;
import java.util.stream.Collectors;
import javax.swing.SwingUtilities;
import org.lsst.ccs.bus.data.ConfigurationInfo;
import org.lsst.ccs.bus.data.KeyValueData;
import org.lsst.ccs.bus.data.KeyValueDataList;
import static org.lsst.ccs.subsystems.fcs.FCSCst.CA_CLAMPXMINUSCONTROLLER_NAME;
import static org.lsst.ccs.subsystems.fcs.FCSCst.CA_CLAMPXPLUSCONTROLLER_NAME;
import static org.lsst.ccs.subsystems.fcs.FCSCst.CA_PLC_DATA_KEY;
import static org.lsst.ccs.subsystems.fcs.FCSCst.CHANGER_TCPPROXY_NAME;
import org.lsst.ccs.subsystems.fcs.StatusDataPublishedByAccelerometer;
import org.lsst.ccs.subsystems.fcs.StatusDataPublishedByCarousel;
import org.lsst.ccs.subsystems.fcs.StatusDataPublishedByCarouselBrakes;
import org.lsst.ccs.subsystems.fcs.StatusDataPublishedByCarouselClamp;
import org.lsst.ccs.subsystems.fcs.StatusDataPublishedByCarouselSocket;
import org.lsst.ccs.subsystems.fcs.StatusDataPublishedByEPOSController;
import org.lsst.ccs.subsystems.fcs.StatusDataPublishedByHYTTC580;
import org.lsst.ccs.subsystems.fcs.StatusDataPublishedByHardware;
import org.lsst.ccs.subsystems.fcs.ui.commons.CanOpenDevicePanel;
import org.lsst.ccs.subsystems.fcs.ui.commons.EPOSControllerPanel;
import org.lsst.ccs.subsystems.fcs.ui.commons.GatherPanel;
import org.lsst.ccs.subsystems.fcs.ui.ControllersViewPanel;
import org.lsst.ccs.subsystems.fcs.ui.commons.InterfaceGeneralGUI;

import static org.lsst.ccs.subsystems.fcs.ui.commons.Tools.getShortComponentName;

/**
 * This panel gathers all the panels needed for the carousel GUI. Except the
 * subsystemPanel which is handled by the GeneralPanel.
 *
 * It takes care of initialization and update of the panels.
 *
 * @author virieux
 */
public class CarouselGatherPanel extends GatherPanel {

    private static final long serialVersionUID = -975949078768889254L;

    private final String[] socketNames = {"carousel/socket1", "carousel/socket2", "carousel/socket3", "carousel/socket4", "carousel/socket5"};
    private final CarouselGeneralViewPanel caGeneralViewPanel = new CarouselGeneralViewPanel();
    private ControllersViewPanel controllersViewPanel;
    protected transient final Map<String, CarouselSocketPanel> socketPanels = new TreeMap<>();

    protected InterfaceCarouselGUI subs;

    //CANopen devices panels
    private EPOSControllerPanel clampXminusControllerPanel;
    private EPOSControllerPanel clampXplusControllerPanel;
    private EPOSControllerPanel carouselControllerPanel;
    private HYTTC580Panel hyTTCPanel;
    private CanOpenDevicePanel cancbxai814Panel;
    private CanOpenDevicePanel cancbxpt100Panel;
    private PLCCarouselPanel plcCarouselPanel;

    /**
     * Initialization of the GUI for data stored in configuration. *
     */
    private class GuiInitialization implements Runnable {

        private final ConfigurationInfo configInfo;

        /**
         * Create a new GuiInitialization from a ConfigurationInfo.
         *
         * @param configInfo
         */
        public GuiInitialization(ConfigurationInfo configInfo) {
            this.configInfo = configInfo;
        }

        @Override
        public void run() {
            /**
             * ClampXminus Controller Panel *
             */
            clampXminusControllerPanel.initializeGui(configInfo);

            /**
             * ClampXplus Controller Panel *
             */
            clampXplusControllerPanel.initializeGui(configInfo);

            /**
             * Carousel Controller Panel *
             */
            carouselControllerPanel.initializeGui(configInfo);

            /**
             * CANOpen HY-TTC Panel*
             */
            hyTTCPanel.initializeGui(configInfo);

            /**
             * CANOpen CAN-CBX-AI814 Panel*
             */
            cancbxai814Panel.initializeGui(configInfo);

            /**
             * CANOpen CAN-CBX-PT100 Panel*
             */
            cancbxpt100Panel.initializeGui(configInfo);

            //** Carousel General View Panel **/
            caGeneralViewPanel.initializeGui(configInfo);

            /**
             * Carousel Sockets Panel *
             */
            socketPanels.entrySet().stream().forEach(entry -> {
                entry.getValue().initializeGui(configInfo);
            });

        }
    }

    /**
     * Build a new CarouselGatherPanel with a tcpProxy name and initialize
     * components.
     *
     * @param tcpProxyName
     */
    public CarouselGatherPanel(String tcpProxyName) {
        super(tcpProxyName);
        initPanels();
    }

    /**
     * Build a new CarouselGatherPanel with a tcpProxy name and a CarouselGUI.
     * It initializes components.
     *
     * @param subs
     * @param tcpProxyName
     */
    public CarouselGatherPanel(InterfaceCarouselGUI subs, String tcpProxyName) {
        super(tcpProxyName);
        initPanels();
        this.subs = subs;
        this.clampXminusControllerPanel.setSubsystem(subs);
        this.clampXplusControllerPanel.setSubsystem(subs);
        this.carouselControllerPanel.setSubsystem(subs);
        this.caGeneralViewPanel.setSubsystem(subs);
        this.controllersViewPanel = new ControllersViewPanel(subs);
    }

    public EPOSControllerPanel getClampXminusControllerPanel() {
        return clampXminusControllerPanel;
    }

    public EPOSControllerPanel getClampXplusControllerPanel() {
        return clampXplusControllerPanel;
    }

    public EPOSControllerPanel getCarouselControllerPanel() {
        return carouselControllerPanel;
    }

    public HYTTC580Panel getHyTTCPanel() {
        return hyTTCPanel;
    }

    public CanOpenDevicePanel getCancbxAI814Panel() {
        return cancbxai814Panel;
    }

    public CanOpenDevicePanel getCancbxPT100Panel() {
        return cancbxpt100Panel;
    }

    public CarouselGeneralViewPanel getCaGeneralViewPanel() {
        return caGeneralViewPanel;
    }
    public ControllersViewPanel getControllersViewPanel() {
        return controllersViewPanel;
    }

    public Map<String, CarouselSocketPanel> getSocketPanels() {
        return socketPanels;
    }

    public PLCCarouselPanel getPLCCarouselPanel() {
        return plcCarouselPanel;
    }

    @Override
    public InterfaceGeneralGUI getSubs() {
        return subs;
    }

    @Override
    public void updateFromStatusData(KeyValueData data) {

        if (CHANGER_TCPPROXY_NAME.equals(data.getKey())) {
            updateCanOpenHardwarePanel((int)data.getValue());

        } else if (data.getValue() instanceof StatusDataPublishedByEPOSController) {
            updateControllerPanel(data.getKey(), (StatusDataPublishedByEPOSController) data.getValue());
            updateCanOpenHardwarePanel(data.getKey(), (StatusDataPublishedByHardware) data.getValue());
            controllersViewPanel.updateFromStatusData(data);

        } else if (data.getValue() instanceof StatusDataPublishedByHYTTC580) {
            updateHYTTC580Panel(getShortComponentName(data.getKey()), (StatusDataPublishedByHYTTC580) data.getValue());
            updateCanOpenHardwarePanel(data.getKey(), (StatusDataPublishedByHardware) data.getValue());

        } else if (data.getValue() instanceof StatusDataPublishedByAccelerometer) {
            caGeneralViewPanel.update((StatusDataPublishedByAccelerometer) data.getValue());
            updateCanOpenHardwarePanel(data.getKey(), (StatusDataPublishedByHardware) data.getValue());

        } else if (data.getValue() instanceof StatusDataPublishedByHardware) {
            updateCanOpenHardwarePanel(data.getKey(), (StatusDataPublishedByHardware) data.getValue());

        } else if (data.getValue() instanceof StatusDataPublishedByCarouselClamp) {
            updateSocketPanels(data.getKey(), (StatusDataPublishedByCarouselClamp) data.getValue());

        } else if (data.getValue() instanceof StatusDataPublishedByCarouselSocket) {
            updateSocketPanels(data.getKey(), (StatusDataPublishedByCarouselSocket) data.getValue());
            caGeneralViewPanel.update((StatusDataPublishedByCarouselSocket) data.getValue());

        } else if (data.getValue() instanceof StatusDataPublishedByCarousel) {
            FCSLOG.finer("STATUS data coming from carousel.");
            caGeneralViewPanel.update((StatusDataPublishedByCarousel) data.getValue());
            updateSocketPanels((StatusDataPublishedByCarousel) data.getValue());

        } else if (data.getValue() instanceof StatusDataPublishedByCarouselBrakes) {
            FCSLOG.finer("STATUS data coming from carousel brakes.");
            caGeneralViewPanel.update((StatusDataPublishedByCarouselBrakes) data.getValue());

        } else if (data.getValue() instanceof KeyValueDataList
                && CA_PLC_DATA_KEY.equals(data.getKey())) {
            Map<String, Serializable> sensorMap
                    = ((KeyValueDataList) data.getValue()).getListOfKeyValueData().stream().collect(Collectors.toMap(KeyValueData::getKey, KeyValueData::getValue));
            plcCarouselPanel.updateFromSensorList(sensorMap);
            caGeneralViewPanel.updateFromSensorList(sensorMap);
        }
    }

    @Override
    public void updateFromConfigurationInfo(ConfigurationInfo configInfo) {
        SwingUtilities.invokeLater(new GuiInitialization(configInfo));
    }

    @Override
    public void initializeGUIWithConfigInfo(ConfigurationInfo configInfo,
            List<String> hardwareNames) {
        super.initializeGUIWithConfigInfo(configInfo, hardwareNames);
        controllersViewPanel.enableOnlineSubsystems(true, false, false);
        SwingUtilities.invokeLater(new GuiInitialization(configInfo));
    }

    /**
     * updates hardware information (fields booted, initialized) from StatusDataPublishedByHardware object.
     * @param deviceToUpdate
     * @param status
     */
    @Override
    public void updateCanOpenHardwarePanel(String deviceToUpdate, StatusDataPublishedByHardware status) {
        super.updateCanOpenHardwarePanel(getShortComponentName(deviceToUpdate), status);
        cancbxai814Panel.updateCanOpenDevice(getShortComponentName(deviceToUpdate), (StatusDataPublishedByHardware) status);
        cancbxpt100Panel.updateCanOpenDevice(getShortComponentName(deviceToUpdate), (StatusDataPublishedByHardware) status);
    }

    public void updateHYTTC580Panel(String deviceToUpdate, StatusDataPublishedByHYTTC580 status) {
        hyTTCPanel.updateDevice(deviceToUpdate, status);
    }

    /**
     * Update the controller Panels from data coming from STATUS bus.
     *
     * @param controllerLongName
     * @param s
     */
    protected void updateControllerPanel(String controllerLongName, StatusDataPublishedByEPOSController s) {
        String controllerName = getShortComponentName(controllerLongName);
        switch (controllerName) {
            case CA_CLAMPXMINUSCONTROLLER_NAME:
                clampXminusControllerPanel.updateController(CA_CLAMPXMINUSCONTROLLER_NAME, s);
                break;

            case CA_CLAMPXPLUSCONTROLLER_NAME:
                clampXplusControllerPanel.updateController(CA_CLAMPXPLUSCONTROLLER_NAME, s);
                break;

            case "carouselController":
                carouselControllerPanel.updateController("carouselController", s);
                caGeneralViewPanel.update(s);
                break;
        }
        if (CA_CLAMPXMINUSCONTROLLER_NAME.equals(controllerName) || CA_CLAMPXPLUSCONTROLLER_NAME.equals(controllerName)) {
            socketPanels.entrySet().stream().forEach((entry) -> {
                entry.getValue().updateController(controllerName, s);
            });
        }
    }

    /**
     * Update the Carousel Socket Panels from the data published on the Status
     * bus by a clamp.
     *
     * @param s
     */
    private void updateSocketPanels(String clampLongName, StatusDataPublishedByCarouselClamp s) {
        String clampName = getShortComponentName(clampLongName);
        /**
         * Carousel Sockets Panel *
         */
        /*To find the name of the socket needed to find the socket panel to be updated.*/
        char num = clampName.charAt(clampName.length() - 1);
        String socketName = "socket" + num;

        socketPanels.get(socketName).update(clampName, s);
    }

    /**
     * Update the Carousel Socket Panels from the data published on the Status
     * bus by carousel.
     *
     * @param s
     */
    private void updateSocketPanels(StatusDataPublishedByCarousel s) {
        /**
         * Carousel Sockets Panel *
         */
        socketPanels.entrySet().stream().forEach((entry) -> {
            entry.getValue().updateSocketFromCarousel(s);
        });
    }

    /**
     * Update the Carousel Socket Panels from the data published on the Status
     * bus by a Carousel Socket.
     *
     * @param s
     */
    private void updateSocketPanels(String socketLongName, StatusDataPublishedByCarouselSocket s) {
        socketPanels.get(getShortComponentName(socketLongName)).updateSocket(s);
    }

    private void initPanels() {
        setName("CarouselGatherPanel");

        //controllers Panels
        clampXminusControllerPanel = new EPOSControllerPanel(getCanOpenHardwareListPanel().getTcpProxyName()+"/clampXminusController");
        clampXplusControllerPanel = new EPOSControllerPanel(getCanOpenHardwareListPanel().getTcpProxyName()+"/clampXplusController");
        carouselControllerPanel = new EPOSControllerPanel(getCanOpenHardwareListPanel().getTcpProxyName()+"/carouselController");

        //Clamps Sensors Reader Device
        hyTTCPanel = new HYTTC580Panel(getCanOpenHardwareListPanel().getTcpProxyName()+"/hyttc580");

        //CANOpen devices to read brakes status and temperatures
        cancbxai814Panel = new CanOpenDevicePanel(getCanOpenHardwareListPanel().getTcpProxyName()+"/ai814");
        cancbxpt100Panel = new CanOpenDevicePanel(getCanOpenHardwareListPanel().getTcpProxyName()+"/pt100");

        //Sockets Panels
        for (String socketName : socketNames) {
            int ind = socketName.lastIndexOf("/");
            String tmpSocketName = ind >= 0 ? socketName.substring(ind + 1) : socketName;
            FCSLOG.info("carouselGatherPanel adding tmpSocketName");
            socketPanels.put(tmpSocketName, new CarouselSocketPanel(socketName));
        }

        //PLC Carousel debug
        plcCarouselPanel = new PLCCarouselPanel();
    }


    /**
     * Reset Panel with initial text, color, reset hardware table and filter
     * table.
     */
    @Override
    public void resetPanel() {
        super.resetPanel();
        socketPanels.entrySet().stream().forEach((entry) -> {
            entry.getValue().resetPanel();
        });
        caGeneralViewPanel.resetPanel();
        clampXminusControllerPanel.resetPanel();
        clampXplusControllerPanel.resetPanel();
        carouselControllerPanel.resetPanel();
        hyTTCPanel.resetPanel();
        plcCarouselPanel.resetPanel();
    }

}
