package org.lsst.ccs.subsystems.fcs.ui;

import org.lsst.ccs.bus.data.KeyValueData;
import org.lsst.ccs.subsystems.fcs.StatusDataPublishedByEPOSController;
import org.lsst.ccs.subsystems.fcs.ui.commons.SingleControllerPanel;
import org.lsst.ccs.subsystems.fcs.ui.commons.Tools;
import org.lsst.ccs.subsystems.fcs.ui.commons.InterfaceGeneralGUI;
import javax.swing.JButton;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.JTextArea;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.GridLayout;
import java.util.List;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.logging.Logger;

import static org.lsst.ccs.subsystems.fcs.FCSCst.*;

public class ControllersViewPanel extends JPanel {
    private static final long serialVersionUID = 1L;
    private static final Logger FCSLOG = Logger.getLogger(ControllersViewPanel.class.getName());
    private InterfaceGeneralGUI subs;
    private final String title = "Controllers Overview";
    private final List<String> errorList = new CopyOnWriteArrayList<>();
    private JButton refreshAllButton;
    private JButton checkAllButton;
    private JLabel onlineFCSSubsystem;
    private SingleControllerPanel acTruckXMinusPanel;
    private SingleControllerPanel acTruckXPlusPanel;
    private SingleControllerPanel acLatchXMinusPanel;
    private SingleControllerPanel acLatchXPlusPanel;
    private SingleControllerPanel acOnlineClampXMinusPanel;
    private SingleControllerPanel acOnlineClampXPlusPanel;
    private SingleControllerPanel acOnlineClampYMinusPanel;
    private SingleControllerPanel carouselPanel;
    private SingleControllerPanel carouselClampXMinusPanel;
    private SingleControllerPanel carouselClampXPlusPanel;
    private SingleControllerPanel loaderHooksPanel;
    private SingleControllerPanel loaderCarrierPanel;
    private JTextArea errorInformation;
    private boolean hasAutochanger = false;
    private boolean hasCarousel = false;
    private boolean hasLoader = false;

    public ControllersViewPanel(InterfaceGeneralGUI subs) {
        this.subs = subs;
        initComponents();
    }

    private void refreshAllActionPerformed() {
        subs.sendCommand("publishData", 1000); // performs also "publishData" on bridge
    }

    private void checkAllActionPerformed() {
        subs.sendCommand("checkFaultForAllControllers", 1000);
    }

    @Override
    public String toString() {
        return title;
    }

    public void enableOnlineSubsystems(boolean hasCarousel, boolean hasAutochanger, boolean hasLoader) {
        this.hasAutochanger = hasAutochanger;
        this.hasCarousel = hasCarousel;
        this.hasLoader = hasLoader;
        onlineFCSSubsystem.setText("Online Subsystems seen at startup: " + (hasAutochanger ? "Autochanger ":"") + (hasCarousel ? " Carousel ":"") + (hasLoader ? " Loader":""));
        enableCarouselControllersPanelState(hasCarousel);
        enableAutochangerControllersPanelState(hasAutochanger);
        enableLoaderControllersPanelState(hasLoader);
    }

    public void enableCarouselControllersPanelState(boolean b) {
        carouselPanel.setEnabled(b);
        carouselClampXMinusPanel.setEnabled(b);
        carouselClampXPlusPanel.setEnabled(b);
    }

    public void enableAutochangerControllersPanelState(boolean b) {
        acTruckXMinusPanel.setEnabled(b);
        acTruckXPlusPanel.setEnabled(b);
        acLatchXMinusPanel.setEnabled(b);
        acLatchXPlusPanel.setEnabled(b);
        acOnlineClampXMinusPanel.setEnabled(b);
        acOnlineClampXPlusPanel.setEnabled(b);
        acOnlineClampYMinusPanel.setEnabled(b);
    }

    public void enableLoaderControllersPanelState(boolean b) {
        loaderHooksPanel.setEnabled(b);
        loaderCarrierPanel.setEnabled(b);
    }

    private void initComponents() {
        // build all individual controller panels
        acTruckXMinusPanel = new SingleControllerPanel("Truck XMinus", "Autochanger", CHANGER_TCPPROXY_NAME + "/" + AC_TRUCKXMINUSCONTROLLER_NAME, subs, this, true);
        acTruckXPlusPanel = new SingleControllerPanel("Truck XPlus", "Autochanger", CHANGER_TCPPROXY_NAME + "/" + AC_TRUCKXPLUSCONTROLLER_NAME, subs, this, true);
        acLatchXMinusPanel = new SingleControllerPanel("Latch Xminus", "Autochanger", CHANGER_TCPPROXY_NAME + "/" + AC_LATCHXMINUSCONTROLLER_NAME, subs, this, false);
        acLatchXPlusPanel = new SingleControllerPanel("Latch Xplus", "Autochanger", CHANGER_TCPPROXY_NAME + "/" + AC_LATCHXPLUSCONTROLLER_NAME, subs, this, false);
        acOnlineClampXMinusPanel = new SingleControllerPanel("Online Clamp Xminus", "Autochanger", CHANGER_TCPPROXY_NAME + "/" + AC_CLAMPXMINUSCONTROLLER_NAME, subs, this, true);
        acOnlineClampXPlusPanel = new SingleControllerPanel("Online Clamp Xplus", "Autochanger", CHANGER_TCPPROXY_NAME + "/" + AC_CLAMPXPLUSCONTROLLER_NAME, subs, this, true);
        acOnlineClampYMinusPanel = new SingleControllerPanel("Online Clamp Yminus", "Autochanger", CHANGER_TCPPROXY_NAME + "/" + AC_CLAMPYMINUSCONTROLLER_NAME, subs, this, true);
        carouselPanel = new SingleControllerPanel("Carousel", "Carousel", CHANGER_TCPPROXY_NAME + "/" + CA_ROTATIONCONTROLLER_NAME, subs, this, false);
        carouselClampXMinusPanel = new SingleControllerPanel("Clamp Xminus", "Carousel", CHANGER_TCPPROXY_NAME + "/" + CA_CLAMPXMINUSCONTROLLER_NAME, subs, this, false);
        carouselClampXPlusPanel = new SingleControllerPanel("Clamp Xplus", "Carousel", CHANGER_TCPPROXY_NAME + "/" + CA_CLAMPXPLUSCONTROLLER_NAME, subs, this, false);
        loaderHooksPanel = new SingleControllerPanel("Hooks", "Loader", LOADER_NAME + "/" + HOOKS_CONTROLLER_NAME, subs, this, false);
        loaderCarrierPanel = new SingleControllerPanel("Carrier", "Loader", LOADER_NAME + "/" + CARRIER_CONTROLLER_NAME, subs, this, false);

        // add panels
        JPanel contentPane = new JPanel(new GridLayout(3, 4, 5, 5));
        contentPane.add(carouselPanel);
        contentPane.add(acOnlineClampXMinusPanel);
        contentPane.add(acOnlineClampXPlusPanel);
        contentPane.add(acOnlineClampYMinusPanel);
        contentPane.add(carouselClampXMinusPanel);
        contentPane.add(acTruckXMinusPanel);
        contentPane.add(acTruckXPlusPanel);
        contentPane.add(loaderHooksPanel);
        contentPane.add(carouselClampXPlusPanel);
        contentPane.add(acLatchXMinusPanel);
        contentPane.add(acLatchXPlusPanel);
        contentPane.add(loaderCarrierPanel);

        // build buttons panel
        JPanel commandPane = new JPanel();
        commandPane.add(refreshAllButton = new JButton("Refresh all"));
        commandPane.add(checkAllButton = new JButton("Check Fault for all"));
        refreshAllButton.addActionListener((java.awt.event.ActionEvent evt) -> refreshAllActionPerformed());
        checkAllButton.addActionListener((java.awt.event.ActionEvent evt) -> checkAllActionPerformed());
        refreshAllButton.setToolTipText("Command: publishData");
        checkAllButton.setToolTipText("Command: checkFault");
        commandPane.add(onlineFCSSubsystem = new JLabel("Online Subsystems: " + (hasAutochanger ? "Autochanger,":"") + (hasCarousel ? " Carousel,":"") + (hasLoader ? " Loader":"")));
        commandPane.add(contentPane);

        // build error output panel
        JPanel errorPane = new JPanel();
        errorInformation = new JTextArea(4, 60);
        errorInformation.setText("");
        errorInformation.setLineWrap(true);
        errorInformation.setWrapStyleWord(true);
        errorInformation.setEnabled(false);
        JScrollPane scrollPane = new JScrollPane(errorInformation, JScrollPane.VERTICAL_SCROLLBAR_AS_NEEDED, JScrollPane.HORIZONTAL_SCROLLBAR_NEVER);
        GridBagConstraints gbc = new GridBagConstraints();
        gbc.gridy = 0;
        errorPane.add(new JLabel("Current errors:"), gbc);
        gbc.fill = GridBagConstraints.HORIZONTAL;
        gbc.weightx = 1.0; // allow horizontal expansion
        errorPane.add(scrollPane, gbc);

        // compose main panel
        setBorder(Tools.getGeneralPanelTitle(title));
        setLayout(new GridBagLayout());
        gbc = new GridBagConstraints();
        gbc.gridx = 0;
        gbc.gridy = 0;
        gbc.anchor = GridBagConstraints.WEST;
        add(commandPane, gbc);
        gbc.gridy++;
        add(contentPane, gbc);
        gbc.gridy++;
        gbc.fill = GridBagConstraints.HORIZONTAL;
        gbc.weightx = 1.0;
        add(errorPane, gbc);
    }

    public void updateFromStatusData(KeyValueData data) {
        switch (Tools.getShortComponentName(data.getKey())) {
            case AC_TRUCKXMINUSCONTROLLER_NAME:
                acTruckXMinusPanel.updateController((StatusDataPublishedByEPOSController) data.getValue());
                break;
            case AC_TRUCKXPLUSCONTROLLER_NAME:
                acTruckXPlusPanel.updateController((StatusDataPublishedByEPOSController) data.getValue());
                break;
            case AC_LATCHXMINUSCONTROLLER_NAME:
                acLatchXMinusPanel.updateController((StatusDataPublishedByEPOSController) data.getValue());
                break;
            case AC_LATCHXPLUSCONTROLLER_NAME:
                acLatchXPlusPanel.updateController((StatusDataPublishedByEPOSController) data.getValue());
                break;
            case AC_CLAMPXMINUSCONTROLLER_NAME:
                acOnlineClampXMinusPanel.updateController((StatusDataPublishedByEPOSController) data.getValue());
                break;
            case AC_CLAMPXPLUSCONTROLLER_NAME:
                acOnlineClampXPlusPanel.updateController((StatusDataPublishedByEPOSController) data.getValue());
                break;
            case AC_CLAMPYMINUSCONTROLLER_NAME:
                acOnlineClampYMinusPanel.updateController((StatusDataPublishedByEPOSController) data.getValue());
                break;
            case CA_ROTATIONCONTROLLER_NAME:
                carouselPanel.updateController((StatusDataPublishedByEPOSController) data.getValue());
                break;
            case HOOKS_CONTROLLER_NAME:
                loaderHooksPanel.updateController((StatusDataPublishedByEPOSController) data.getValue());
                break;
            case CA_CLAMPXMINUSCONTROLLER_NAME:
                carouselClampXMinusPanel.updateController((StatusDataPublishedByEPOSController) data.getValue());
                break;
            case CA_CLAMPXPLUSCONTROLLER_NAME:
                carouselClampXPlusPanel.updateController((StatusDataPublishedByEPOSController) data.getValue());
                break;
            case CARRIER_CONTROLLER_NAME:
                loaderCarrierPanel.updateController((StatusDataPublishedByEPOSController) data.getValue());
                break;
            default:
                FCSLOG.warning(String.format("\t[ControllerView] %s not recognized!", Tools.getShortComponentName(data.getKey())));
        }
    }

    public void updateErrorListing(String controllerName, int errorCode, String errorName) {
        // no error since last clear
        String message = String.format("%-40s%-10s%s %n", controllerName, "0x" + Integer.toHexString(errorCode), errorName);
        errorList.add(message);
        errorInformation.append(message);

        FCSLOG.fine("[ControllersView] -- call to updateErrorListing from " + controllerName);
    }

    public void clearErrorListing(String controllerName) {
        if (errorList.removeIf(s -> s.startsWith(controllerName))) {
            // rebuild content of errorInformation panel
            errorInformation.setText("");
            for (String message : errorList) {
                errorInformation.append(message);
            }
        }
    }
}
