
package org.lsst.ccs.subsystems.fcs.ui;

import javax.swing.*;
import java.awt.*;
import java.awt.event.ActionEvent;
import java.io.Serializable;
import java.util.Map;
import java.util.logging.Logger;
import java.util.stream.Collectors;
import org.lsst.ccs.bus.data.KeyValueData;
import org.lsst.ccs.bus.data.KeyValueDataList;
import static org.lsst.ccs.subsystems.fcs.FCSCst.AC_PLC_DATA_KEY;
import static org.lsst.ccs.subsystems.fcs.FCSCst.AT_STANDBY;
import static org.lsst.ccs.subsystems.fcs.FCSCst.CA_PLC_DATA_KEY;
import static org.lsst.ccs.subsystems.fcs.FCSCst.LOADER_PLC_DATA_KEY;
import static org.lsst.ccs.subsystems.fcs.FCSCst.NOT_AT_STANDBY;
import static org.lsst.ccs.subsystems.fcs.FCSCst.NO_FILTER;
import static org.lsst.ccs.subsystems.fcs.FCSCst.UNKNOWN;
import org.lsst.ccs.subsystems.fcs.FcsEnumerations;
import org.lsst.ccs.subsystems.fcs.StatusDataPublishedByAutoChangerTrucks;
import org.lsst.ccs.subsystems.fcs.StatusDataPublishedByAutochangerThreeClamps;
import org.lsst.ccs.subsystems.fcs.StatusDataPublishedByAutochangerTwoLatches;
import org.lsst.ccs.subsystems.fcs.StatusDataPublishedByCarousel;
import org.lsst.ccs.subsystems.fcs.StatusDataPublishedByEPOSController;
import org.lsst.ccs.subsystems.fcs.StatusDataPublishedByLoaderCarrier;
import org.lsst.ccs.subsystems.fcs.StatusDataPublishedByLoaderClamp;
import static org.lsst.ccs.subsystems.fcs.loader.ui.PLCLoaderPanel.FILTERPRESENCE;
import static org.lsst.ccs.subsystems.fcs.loader.ui.PLCLoaderPanel.FILTERPRESENCEB;
import org.lsst.ccs.subsystems.fcs.ui.commons.DigitalSwitch;
import org.lsst.ccs.subsystems.fcs.ui.commons.FixedSizePanel;
import org.lsst.ccs.subsystems.fcs.ui.commons.Tools;
import org.lsst.ccs.subsystems.fcs.ui.commons.VerticalTextComponent;
import static org.lsst.ccs.subsystems.fcs.ui.commons.Tools.BIG_FONT;
import static org.lsst.ccs.subsystems.fcs.ui.commons.Tools.TIRETS;
import static org.lsst.ccs.subsystems.fcs.ui.commons.Tools.UNKNOWN_STATE;
import static org.lsst.ccs.subsystems.fcs.ui.commons.Tools.ZERO_VALUE;
import static org.lsst.ccs.subsystems.fcs.ui.commons.Tools.greenColor;
import static org.lsst.ccs.subsystems.fcs.ui.commons.Tools.insets_std;
import static org.lsst.ccs.subsystems.fcs.ui.commons.Tools.NONE;

/**
 *
 * @author virieux
 */
public class FcsGeneralViewPanel extends JPanel {
    private static final long serialVersionUID = 1L;
    private static final Logger FCSLOG = Logger.getLogger(FcsGeneralViewPanel.class.getName());

    private InterfaceFcsGUI subs;

    private final String title = "FCS Overview";


    private JPanel autochangerPanel;
    private JPanel carouselPanel;
    private JPanel loaderPanel;
    private JPanel loaderControlPanel;
    private JPanel controlPanel;
    private JPanel subCommandsPanel;
    private JPanel rotatePanel;
    private JPanel setFilterPanel;

    /*autochanger information*/
    private final JLabel filterONLINELabel = new JLabel(NO_FILTER);
    private final JLabel filterOnTrucksLabel = new JLabel(NO_FILTER);
    private final JLabel onlineClampsStateLabel = new JLabel("UNKNOWN");
    private final JLabel latchesStateLabel = new JLabel("UNKNOWN");
    private final JLabel trucksPositionLabel = new JLabel("UNKNOWN");
    private final JLabel trucksPositionNameLabel = new JLabel("UNKNOWN");

    /* carousel information */
    private final JLabel actualPositionLabel = new JLabel();
    private final JLabel actualSpeedLabel = new JLabel();
    private final JLabel actualCurrentLabel = new JLabel();
    private final JLabel standbyLabel = new JLabel();
    private final DigitalSwitch standbySwitch = new DigitalSwitch();
    private final JLabel socketAtStandbyLabel = new JLabel();
    private final JLabel ioModuleStatusLabel = new JLabel();
    private final JLabel filterAtStandbyLabel = new JLabel();
    private final JLabel caClampsStateLabel = new JLabel();
    private final JLabel deltaPositionLabel = new JLabel();

    /* loader information */
    private final JLabel loaderClampHomingLabel = new JLabel();
    private final JLabel filterOnLoaderLabel = new JLabel();
    private final JLabel loaderClampStateLabel = new JLabel();
    private final JLabel loaderClampPositionLabel = new JLabel();
    private final JLabel loaderCarrierPositionLabel = new JLabel();
    private final JLabel loaderCarrierPositionNameLabel = new JLabel();

    /* status information */
    private final FcsCameraProtectionPanel cameraStatusPanel = new FcsCameraProtectionPanel();

    /* control panel */
    private final JButton setFilterButton = new JButton("Set Filter");
    private final JButton setNoFilterButton = new JButton("Set No Filter");
    private final JButton loadFilterButton = new JButton("Load filter into camera");
    private final JButton unloadFilterButton = new JButton("Unload filter from camera");
    private final JButton connectLoaderButton = new JButton("Connect loader");
    private final JButton disconnectLoaderButton = new JButton("Disconnect loader");
    private final JTextField filterField = new JTextField(2);

    private final JToggleButton sleepButton = new JToggleButton("Sleep");
    private final Color sleepButtonColor = new Color(255, 80, 80);
    private final JToggleButton wakeUpButton = new JToggleButton("Wake up");
    private final Color wakeUpButtonColor = new Color(80, 255, 80);
    private final JToggleButton stayUpButton = new JToggleButton("Stay up");
    private final Color stayUpButtonColor = new Color(80, 80, 255);

    /* set filter sub commands panel */
    private final JButton setFilterAtHandoffForLoaderButton = new JButton("Set Filter for loader");
    private final JButton setNoFilterAtHandoffForLoaderButton = new JButton("Set No Filter for loader");
    private final JButton storeFilterOnCarouselButton = new JButton("Store filter on Carousel");
    private final JButton grabFilterAtStandbyButton = new JButton("Grab filter at Standby");
    private final JButton disengageFilterFromCarouselButton = new JButton("Disengage filter from carousel");
    private final JButton moveAndClampFilterOnlineButton = new JButton("Move and clamp filter Online");
    private final JButton unclampAndMoveFilterToHandoffButton = new JButton("Unclamp and move filter to Handoff");

    /* Panel for command rotateSocketToStandbyButton*/
    private final JButton rotateSocketToStandbyButton = new JButton("Rotate socket to standby");
    private final String[] socketList = {
        "        ", "Socket 1", "Socket 2", "Socket 3", "Socket 4", "Socket 5"};
    private final JComboBox<String> socketMenu = new JComboBox<>(socketList);
    private final JLabel socketLabel = new JLabel("Choose destination socket: ");
    // TODO use the available filters to create a drop down menu
    // private final String[] filterList = {"   ", "u", "g", "r", "i", "z", "y"};
    // private final JComboBox<String> filterMenu = new JComboBox<>(filterList);
    private final JLabel filterLabel = new JLabel("Selected filter ID:");
    /* socket to rotate at Standby */
    private int socketID;

    /* filter ID to set */
    private int filterID;

    public FcsGeneralViewPanel() {
        initComponents();
    }

    public void setSubs(InterfaceFcsGUI subs) {
        this.subs = subs;
    }


    void updateFromStatusData(KeyValueData data) {
        if (data.getValue() instanceof StatusDataPublishedByAutochangerThreeClamps) {
            updateClamps((StatusDataPublishedByAutochangerThreeClamps) data.getValue());
        } else if (data.getValue() instanceof StatusDataPublishedByAutochangerTwoLatches) {
            updateLatches((StatusDataPublishedByAutochangerTwoLatches) data.getValue());
        } else if (data.getValue() instanceof StatusDataPublishedByAutoChangerTrucks) {
            updateTrucks((StatusDataPublishedByAutoChangerTrucks) data.getValue());
        } else if (data.getValue() instanceof StatusDataPublishedByCarousel) {
            updateCarousel((StatusDataPublishedByCarousel) data.getValue());
        } else if (data.getValue() instanceof StatusDataPublishedByLoaderClamp) {
            updateLoaderClamp((StatusDataPublishedByLoaderClamp) data.getValue());
        } else if (data.getValue() instanceof StatusDataPublishedByLoaderCarrier) {
            updateLoaderCarrier((StatusDataPublishedByLoaderCarrier) data.getValue());
        } else if (data.getValue() instanceof KeyValueDataList
                && LOADER_PLC_DATA_KEY.equals(data.getKey())) {
            cameraStatusPanel.updateLoaderFromSensorList((KeyValueDataList) data.getValue());
            updateFromSensorList((KeyValueDataList) data.getValue());
        } else if (data.getValue() instanceof KeyValueDataList
                && AC_PLC_DATA_KEY.equals(data.getKey())) {
            cameraStatusPanel.updateAutochangerFromSensorList((KeyValueDataList) data.getValue());
        } else if (data.getValue() instanceof KeyValueDataList
                && CA_PLC_DATA_KEY.equals(data.getKey())) {
            cameraStatusPanel.updateCarouselFromSensorList((KeyValueDataList) data.getValue());
        }
    }

    void updateCarousel(StatusDataPublishedByCarousel s) {
        FCSLOG.finer(() -> getName() + " updating from carousel data:");
        SwingUtilities.invokeLater(new UpdateCarousel(s));
    }

    void updateController(StatusDataPublishedByEPOSController s) {
        FCSLOG.finer(() -> getName() + " updating from carousel controller data:");
        SwingUtilities.invokeLater(new UpdateController(s));
    }

    void updateCarrierController(StatusDataPublishedByEPOSController s) {
        FCSLOG.finer(() -> getName() + " updating from loader carrier controller data:");
        SwingUtilities.invokeLater(new UpdateCarrierController(s));
    }

    void updateClampController(StatusDataPublishedByEPOSController s) {
        FCSLOG.finer(getName() + " updating from loader clamp controller data:");
        SwingUtilities.invokeLater(new UpdateClampController(s));
    }

    void updateTrucks(StatusDataPublishedByAutoChangerTrucks s) {
        SwingUtilities.invokeLater(new UpdateTrucks(s));
    }

    void updateLatches(StatusDataPublishedByAutochangerTwoLatches s) {
        SwingUtilities.invokeLater(new UpdateLatches(s));
    }

    void updateClamps(StatusDataPublishedByAutochangerThreeClamps s) {
        SwingUtilities.invokeLater(new UpdateOnlineClamps(s));
    }

    void updateLoaderClamp(StatusDataPublishedByLoaderClamp s) {
        SwingUtilities.invokeLater(new UpdateLoaderClamp(s));
    }

    void updateLoaderCarrier(StatusDataPublishedByLoaderCarrier s) {
        SwingUtilities.invokeLater(new UpdateLoaderCarrier(s));
    }

    void updateFromSensorList(KeyValueDataList kvdl) {
        SwingUtilities.invokeLater(new UpdateFromSensorList(kvdl));
    }
    /**
     * A Runnable class to update carousel panel from an object
     * StatusDataPublishedByCarousel.
     */
    public class UpdateCarousel implements Runnable {

        private final StatusDataPublishedByCarousel s;

        public UpdateCarousel(StatusDataPublishedByCarousel s) {
            this.s = s;
        }

        @Override
        public void run() {
            actualPositionLabel.setText(Integer.toString(s.getPosition()));

            if (s.isAtStandby() && !s.isMoving()) {
                standbyLabel.setText(AT_STANDBY);
                standbySwitch.setColor(greenColor);
                socketAtStandbyLabel.setText(s.getSocketAtStandbyName());
                socketAtStandbyLabel.setFont(BIG_FONT);
                //if not filter at standby s.getFilterAtStandbyName() == NO_FILTER
                filterAtStandbyLabel.setText(s.getFilterAtStandbyObservatoryName());
                filterAtStandbyLabel.setFont(BIG_FONT);
                caClampsStateLabel.setText(s.getClampsStateAtStandby().toString());
                caClampsStateLabel.setForeground(s.getClampsStateAtStandby().getColor());
                ioModuleStatusLabel.setText(s.getIOStatusAtStandby().toString());
                ioModuleStatusLabel.setForeground(s.getIOStatusAtStandby().getColor());
                deltaPositionLabel.setText(Integer.toString(s.getDeltaPositionAtStandby()));

            } else {
                standbyLabel.setText(NOT_AT_STANDBY);
                standbySwitch.setColor(Color.orange);
                if (s.isMoving()) {
                    socketAtStandbyLabel.setText(TIRETS);
                } else {
                    //socketName could be "NO_SOCKET_AT_STANDBY" or "ERROR_READING_ID"
                    socketAtStandbyLabel.setText(s.getSocketAtStandbyName());
                }
                filterAtStandbyLabel.setText(TIRETS);
                caClampsStateLabel.setText(TIRETS);
                ioModuleStatusLabel.setText(TIRETS);
            }
        }
    }

    /**
     * A Runnable to update controller panel with data published by carousel
     * controller.
     */
    public class UpdateController implements Runnable {

        private final StatusDataPublishedByEPOSController s;

        public UpdateController(StatusDataPublishedByEPOSController s) {
            this.s = s;
        }

        @Override
        public void run() {
            if (s.isBooted() && s.getState() != null) {
                actualSpeedLabel.setText(Double.toString(s.getVelocity()));
                actualCurrentLabel.setText(Double.toString(s.getAverageCurrent()));
            }
        }
    }

    /**
     * A Runnable to update panel with data published by carrier controller.
     */
    public class UpdateCarrierController implements Runnable {

        private final StatusDataPublishedByEPOSController s;

        public UpdateCarrierController(StatusDataPublishedByEPOSController s) {
            this.s = s;
        }

        @Override
        public void run() {
            if (s.isBooted() && s.getState() != null) {
                loaderCarrierPositionLabel.setText(Integer.toString(s.getPosition()));
                loaderCarrierPositionNameLabel.setForeground(greenColor);
            }
        }
    }

    /**
     * A Runnable to update panel with data published by clamp controller.
     */
    public class UpdateClampController implements Runnable {

        private final StatusDataPublishedByEPOSController s;

        public UpdateClampController(StatusDataPublishedByEPOSController s) {
            this.s = s;
        }

        @Override
        public void run() {
            if (s.isBooted() && s.getState() != null) {
                loaderClampPositionLabel.setText(Integer.toString(s.getPosition()));
            }
        }
    }

    /**
     * A Runnable to initialize trucks data from status data published on the
     * STATUS bus by autochanger trucks.
     */
    private class UpdateTrucks implements Runnable {

        private StatusDataPublishedByAutoChangerTrucks s;

        public UpdateTrucks(StatusDataPublishedByAutoChangerTrucks status) {
            this.s = status;
        }

        @Override
        public void run() {
            trucksPositionLabel.setText(String.valueOf(s.getDriverPosition()));
            trucksPositionNameLabel.setForeground(greenColor);
            if (s.isAtHandoff()) {
                trucksPositionNameLabel.setText("HANDOFF");
            } else if (s.isAtOnline()) {
                trucksPositionNameLabel.setText("ONLINE");
                filterONLINELabel.setText(filterOnTrucksLabel.getText());
            } else if (s.isAtStandby()) {
                trucksPositionNameLabel.setText("STANDBY");
            } else {
                trucksPositionNameLabel.setText("IN_TRAVEL");
                trucksPositionNameLabel.setForeground(Color.YELLOW);
                filterONLINELabel.setText(NO_FILTER);
            }
        }

    }

    /**
     * A Runnable to update latches data from status data published on the
     * STATUS bus by autochanger latches.
     */
    class UpdateLatches implements Runnable {

        private StatusDataPublishedByAutochangerTwoLatches s;

        public UpdateLatches(StatusDataPublishedByAutochangerTwoLatches status) {
            this.s = status;
        }

        @Override
        public void run() {
            latchesStateLabel.setText(s.getLockStatus().toString());
            latchesStateLabel.setForeground(s.getLockStatus().getColor());
            filterOnTrucksLabel.setFont(BIG_FONT);
            if (s.getFilterPresenceStatus().equals(FcsEnumerations.FilterPresenceStatus.ENGAGED)) {
                filterOnTrucksLabel.setText(s.getFilterObservatoryName());
            } else {
                filterOnTrucksLabel.setText("No Filter");
            }
        }
    }

    /**
     * A Runnable to update ONLINE clamps data from status data published on the
     * STATUS bus by autochanger ONLINE clamps.
     */
    class UpdateOnlineClamps implements Runnable {

        private StatusDataPublishedByAutochangerThreeClamps s;

        public UpdateOnlineClamps(StatusDataPublishedByAutochangerThreeClamps status) {
            this.s = status;
        }

        @Override
        public void run() {
            onlineClampsStateLabel.setText(s.getLockStatus().toString());
            onlineClampsStateLabel.setForeground(s.getLockStatus().getColor());
        }
    }

    /**
     * A Runnable class to update loader panel from an object
     * StatusDataPublishedByLoaderClamp.
     */
    public class UpdateLoaderClamp implements Runnable {

        private final StatusDataPublishedByLoaderClamp s;

        public UpdateLoaderClamp(StatusDataPublishedByLoaderClamp s) {
            this.s = s;
        }

        @Override
        public void run() {
            /*Update of loader Panel*/
            if (s.isHomingDone()) {
                loaderClampHomingLabel.setText("DONE");
                loaderClampHomingLabel.setForeground(greenColor);
            } else {
                loaderClampHomingLabel.setText(TIRETS);
                loaderClampHomingLabel.setForeground(greenColor);
            }
            loaderClampStateLabel.setText(s.getClampState().name());
            loaderClampStateLabel.setForeground(s.getClampState().getColor());
        }
    }

    /**
     * A Runnable class to update loader panel from an object
     * StatusDataPublishedByLoaderCarrier.
     */
    public class UpdateLoaderCarrier implements Runnable {

        private final StatusDataPublishedByLoaderCarrier s;

        public UpdateLoaderCarrier(StatusDataPublishedByLoaderCarrier s) {
            this.s = s;
        }

        @Override
        public void run() {
            /*Update of loader Panel*/

            if (s.isAtStorage()) {
                loaderCarrierPositionNameLabel.setText("STORAGE");
            } else if (s.isAtHandoff()) {
                loaderCarrierPositionNameLabel.setText("HANDOFF");
            } else if (s.isAtEngaged()) {
                loaderCarrierPositionNameLabel.setText("ENGAGED");
            } else {
                loaderCarrierPositionNameLabel.setText("IN_TRAVEL");
                loaderCarrierPositionNameLabel.setForeground(Color.YELLOW);
            }
        }
    }


    /**
     * A Runnable class to update FcsGeneralView from a KeyValueDataList of sensors.
     */
    public class UpdateFromSensorList implements Runnable {

        private final KeyValueDataList kvdl;

        public UpdateFromSensorList(KeyValueDataList kvdl) {
            this.kvdl = kvdl;
        }

        @Override
        public void run() {
            /* sMap is the map obtained from kvdl : key= (String) sensorName, value = (int) sensorValue */
            Map<String, Serializable> sMap =
                    kvdl.getListOfKeyValueData().stream().collect(Collectors.toMap(KeyValueData::getKey, KeyValueData::getValue));

            if (1 == (int) sMap.get(FILTERPRESENCE) && 1 == (int) sMap.get(FILTERPRESENCEB)) {
                /* a filter is on loader*/
                filterOnLoaderLabel.setText("YES");
                filterOnLoaderLabel.setForeground(greenColor);
            } else {
                filterOnLoaderLabel.setText(NO_FILTER);
                filterOnLoaderLabel.setForeground(Color.gray);
            }
        }

    }

    private void resetPowerSaveButtons() {
        JToggleButton[] buttons = {sleepButton, wakeUpButton, stayUpButton};
        for (JToggleButton button: buttons) {
            button.setFocusable(true);
            button.setSelected(false);
            button.setBackground(null);
        }
    }

    private void sleepButtonPressed() {
        resetPowerSaveButtons();
        sleepButton.setSelected(true);
        sleepButton.setFocusable(false);;
        sleepButton.setBackground(sleepButtonColor);
    }

    private void wakeUpButtonPressed() {
        resetPowerSaveButtons();
        wakeUpButton.setSelected(true);
        wakeUpButton.setFocusable(false);
        wakeUpButton.setBackground(wakeUpButtonColor);
    }

    private void stayUpButtonPressed() {
        resetPowerSaveButtons();
        stayUpButton.setSelected(true);
        stayUpButton.setFocusable(false);
        stayUpButton.setBackground(stayUpButtonColor);
    }

    public void initializePowerSaveButtons(boolean powerSave, boolean allowed) {
        if ( powerSave ) {
            sleepButtonPressed();
        } else if ( allowed ) {
            wakeUpButtonPressed();
        } else {
            stayUpButtonPressed();
        }
    }

    // <editor-fold defaultstate="collapsed" desc="Form Description">
    private void initComponents() {

        /*******************************************************************************
         ** Buttons initialization
         * *****************************************************************************/
        filterField.setToolTipText("Filter IDs can be found in FCS>Carousel>Filter list");
        setFilterButton.setToolTipText("command: setFilter");
        setNoFilterButton.setToolTipText("command: setNoFilter");
        setFilterAtHandoffForLoaderButton.setToolTipText("command: setFilterAtHandoffForLoader");
        setNoFilterAtHandoffForLoaderButton.setToolTipText("command: setNoFilterAtHandoffForLoader");
        storeFilterOnCarouselButton.setToolTipText("command: storeFilterOnCarousel");
        rotateSocketToStandbyButton.setToolTipText("command: rotateSocketToStandby");
        grabFilterAtStandbyButton.setToolTipText("command: grabFilterAtStandby");
        disengageFilterFromCarouselButton.setToolTipText("command: disengageFilterFromCarousel");
        moveAndClampFilterOnlineButton.setToolTipText("command: moveAndClampFilterOnline");
        unclampAndMoveFilterToHandoffButton.setToolTipText("command: unclampAndMoveFilterToHandoff");
        connectLoaderButton.setToolTipText("command: connectLoaderCANbus");
        loadFilterButton.setToolTipText("command: loadFilter");
        unloadFilterButton.setToolTipText("command: unloadFilter");
        disconnectLoaderButton.setToolTipText("command: disconnectLoaderCANbus");
        sleepButton.setToolTipText("command: wakeFilterChanger 0");
        wakeUpButton.setToolTipText("command: wakeFilterChanger 1");
        stayUpButton.setToolTipText("command: wakeFilterChanger 2");

        setFilterButton.addActionListener((ActionEvent evt) -> {
            setFilterButtonActionPerformed(evt);
        });

        setFilterAtHandoffForLoaderButton.addActionListener((ActionEvent evt) -> {
            setFilterAtHandoffForLoaderButtonActionPerformed(evt);
        });

        setNoFilterButton.addActionListener((ActionEvent evt) -> {
            setNoFilterButtonActionPerformed(evt);
        });

        setNoFilterAtHandoffForLoaderButton.addActionListener((ActionEvent evt) -> {
            setNoFilterAtHandoffForLoaderButtonActionPerformed(evt);
        });

        loadFilterButton.addActionListener((ActionEvent evt) -> {
            loadFilterButtonActionPerformed(evt);
        });

        unloadFilterButton.addActionListener((ActionEvent evt) -> {
            unloadFilterButtonActionPerformed(evt);
        });

        connectLoaderButton.addActionListener((ActionEvent evt) -> {
            showLoaderPanels(true);
            connectLoaderButtonActionPerformed(evt);
        });

        disconnectLoaderButton.addActionListener((ActionEvent evt) -> {
            showLoaderPanels(false);
            disconnectLoaderButtonActionPerformed(evt);
        });

        storeFilterOnCarouselButton.addActionListener((ActionEvent evt) -> {
            storeFilterOnCarouselActionPerformed(evt);
        });

        rotateSocketToStandbyButton.addActionListener((ActionEvent evt) -> {
            rotateSocketToStandbyActionPerformed(evt);
        });

        grabFilterAtStandbyButton.addActionListener((ActionEvent evt) -> {
            grabFilterAtStandbyActionPerformed(evt);
        });

        disengageFilterFromCarouselButton.addActionListener((ActionEvent evt) -> {
            disengageFilterFromCarouselActionPerformed(evt);
        });

        moveAndClampFilterOnlineButton.addActionListener((ActionEvent evt) -> {
            moveAndClampFilterOnlineActionPerformed(evt);
        });

        unclampAndMoveFilterToHandoffButton.addActionListener((ActionEvent evt) -> {
            unclampAndMoveFilterToHandoffActionPerformed(evt);
        });

        sleepButton.addActionListener((ActionEvent evt) -> {
            wakeFilterChangerActionPerformed(evt, 0);
            sleepButtonPressed();
        });
        wakeUpButton.addActionListener((ActionEvent evt) -> {
            wakeFilterChangerActionPerformed(evt, 1);
            wakeUpButtonPressed();
        });
        stayUpButton.addActionListener((ActionEvent evt) -> {
            wakeFilterChangerActionPerformed(evt, 2);
            stayUpButtonPressed();
        });


        GridBagConstraints gbc1;
        /*****************************************************************************
        ** Control Panel
        *****************************************************************************/
        JPanel wakeUpPanel = new JPanel();
        wakeUpPanel.setBorder(BorderFactory.createTitledBorder("Power status"));
        wakeUpPanel.setLayout(new GridBagLayout());
        gbc1 = new GridBagConstraints();
        gbc1.insets = insets_std;
        gbc1.gridx = 0;
        gbc1.gridy = 0;
        gbc1.gridwidth = 1;
        wakeUpPanel.add(sleepButton, gbc1);
        gbc1.gridx++;
        wakeUpPanel.add(wakeUpButton, gbc1);
        gbc1.gridx++;
        wakeUpPanel.add(stayUpButton, gbc1);

        setFilterPanel = new JPanel();
        setFilterPanel.setBorder(BorderFactory.createLineBorder(Color.GRAY));
        setFilterPanel.setLayout(new GridBagLayout());
        gbc1 = new GridBagConstraints();
        gbc1.insets = insets_std;
        gbc1.gridx = 0;
        gbc1.gridy = 0;
        gbc1.anchor = GridBagConstraints.WEST;
        gbc1.fill = GridBagConstraints.HORIZONTAL;
        gbc1.gridwidth = 3;
        setFilterPanel.add(setFilterButton, gbc1);
        gbc1.gridy++;
        setFilterPanel.add(setFilterAtHandoffForLoaderButton, gbc1);
        gbc1.gridy++;
        gbc1.gridwidth = 1;
        // gbc1.fill = GridBagConstraints.HORIZONTAL;
        setFilterPanel.add(filterLabel, gbc1);
        gbc1.gridx++;
        setFilterPanel.add(filterField, gbc1);

        controlPanel = new JPanel();
        controlPanel.setBorder(BorderFactory.createTitledBorder("Filter commands (level 0)"));
        controlPanel.setLayout(new GridBagLayout());
        gbc1 = new GridBagConstraints();
        gbc1.insets = insets_std;
        gbc1.gridx = 0;
        gbc1.gridy = 0;
        gbc1.anchor = GridBagConstraints.CENTER;
        gbc1.fill = GridBagConstraints.HORIZONTAL;
        gbc1.gridwidth = 1;
        controlPanel.add(setFilterPanel, gbc1);
        gbc1.gridy++;
        controlPanel.add(setNoFilterButton, gbc1);
        gbc1.gridy++;
        controlPanel.add(setNoFilterAtHandoffForLoaderButton, gbc1);

        JPanel mainControlPanel = new JPanel();
        mainControlPanel.setLayout(new GridBagLayout());
        gbc1 = new GridBagConstraints();
        gbc1.insets = insets_std;
        gbc1.gridx = 0;
        gbc1.gridy = 0;
        gbc1.anchor = GridBagConstraints.CENTER;
        gbc1.fill = GridBagConstraints.VERTICAL;
        mainControlPanel.add(wakeUpPanel, gbc1);
        gbc1.gridy++;
        mainControlPanel.add(controlPanel, gbc1);

        /*****************************************************************************
        ** Sub-commands Control Panel
        *****************************************************************************/
        /* Panel for command rotateSocketToStandbyButton */
        rotatePanel = new JPanel();
        rotatePanel.setBorder(BorderFactory.createLineBorder(Color.GRAY));
        rotatePanel.setLayout(new GridBagLayout());
        gbc1 = new GridBagConstraints();
        gbc1.insets = insets_std;
        gbc1.gridx = 0;
        gbc1.gridy = 0;
        gbc1.anchor = GridBagConstraints.WEST;
        gbc1.fill = GridBagConstraints.HORIZONTAL;
        gbc1.gridwidth = 2;
        rotatePanel.add(rotateSocketToStandbyButton, gbc1);
        gbc1.gridy++;
        gbc1.gridwidth = 1;
        gbc1.fill = GridBagConstraints.NONE;
        rotatePanel.add(socketLabel, gbc1);
        gbc1.gridx++;
        rotatePanel.add(socketMenu, gbc1);
        /* end of Panel for command rotateSocketToStandbyButton */

        subCommandsPanel = new JPanel();
        subCommandsPanel.setBorder(BorderFactory.createTitledBorder("Filter subcommands (level 1)"));
        subCommandsPanel.setLayout(new GridBagLayout());
        gbc1 = new GridBagConstraints();
        gbc1.insets = insets_std;
        gbc1.gridx = 0;
        gbc1.gridy = 0;
        gbc1.anchor = GridBagConstraints.WEST;
        gbc1.fill = GridBagConstraints.HORIZONTAL;
        gbc1.gridwidth = 3;
        subCommandsPanel.add(storeFilterOnCarouselButton, gbc1);
        gbc1.gridy++;
        subCommandsPanel.add(rotatePanel, gbc1);
        gbc1.gridy++;
        subCommandsPanel.add(grabFilterAtStandbyButton, gbc1);
        gbc1.gridy++;
        subCommandsPanel.add(disengageFilterFromCarouselButton, gbc1);
        gbc1.gridy++;
        subCommandsPanel.add(moveAndClampFilterOnlineButton, gbc1);
        gbc1.gridy++;
        subCommandsPanel.add(unclampAndMoveFilterToHandoffButton, gbc1);

        /*****************************************************************************
         ** end of sub-commands Control Panel
         * **************************************************************************/

        /*****************************************************************************
        ** Loader commands
        *****************************************************************************/
        JPanel loaderLoadPanel = new JPanel();
        loaderLoadPanel.setBorder(BorderFactory.createLineBorder(Color.GRAY));
        loaderLoadPanel.setLayout(new GridBagLayout());
        gbc1 = new GridBagConstraints();
        gbc1.insets = insets_std;
        gbc1.gridx = 0;
        gbc1.gridy = 0;
        gbc1.anchor = GridBagConstraints.WEST;
        gbc1.fill = GridBagConstraints.HORIZONTAL;
        loaderLoadPanel.add(loadFilterButton, gbc1);
        gbc1.gridy++;
        loaderLoadPanel.add(unloadFilterButton, gbc1);

        loaderControlPanel = new JPanel();
        loaderControlPanel.setBorder(BorderFactory.createTitledBorder("Loader commands (level 0)"));
        loaderControlPanel.setLayout(new GridBagLayout());
        // authorized commands if loader is connected
        gbc1.insets = insets_std;
        gbc1.gridx = 0;
        gbc1.gridy = 0;
        gbc1.anchor = GridBagConstraints.WEST;
        gbc1.fill = GridBagConstraints.HORIZONTAL;
        loaderControlPanel.add(connectLoaderButton, gbc1);
        gbc1.gridy++;
        loaderControlPanel.add(loaderLoadPanel, gbc1);
        gbc1.gridy++;
        loaderControlPanel.add(disconnectLoaderButton, gbc1);

        /**
         ******************************************************************************
         ** Whole Panel
         * *****************************************************************************
         */
        /*General Information Panel*/
        setBorder(Tools.getGeneralPanelTitle(title));
        setLayout(new java.awt.GridBagLayout());
        GridBagConstraints gbc = new GridBagConstraints();
        gbc.insets = insets_std;
        gbc.anchor = GridBagConstraints.BASELINE_LEADING;
        gbc.fill = GridBagConstraints.BOTH;
        gbc.gridx = 0;
        gbc.gridy = 0;
        gbc.gridwidth = 3;
        add(cameraStatusPanel, gbc);

        gbc.gridy++;
        gbc.gridwidth = 3;
        JPanel statusPanel = new JPanel();
        statusPanel.setBorder(BorderFactory.createEmptyBorder());
        statusPanel.setLayout(new java.awt.GridBagLayout());
        GridBagConstraints statusGbc = new GridBagConstraints();
        statusGbc.anchor = GridBagConstraints.BASELINE_LEADING;
        statusGbc.gridx = 0;
        statusGbc.gridy = 0;
        statusGbc.insets = insets_std;
        statusPanel.add(createCarouselPanel(), statusGbc);
        statusGbc.gridx++;
        statusPanel.add(createAutochangerPanel(), statusGbc);
        statusGbc.gridx++;
        statusPanel.add(createLoaderPanel(), statusGbc);
        add(statusPanel, gbc);

        gbc.gridwidth = 1;
        gbc.gridx = 0;
        gbc.gridy++;
        add(mainControlPanel, gbc);
        gbc.gridx++;
        add(subCommandsPanel, gbc);
        gbc.gridx++;
        add(loaderControlPanel, gbc);
        /**
         ******************************************************************************
         ** end of Whole Panel
         * *****************************************************************************
         */
        setDefaultValues();

        // By default the loader is never connected to the camera when the FCS is started
        showLoaderPanels(false);
    }

    private JPanel createCarouselPanel() {
        JPanel contentPane = new JPanel();
        contentPane.setBorder(BorderFactory.createTitledBorder(BorderFactory.createLineBorder(VerticalTextComponent.getSubsystemColor("Carousel"), 2), ""));
        contentPane.setLayout(new GridBagLayout());
        GridBagConstraints gbc = new GridBagConstraints();
        gbc.insets = insets_std;
        //first column
        gbc.gridx = 0;
        gbc.gridy = 0;
        gbc.anchor = GridBagConstraints.LINE_START;
        contentPane.add(new JLabel("Position: "), gbc);
        gbc.gridy++;
        contentPane.add(new JLabel("Delta position: "), gbc);
        gbc.gridy++;
        contentPane.add(new JLabel("Speed: "), gbc);
        gbc.gridy++;
        contentPane.add(new JLabel("Average Current: "), gbc);
        gbc.gridy++;
        //contentPane.add(standbyLabel, gbc);
        contentPane.add(new FixedSizePanel(standbyLabel, Tools.getPlaceHolder("carousel"), false), gbc);
        gbc.gridy++;
        contentPane.add(new JLabel("Socket at STANDBY: "), gbc);
        gbc.gridy++;
        contentPane.add(new JLabel("Filter at STANDBY: "), gbc);
        gbc.gridy++;
        contentPane.add(new JLabel("IO status: "), gbc);
        gbc.gridy++;
        contentPane.add(new JLabel("Clamps status: "), gbc);
        //second column
        gbc.gridx++;
        gbc.gridy = 0;
        gbc.anchor = GridBagConstraints.LINE_END;
        contentPane.add(actualPositionLabel, gbc);
        gbc.gridy++;
        contentPane.add(deltaPositionLabel, gbc);
        gbc.gridy++;
        contentPane.add(actualSpeedLabel, gbc);
        gbc.gridy++;
        contentPane.add(actualCurrentLabel, gbc);
        gbc.gridy++;
        standbySwitch.setColor(Color.gray);
        contentPane.add(standbySwitch, gbc);
        gbc.gridy++;
        contentPane.add(socketAtStandbyLabel, gbc);
        gbc.gridy++;
        contentPane.add(filterAtStandbyLabel, gbc);
        gbc.gridy++;
        contentPane.add(ioModuleStatusLabel, gbc);
        gbc.gridy++;
        contentPane.add(caClampsStateLabel, gbc);

        carouselPanel = new JPanel();
        carouselPanel.setLayout(new BoxLayout(carouselPanel, BoxLayout.X_AXIS));
        carouselPanel.add(new VerticalTextComponent("Carousel"));
        carouselPanel.add(contentPane);

        JPanel fixedCarouselPanel = new JPanel();
        fixedCarouselPanel.setLayout(new CardLayout());
        fixedCarouselPanel.add(carouselPanel);
        fixedCarouselPanel.add(Tools.getPanelPlaceHolder("carousel_pane"));

        return fixedCarouselPanel;
    }

    private JPanel createLoaderPanel() {
        JPanel contentPane = new JPanel();
        contentPane.setBorder(BorderFactory.createTitledBorder(BorderFactory.createLineBorder(VerticalTextComponent.getSubsystemColor("Loader"), 2), ""));
        contentPane.setLayout(new GridBagLayout());
        GridBagConstraints gbc = new GridBagConstraints();
        gbc.insets = insets_std;
        //first column
        gbc.gridx = 0;
        gbc.gridy = 0;
        gbc.anchor = GridBagConstraints.LINE_START;
        contentPane.add(new JLabel("Clamp homing: "), gbc);
        gbc.gridy++;
        contentPane.add(new JLabel("Filter on loader: "), gbc);
        gbc.gridy++;
        contentPane.add(new JLabel("Clamp state: "), gbc);
        gbc.gridy++;
        contentPane.add(new JLabel("Clamp position: "), gbc);
        gbc.gridy++;
        contentPane.add(new JLabel("Carrier state: "), gbc);
        gbc.gridy++;
        contentPane.add(new JLabel("Carrier position: "), gbc);
        for (int i = 0; i<2; i++) {
            gbc.gridy++;
            contentPane.add(new FixedSizePanel(Tools.getPlaceHolder("loader")), gbc);
        }
        gbc.gridy++;
        contentPane.add(new FixedSizePanel(new DigitalSwitch("fake", "fake")), gbc);

        //second column
        gbc.gridx++;
        gbc.gridy = 0;
        gbc.anchor = GridBagConstraints.LINE_END;
        contentPane.add(loaderClampHomingLabel, gbc);
        gbc.gridy++;
        contentPane.add(filterOnLoaderLabel, gbc);
        gbc.gridy++;
        contentPane.add(loaderClampStateLabel, gbc);
        gbc.gridy++;
        contentPane.add(loaderClampPositionLabel, gbc);
        gbc.gridy++;
        contentPane.add(loaderCarrierPositionNameLabel, gbc);
        gbc.gridy++;
        contentPane.add(loaderCarrierPositionLabel, gbc);
        gbc.gridy++;
        contentPane.add(new FixedSizePanel(Tools.getPlaceHolder("loader")), gbc);

        loaderPanel = new JPanel();
        loaderPanel.setLayout(new BoxLayout(loaderPanel, BoxLayout.X_AXIS));
        loaderPanel.add(new VerticalTextComponent("Loader"), BorderLayout.LINE_START);
        loaderPanel.add(contentPane, BorderLayout.CENTER);

        return loaderPanel;
    }

    private JPanel createAutochangerPanel() {
        JPanel contentPane = new JPanel();
        contentPane.setBorder(BorderFactory.createTitledBorder(BorderFactory.createLineBorder(VerticalTextComponent.getSubsystemColor("Autochanger"), 2), ""));
        contentPane.setLayout(new GridBagLayout());
        GridBagConstraints gbc = new GridBagConstraints();

        gbc.insets = insets_std;
        //first column
        gbc.gridx = 0;
        gbc.gridy = 0;
        gbc.anchor = GridBagConstraints.LINE_START;
        contentPane.add(new JLabel("Filter online: "), gbc);
        gbc.gridy++;
        contentPane.add(new JLabel("Filter on trucks: "), gbc);
        gbc.gridy++;
        contentPane.add(new JLabel("Online clamps state: "), gbc);
        gbc.gridy++;
        contentPane.add(new JLabel("Latches state: "), gbc);
        gbc.gridy++;
        contentPane.add(new JLabel("Trucks state: "), gbc);
        gbc.gridy++;
        contentPane.add(new JLabel("Trucks position: "), gbc);
        for (int i = 0; i<2; i++) {
            gbc.gridy++;
            contentPane.add(new FixedSizePanel(Tools.getPlaceHolder("autochanger")), gbc);
        }
        gbc.gridy++;
        contentPane.add(new FixedSizePanel(new DigitalSwitch("fake", "fake")), gbc);

        //second column
        gbc.gridx++;
        gbc.gridy = 0;
        gbc.anchor = GridBagConstraints.LINE_END;
        contentPane.add(filterONLINELabel, gbc);
        gbc.gridy++;
        contentPane.add(filterOnTrucksLabel, gbc);
        gbc.gridy++;
        contentPane.add(onlineClampsStateLabel, gbc);
        gbc.gridy++;
        contentPane.add(latchesStateLabel, gbc);
        gbc.gridy++;
        contentPane.add(trucksPositionNameLabel, gbc);
        gbc.gridy++;
        contentPane.add(trucksPositionLabel, gbc);
        gbc.gridy++;
        contentPane.add(new FixedSizePanel(Tools.getPlaceHolder("autochanger_status")), gbc);


        autochangerPanel = new JPanel();
        autochangerPanel.setLayout(new BoxLayout(autochangerPanel, BoxLayout.X_AXIS));
        autochangerPanel.add(new VerticalTextComponent("Autochanger"));
        autochangerPanel.add(contentPane);

        JPanel fixedAutochangerPanel = new JPanel();
        fixedAutochangerPanel.setLayout(new CardLayout());
        fixedAutochangerPanel.add(autochangerPanel);
        fixedAutochangerPanel.add(Tools.getPanelPlaceHolder("autochanger_pane"));

        return fixedAutochangerPanel;
    }

    /**
     * set default values for components color and labels.
     */
    private void setDefaultValues() {
        /*carousel*/
        actualPositionLabel.setText(ZERO_VALUE);
        actualSpeedLabel.setText(ZERO_VALUE);
        actualCurrentLabel.setText(ZERO_VALUE);
        standbyLabel.setText(NOT_AT_STANDBY);
        standbySwitch.setColor(Color.gray);
        socketAtStandbyLabel.setText(NONE);
        filterAtStandbyLabel.setText(NO_FILTER);
        caClampsStateLabel.setText(UNKNOWN_STATE);
        ioModuleStatusLabel.setText(UNKNOWN_STATE);

        /*autochanger*/
        trucksPositionLabel.setText(ZERO_VALUE);
        trucksPositionNameLabel.setText(UNKNOWN_STATE);
        onlineClampsStateLabel.setText(UNKNOWN_STATE);
        latchesStateLabel.setText(UNKNOWN_STATE);

        /* loader */
        loaderClampHomingLabel.setText(UNKNOWN);
        filterOnLoaderLabel.setText(UNKNOWN);
        loaderClampStateLabel.setText(UNKNOWN);
        loaderClampPositionLabel.setText(ZERO_VALUE);
        loaderCarrierPositionNameLabel.setText(UNKNOWN);
        loaderCarrierPositionLabel.setText(ZERO_VALUE);
    }

    /**
     * Reset Panels to default values.
     */
    public void resetPanel() {
        setDefaultValues();
        cameraStatusPanel.resetPanel();
    }

    /**
     * Toggle the visibility of the loader panels / commands of the overview
     * The goal is to avoid confusing people with meaningless default values
     * when the loader is not connected to the camera
     *
     * @param visible if true, show the loader panels and enable the commands
     */
    private void showLoaderPanels(boolean visible) {
        cameraStatusPanel.showLoaderPanel(visible);
        loadFilterButton.setEnabled(visible);
        unloadFilterButton.setEnabled(visible);
        loaderPanel.setVisible(visible);
        loaderPanel.revalidate();
        loaderPanel.repaint();
    }


    @Override
    public String toString() {
        return title;
    }

    // private String getSelectedFilter() {
    //     String chosenFilter = (String) filterMenu.getSelectedItem();
    //     filterMenu.setSelectedIndex(0);
    //     return chosenFilter;
    // }

    private void setFilterButtonActionPerformed(ActionEvent evt) {
        // String filterName = getSelectedFilter();
        filterID = Integer.parseInt(filterField.getText());
        subs.setFilter(filterID);
    }

    private void setFilterAtHandoffForLoaderButtonActionPerformed(ActionEvent evt) {
        // String filterName = getSelectedFilter();
        filterID = Integer.parseInt(filterField.getText());
        subs.setFilterAtHandoffForLoader(filterID);
    }

    private void setNoFilterButtonActionPerformed(ActionEvent evt) {
        subs.setFilter(0);
    }

    private void setNoFilterAtHandoffForLoaderButtonActionPerformed(ActionEvent evt) {
        subs.setFilterAtHandoffForLoader(0);
    }

    private void loadFilterButtonActionPerformed(ActionEvent evt) {
        subs.loadFilter();
    }

    private void unloadFilterButtonActionPerformed(ActionEvent evt) {
        subs.unloadFilter();
    }

    private void connectLoaderButtonActionPerformed(ActionEvent evt) {
        subs.connectLoader();
    }

    private void disconnectLoaderButtonActionPerformed(ActionEvent evt) {
        subs.disconnectLoader();
    }

    private void storeFilterOnCarouselActionPerformed(ActionEvent evt) {
        subs.storeFilterOnCarousel();
    }

    private void rotateSocketToStandbyActionPerformed(ActionEvent evt) {
        int selectedSocketNb = socketMenu.getSelectedIndex();
        if (selectedSocketNb != 0) {
            this.socketID = selectedSocketNb;
            subs.rotateSocketToStandby(socketID);
            socketMenu.setSelectedIndex(0);
        }
    }

    private void grabFilterAtStandbyActionPerformed(ActionEvent evt) {
        subs.grabFilterAtStandby();
    }

    private void disengageFilterFromCarouselActionPerformed(ActionEvent evt) {
        subs.disengageFilterFromCarousel();
    }

    private void moveAndClampFilterOnlineActionPerformed(ActionEvent evt) {
        subs.moveAndClampFilterOnline();
    }

    private void unclampAndMoveFilterToHandoffActionPerformed(ActionEvent evt) {
        subs.unclampAndMoveFilterToHandoff();
    }

    private void wakeFilterChangerActionPerformed(ActionEvent evt, int mode) {
        subs.wakeFilterChanger(mode);
    }

    /**
     * This main launches an GUI for the whole subsystem "fcs". It's used only
     * for tests.
     *
     * @param args the command line arguments
     */
    public static void main(String[] args) {

        FcsGeneralViewPanel d = new FcsGeneralViewPanel();
        JFrame frame = new JFrame("FCS General View Panel");
        frame.setContentPane(d);
        frame.pack();
        frame.setVisible(true);
        frame.setDefaultCloseOperation(JFrame.EXIT_ON_CLOSE);
    }

}
