
package org.lsst.ccs.subsystems.fcs.loader.ui;

import java.io.Serializable;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;
import javax.swing.JFrame;
import javax.swing.SwingUtilities;
import org.lsst.ccs.bus.data.ConfigurationInfo;
import org.lsst.ccs.bus.data.KeyValueData;
import org.lsst.ccs.bus.data.KeyValueDataList;
import static org.lsst.ccs.subsystems.fcs.FCSCst.CARRIER_CONTROLLER_NAME;
import static org.lsst.ccs.subsystems.fcs.FCSCst.HOOKS_CONTROLLER_NAME;
import static org.lsst.ccs.subsystems.fcs.FCSCst.LOADER_CARRIER_NAME;
import static org.lsst.ccs.subsystems.fcs.FCSCst.LOADER_CLAMP_NAME;
import static org.lsst.ccs.subsystems.fcs.FCSCst.LOADER_NAME;
import static org.lsst.ccs.subsystems.fcs.FCSCst.LOADER_PLUTOGATEWAY_NAME;
import static org.lsst.ccs.subsystems.fcs.FCSCst.LOADER_TCPPROXY_NAME;
import org.lsst.ccs.subsystems.fcs.StatusDataPublishedByEPOSController;
import org.lsst.ccs.subsystems.fcs.StatusDataPublishedByHardware;
import org.lsst.ccs.subsystems.fcs.StatusDataPublishedByLoaderCarrier;
import org.lsst.ccs.subsystems.fcs.StatusDataPublishedByLoaderClamp;
import org.lsst.ccs.subsystems.fcs.StatusDataPublishedByPlutoGateway;
import org.lsst.ccs.subsystems.fcs.ui.commons.EPOSControllerPanel;
import org.lsst.ccs.subsystems.fcs.ui.commons.GatherPanel;
import org.lsst.ccs.subsystems.fcs.ui.commons.InterfaceGeneralGUI;
import org.lsst.ccs.subsystems.fcs.ui.commons.PlutoGatewayPanel;
import org.lsst.ccs.subsystems.fcs.ui.ControllersViewPanel;
import static org.lsst.ccs.subsystems.fcs.ui.commons.Tools.getShortComponentName;

/**
 * This panel gathers all the panels needed for the loader GUI.
 * Except the subsystemPanel which is handled by the GeneralPanel.
 *
 * It takes care of initialization and update of the panels.
 *
 * @author virieux
 */
public class LoaderGatherPanel extends GatherPanel {
    private static final long serialVersionUID = 4741041603580940956L;

    private InterfaceLoaderGUI subs;

    //CANopen devices panels
    private EPOSControllerPanel hooksControllerPanel;
    private EPOSControllerPanel carrierControllerPanel;
    private PlutoGatewayPanel plutoGatewayPanel;

    /** Loader hardware view Panel*/
    private LoaderGeneralViewPanel loaderGeneralViewPanel;
    private ControllersViewPanel controllersViewPanel;
    private LoaderClampPanel loaderClampPanel;
    private LoaderCarrierPanel loaderCarrierPanel;

    /** PLC debug panel */
    private PLCLoaderPanel plcLoaderPanel;


    /**
     * A Runnable to update the carrier controller panel.
     */
    class UpdateCarrierController implements Runnable {

        private final String name;
        private final StatusDataPublishedByEPOSController s;

        public UpdateCarrierController(String name, StatusDataPublishedByEPOSController s) {
            this.name = name;
            this.s = s;
        }

        @Override
        public void run() {
            carrierControllerPanel.updateController(name, s);
        }
    }

    /**
     * A Runnable to update the clamp controller panel.
     */
    class UpdateClampController implements Runnable {

        private final String name;
        private final StatusDataPublishedByEPOSController s;

        public UpdateClampController(String name, StatusDataPublishedByEPOSController s) {
            this.name = name;
            this.s = s;
        }

        @Override
        public void run() {
            hooksControllerPanel.updateController(name, s);
        }
    }

    /**
     * Creates new form LoaderTopPanel
     * @param tcpProxyName
     */
    public LoaderGatherPanel(String tcpProxyName) {
        super(tcpProxyName);
        initPanels();
        setName("LoaderGatherPanel");
    }

    public LoaderGatherPanel(InterfaceLoaderGUI subs, String tcpProxyName) {
        super(tcpProxyName);
        initPanels();
        this.subs = subs;
        loaderGeneralViewPanel.setSubsystem(subs);
        loaderCarrierPanel.setSubsystem(subs);
        loaderClampPanel.setSubsystem(subs);
        this.hooksControllerPanel.setSubsystem(subs);
        this.carrierControllerPanel.setSubsystem(subs);
        this.controllersViewPanel = new ControllersViewPanel(subs);
        setName("LoaderGatherPanel");
    }


    public EPOSControllerPanel getHooksControllerPanel() {
        return hooksControllerPanel;
    }

    public EPOSControllerPanel getCarrierControllerPanel() {
        return carrierControllerPanel;
    }

    public PlutoGatewayPanel getPlutoGatewayPanel() {
        return plutoGatewayPanel;
    }

    public LoaderGeneralViewPanel getLoaderGeneralViewPanel() {
        return loaderGeneralViewPanel;
    }
    public ControllersViewPanel getControllersViewPanel() {
        return controllersViewPanel;
    }
    public LoaderClampPanel getLoaderClampPanel() {
        return loaderClampPanel;
    }

    public LoaderCarrierPanel getLoaderCarrierPanel() {
        return loaderCarrierPanel;
    }

    public PLCLoaderPanel getPLCLoaderPanel() {
        return plcLoaderPanel;
    }

    @Override
    public InterfaceGeneralGUI getSubs() {
        return subs;
    }

    @Override
    public void initializeGUIWithConfigInfo(ConfigurationInfo configInfo,
            List<String> hardwareNames)  {
        /**
         * top panel initialization
         */
        super.initializeGUIWithConfigInfo(configInfo, hardwareNames);
        controllersViewPanel.enableOnlineSubsystems(false, false, true);
        updatePanelsFromConfigInfo(configInfo);
    }

    @Override
    public void initializeGuiWithLoaderSensorsList(ConfigurationInfo configInfo, List<String> sensorsList) {
        FCSLOG.info("initializeGuiWithLoaderSensorsList: " + sensorsList.toString());
        plutoGatewayPanel.initializeGuiWithSensorsList(configInfo, sensorsList);
    }


    private void updatePanelsFromConfigInfo(ConfigurationInfo configInfo) {
        /**
         * Clamp Controller Panel *
         */
        hooksControllerPanel.initializeGui(configInfo);
        /**
         * Carrier Controller Panel *
         */
        carrierControllerPanel.initializeGui(configInfo);

        /**
         * PlutoGateway Panel *
         */
        plutoGatewayPanel.initializeGui(configInfo);

        /**
         * LoaderGeneralViewPanel initialization *
         */
        loaderGeneralViewPanel.initializeGui(configInfo);

        /**
         * Loader Clamp Panel *
         */
        loaderClampPanel.initializeGui(configInfo);

        /**
         * Loader Carrier  Panel *
         */
        loaderCarrierPanel.initializeGui(configInfo);
    }


    private void updateClamp(StatusDataPublishedByLoaderClamp statusDataPublishedByLoaderClamp) {
        this.loaderClampPanel.updateClamp(statusDataPublishedByLoaderClamp);
        this.loaderGeneralViewPanel.updateClamp(statusDataPublishedByLoaderClamp);
        this.plcLoaderPanel.updateClamp(statusDataPublishedByLoaderClamp);
    }

    private void updateCarrier(StatusDataPublishedByLoaderCarrier statusDataPublishedByLoaderCarrier) {
        this.loaderCarrierPanel.updateCarrier(statusDataPublishedByLoaderCarrier);
        this.loaderGeneralViewPanel.updateCarrier(statusDataPublishedByLoaderCarrier);
        this.plcLoaderPanel.updateCarrier(statusDataPublishedByLoaderCarrier);
    }

    private void updateCarrierController(String name, StatusDataPublishedByEPOSController status) {
        SwingUtilities.invokeLater(new UpdateCarrierController(name, status));
        loaderCarrierPanel.updateController(status);
        loaderGeneralViewPanel.updateCarrierController(status);
    }

    private void updateClampController(String name, StatusDataPublishedByEPOSController status) {
        SwingUtilities.invokeLater(new UpdateClampController(name, status));
        loaderClampPanel.updateController(status);
        loaderGeneralViewPanel.updateClampController(status);
    }

    private void updateController(String controllerName, StatusDataPublishedByEPOSController controllerStatus) {
        if (CARRIER_CONTROLLER_NAME.equals(controllerName)) {
            updateCarrierController(controllerName,controllerStatus);
        } else if (HOOKS_CONTROLLER_NAME.equals(controllerName)) {
            updateClampController(controllerName, controllerStatus);
        }
    }

    @Override
    public void updateFromStatusData(KeyValueData data) {
        if (LOADER_TCPPROXY_NAME.equals(data.getKey())) {
            updateCanOpenHardwarePanel((int) data.getValue());

        } else if (data.getValue() instanceof StatusDataPublishedByEPOSController) {
            updateCanOpenHardwarePanel(data.getKey(), (StatusDataPublishedByHardware) data.getValue());
            updateController(getShortComponentName(data.getKey()), (StatusDataPublishedByEPOSController) data.getValue());
            controllersViewPanel.updateFromStatusData(data);

        } else if (data.getValue() instanceof StatusDataPublishedByPlutoGateway) {
            updateCanOpenHardwarePanel(data.getKey(), (StatusDataPublishedByHardware) data.getValue());
            this.plutoGatewayPanel.updatePlutoGateway(getShortComponentName(data.getKey()), (StatusDataPublishedByPlutoGateway) data.getValue());

        } else if (data.getValue() instanceof StatusDataPublishedByHardware) {
            updateCanOpenHardwarePanel(data.getKey(), (StatusDataPublishedByHardware) data.getValue());

        } else if (data.getValue() instanceof StatusDataPublishedByLoaderClamp) {
            updateClamp((StatusDataPublishedByLoaderClamp) data.getValue());

        } else if (data.getValue() instanceof StatusDataPublishedByLoaderCarrier) {
            updateCarrier((StatusDataPublishedByLoaderCarrier) data.getValue());

        } else if (data.getValue() instanceof KeyValueDataList
                && LOADER_PLUTOGATEWAY_NAME.equals(getShortComponentName(data.getKey()))) {
            Map<String, Serializable> sensorMap
                    = ((KeyValueDataList) data.getValue()).getListOfKeyValueData().stream().collect(Collectors.toMap(KeyValueData::getKey, KeyValueData::getValue));

            plutoGatewayPanel.updateSensor((KeyValueDataList) data.getValue());
            plcLoaderPanel.updateFromSensorList(sensorMap);
            loaderGeneralViewPanel.updateFromSensorList(sensorMap);
            loaderClampPanel.updateFromSensorList(sensorMap);
        }
    }

    /**
     * Update GUI when a StatusConfigurationInfo message is received on the status bus.
     * @param configInfo
     */
    @Override
    public void updateFromConfigurationInfo(ConfigurationInfo configInfo) {
        updatePanelsFromConfigInfo(configInfo);
    }

    // <editor-fold defaultstate="collapsed" desc="Form description">
    private void initPanels() {
        //controllers Panels
        hooksControllerPanel = new EPOSControllerPanel(LOADER_TCPPROXY_NAME + "/" + HOOKS_CONTROLLER_NAME);
        carrierControllerPanel = new EPOSControllerPanel(LOADER_TCPPROXY_NAME + "/" + CARRIER_CONTROLLER_NAME);

        //plutoGateway panel
        plutoGatewayPanel = new PlutoGatewayPanel(LOADER_TCPPROXY_NAME + "/" + LOADER_PLUTOGATEWAY_NAME);

        //loader hardware panels
        loaderGeneralViewPanel = new LoaderGeneralViewPanel();
        loaderClampPanel = new LoaderClampPanel(LOADER_NAME + "/" + LOADER_CLAMP_NAME);
        loaderCarrierPanel = new LoaderCarrierPanel(LOADER_NAME + "/" + LOADER_CARRIER_NAME);

        //PLC loader panel
        plcLoaderPanel = new PLCLoaderPanel();
    }

    /**
     * To reset panel to default values.
     */
    @Override
    public void resetPanel() {
        super.resetPanel();
        hooksControllerPanel.resetPanel();
        carrierControllerPanel.resetPanel();
        loaderGeneralViewPanel.resetPanel();
        loaderClampPanel.resetPanel();
        loaderCarrierPanel.resetPanel();
        plcLoaderPanel.resetPanel();
        plutoGatewayPanel.resetPanel();
    }

    public static void main(String[] argv) {

        LoaderGatherPanel d = new LoaderGatherPanel(LOADER_TCPPROXY_NAME);
        JFrame frame = new JFrame("Loader High Panel");
        frame.setContentPane(d);
        frame.pack();
        frame.setVisible(true);
        frame.setDefaultCloseOperation(JFrame.EXIT_ON_CLOSE);

    }

}
