package org.lsst.ccs.subsystems.fcs.carousel.ui;

import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.util.Map;
import javax.swing.BorderFactory;
import javax.swing.JFrame;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.SwingConstants;
import javax.swing.SwingUtilities;
import org.lsst.ccs.bus.data.ConfigurationInfo;
import static org.lsst.ccs.subsystems.fcs.FCSCst.CA_CLAMPXMINUSCONTROLLER_NAME;
import static org.lsst.ccs.subsystems.fcs.FCSCst.CA_CLAMPXPLUSCONTROLLER_NAME;
import org.lsst.ccs.subsystems.fcs.StatusDataPublishedByCarouselClamp;
import org.lsst.ccs.subsystems.fcs.StatusDataPublishedByEPOSController;
import org.lsst.ccs.subsystems.fcs.ui.commons.ControllerStatePanel;
import org.lsst.ccs.subsystems.fcs.ui.commons.Tools;
import static org.lsst.ccs.subsystems.fcs.ui.commons.Tools.BIG_FONT;
import static org.lsst.ccs.subsystems.fcs.ui.commons.Tools.UNKNOWN_STATE;
import static org.lsst.ccs.subsystems.fcs.ui.commons.Tools.ZERO_VALUE;
import static org.lsst.ccs.subsystems.fcs.ui.commons.Tools.insets_std;
import static org.lsst.ccs.subsystems.fcs.ui.commons.Tools.NICE_PINK;

/**
 * A Panel to display data coming from a carousel clamp.
 *
 * @author virieux
 */
public class CarouselClampPanel extends JPanel {

    private static final long serialVersionUID = -1273081770223753869L;
    private final String fullPathClampName;
    private final String lockSensorName;
    private final String filterPresenceSensorName;
    private JLabel clampStateLabel;

    private final JLabel explainText = new JLabel("<HTML><P>UNLOCKED: <P>between minValue and offset1 "
            + "<P>LOCKED: <P>between minLocked and maxLimit</HTML>");

    /* lockSensorPanel*/
    private AnalogicSensorPanel lockSensorPanel;

    /* filterPresenceSensorPanel*/
    private FilterPresenceSensorPanel filterPresenceSensorPanel;

    /* general parameters*/
    private JPanel paramsPanel;
    private final JLabel currentToUnlockLabel = new JLabel("0");
    private final JLabel currentToPrepareUnlockLabel = new JLabel("0");

    /*Unclamping info*/
    private final ControllerStatePanel controllerPanel;

    public CarouselClampPanel(String clampName) {
        this.fullPathClampName = clampName;

        /* to find the name of the sensors of this clamp*/
        String[] words = clampName.split("X");
        lockSensorName = "lockSensor" + "X" + words[1];
        filterPresenceSensorName = "filterPresence" + "X" + words[1];
        String controllerName = Tools.getShortComponentName(fullPathClampName).contains("Xminus") ? CA_CLAMPXMINUSCONTROLLER_NAME : CA_CLAMPXPLUSCONTROLLER_NAME;
        controllerPanel = new ControllerStatePanel(controllerName, false);
        initComponents();
    }

    /**
     * Initialize the GUI with the configuration data.
     *
     * @param configInfo
     */
    public void initializeGui(ConfigurationInfo configInfo) {
        Map<String, String> config = configInfo.getCurrentValuesFor(fullPathClampName);
        currentToUnlockLabel.setText(config.get("currentToUnlock"));
        Map<String, String> configCa = configInfo.getCurrentValuesFor("carousel");
        currentToPrepareUnlockLabel.setText(configCa.get("currentToPrepareUnlock"));

        lockSensorPanel.initializeGui(configInfo);
        filterPresenceSensorPanel.initializeGui(configInfo);

    }

    /**
     * Update from status data published by a Clamp on Status bus.
     *
     * @param s
     */
    protected void updateClamp(StatusDataPublishedByCarouselClamp s) {
        SwingUtilities.invokeLater(new UpdateClamp(s));
    }

    /**
     * Update from status data published by a controller.
     * @param s
     */
    public void updateController(StatusDataPublishedByEPOSController s) {
        controllerPanel.update(s);
    }

    /**
     * A Runnable class to update Panel from a
     * StatusDataPublishedByCarouselClamp.
     */
    public class UpdateClamp implements Runnable {

        private StatusDataPublishedByCarouselClamp s;

        public UpdateClamp(StatusDataPublishedByCarouselClamp s) {
            this.s = s;
        }

        @Override
        public void run() {
            clampStateLabel.setText(s.getClampState().toString());
            clampStateLabel.setForeground(s.getClampState().getColor());
            lockSensorPanel.update(s.getLockSensorValue(), s.getLockStatus());
            lockSensorPanel.updateOffset1(s.getLockSensorOffset1());
            filterPresenceSensorPanel.update(s.getFilterPositionSensorValue(), s.getFilterPresenceStatus());
            filterPresenceSensorPanel.updateOffset1(s.getFilterPresenceMinLimit());
            filterPresenceSensorPanel.updateOffset2(s.getFilterPresenceOffset2());
        }
    }

    public void updateMinLocked(long value) {
        lockSensorPanel.updateMinLocked(value);
    }


    // <editor-fold defaultstate="collapsed" desc="Form description">
    private void initComponents() {

        /*Clamp State*/
        clampStateLabel = new JLabel(UNKNOWN_STATE);
        clampStateLabel.setFont(BIG_FONT);

        /**
         ******************************************************************************
         ** Sensors Panels
         * *****************************************************************************
         */
        lockSensorPanel = new AnalogicSensorPanel(fullPathClampName, lockSensorName, SwingConstants.HORIZONTAL);
        filterPresenceSensorPanel = new FilterPresenceSensorPanel(fullPathClampName, filterPresenceSensorName,
                SwingConstants.VERTICAL);

        /**
         ******************************************************************************
         ** end Sensors Panel
         * *****************************************************************************
         */

        /**
         ******************************************************************************
         ** General Parameters Panels
         * *****************************************************************************
         */
        paramsPanel = new JPanel();
        paramsPanel.setBorder(Tools.getConfigurationTitle());
        paramsPanel.setLayout(new GridBagLayout());
        GridBagConstraints gbcp = new GridBagConstraints();
        gbcp.insets = insets_std;
        gbcp.gridx = 0;
        gbcp.gridy = 0;
        /*first columm*/
        gbcp.anchor = GridBagConstraints.LINE_START;
        paramsPanel.add(new JLabel("Current to unlock:"), gbcp);
        gbcp.gridy++;
        paramsPanel.add(new JLabel("<HTML>Current to <P>prepare unlock:</HTML>"), gbcp);
        gbcp.gridx++;
        /*second columm*/
        gbcp.anchor = GridBagConstraints.LINE_END;
        gbcp.gridx = 1;
        gbcp.gridy = 0;
        paramsPanel.add(currentToUnlockLabel, gbcp);
        gbcp.gridy++;
        paramsPanel.add(currentToPrepareUnlockLabel, gbcp);
        /**
         ******************************************************************************
         ** end of General Parameters Panels
         * *****************************************************************************
         */

        /**
         * /**
         *
         ******************************************************************************
         ** Whole Panel
         * *****************************************************************************
         */
        String clampName = Tools.getShortComponentName(fullPathClampName);
        setBorder(BorderFactory.createTitledBorder(BorderFactory.createLineBorder(NICE_PINK), clampName));
        GridBagConstraints gbc = new GridBagConstraints();
        setLayout(new GridBagLayout());
        gbc.insets = insets_std;
        gbc.gridx = 0;
        gbc.gridy = 0;
        /*clamp state*/
        add(clampStateLabel, gbc);

        /*parameters Panel*/
        gbc.gridx++;
        add(controllerPanel, gbc);

        /* explanation text*/
        gbc.gridx = 0;
        gbc.gridy++;
        add(explainText, gbc);

        /* unclamping panel*/
        gbc.gridx++;
        add(paramsPanel, gbc);

        gbc.gridx = 0;
        gbc.gridy++;
        add(lockSensorPanel, gbc);

        gbc.gridx++;
        add(filterPresenceSensorPanel, gbc);
        /**
         ******************************************************************************
         ** end Whole Panel
         * *****************************************************************************
         */

        setDefaultValues();
    }

    public void setDefaultValues() {
        currentToUnlockLabel.setText(ZERO_VALUE);
        currentToPrepareUnlockLabel.setText(ZERO_VALUE);
        clampStateLabel.setText(UNKNOWN_STATE);
        controllerPanel.resetPanel();
    }

    /**
     * Reset panel to default values.
     */
    public void resetPanel() {
        setDefaultValues();
        lockSensorPanel.resetPanel();
        filterPresenceSensorPanel.resetPanel();
    }

    public void resetControllerPanel() {
        controllerPanel.resetPanel();
    }

    /**
     * Just to test the Panel.
     *
     * @param argv
     */
    public static void main(String[] argv) {

        CarouselClampPanel d = new CarouselClampPanel("clampXminus");
        JFrame frame = new JFrame("Test");
        frame.setContentPane(d);
        frame.pack();
        frame.setVisible(true);
        frame.setDefaultCloseOperation(JFrame.EXIT_ON_CLOSE);

    }
}
