package org.lsst.ccs.subsystems.fcs.ui;

import org.lsst.ccs.bus.data.KeyValueData;
import org.lsst.ccs.subsystems.fcs.StatusDataPublishedByAutochangerSeneca;
import org.lsst.ccs.subsystems.fcs.StatusDataPublishedByCarouselBrakes;
import org.lsst.ccs.subsystems.fcs.StatusDataPublishedByHYTTC580TemperaturesDevices;
import org.lsst.ccs.subsystems.fcs.ui.commons.InterfaceGeneralGUI;
import org.lsst.ccs.subsystems.fcs.ui.commons.Tools;

import javax.swing.*;
import javax.swing.table.DefaultTableModel;
import java.text.DecimalFormat;
import java.util.logging.Logger;

/**
 * TemperaturesViewPanel
 * Displays all temperatures from Carousel and Autochanger.
 * Note: no temperatures on Loader
 */
public class TemperaturesViewPanel extends JPanel {
    private static final Logger FCSLOG = Logger.getLogger(TemperaturesViewPanel.class.getName());
    private static final int tempCol = 1;
    private static final int noOfSockets = 5;
    private static final int noOfAutochangerTemperatures = 8;
    private final InterfaceGeneralGUI subs;
    private final String title = "Temperatures Overview";
    private final DecimalFormat formatter = new DecimalFormat("#####0.00");
    private DefaultTableModel caTableModel;
    private DefaultTableModel acTableModel;

    public TemperaturesViewPanel(InterfaceGeneralGUI subs) {
        this.subs = subs;
        initComponents();
    }

    @Override
    public String toString() {
        return title;
    }

    private void initComponents() {
        java.util.Vector<String> columnNames = new java.util.Vector<String>() {{
            add("Name");
            add("Temperature (°C)");
        }};
        caTableModel = new DefaultTableModel(columnNames, 0);
        JTable caTempTable = new JTable(caTableModel);
        initCarouselTable();
        acTableModel = new DefaultTableModel(columnNames, 0);
        JTable acTempTable = new JTable(acTableModel);
        initAutochangerTable();

        setBorder(Tools.getGeneralPanelTitle(title));
        JScrollPane scrollPane1 = new JScrollPane(caTempTable);
        scrollPane1.setBorder(BorderFactory.createTitledBorder("Carousel temperature sensors"));
        scrollPane1.setPreferredSize(new java.awt.Dimension(600, 250));
        JScrollPane scrollPane2 = new JScrollPane(acTempTable);
        scrollPane2.setBorder(BorderFactory.createTitledBorder("Autochanger temperature sensors"));
        scrollPane2.setPreferredSize(new java.awt.Dimension(600, 200));
        JPanel panel = new JPanel(new java.awt.BorderLayout());
        panel.add(scrollPane1, java.awt.BorderLayout.NORTH);
        panel.add(scrollPane2, java.awt.BorderLayout.CENTER);
        add(panel);
    }

    private void initCarouselTable() {
        caTableModel.addRow(new Object[]{"Motor in Bay D", ""});
        caTableModel.addRow(new Object[]{"Brake in Bay I", ""});
        caTableModel.addRow(new Object[]{"Brake in Bay N", ""});
        caTableModel.addRow(new Object[]{"Brake in Bay X", ""});
        caTableModel.addRow(new Object[]{"Clamps (average)", ""});
        caTableModel.addRow(new Object[]{"TTC580", ""});
        for (int i=1; i<=noOfSockets; i++) {
            caTableModel.addRow(new Object[]{"Socket " + i, ""});
        }
        caTableModel.fireTableDataChanged();
    }

    private void initAutochangerTable() {
        for (StatusDataPublishedByAutochangerSeneca.TempId tempId : StatusDataPublishedByAutochangerSeneca.TempId.values()) {
            acTableModel.addRow(new Object[]{tempId.getDescription(), ""});
        }
        acTableModel.fireTableDataChanged();
    }

    protected void updateFromStatusData(KeyValueData data) {
        SwingUtilities.invokeLater(new Update(data));
    }

    private enum carouselRow {
        BayD, BayI, BayN, BayX, AverageClamps, TTC580, Socket1
    }

    public class Update implements Runnable {
        private final KeyValueData data;

        public Update(KeyValueData data) {
            this.data = data;
        }

        @Override
        public void run() {
            Class<?> actualClass = data.getValue().getClass();
            switch (actualClass.getSimpleName()) {
                case "StatusDataPublishedByCarouselBrakes":
                    updateCarouselBrakes((StatusDataPublishedByCarouselBrakes) data.getValue());
                    break;
                case "StatusDataPublishedByHYTTC580TemperaturesDevices":
                    updateTTC580Temperatures((StatusDataPublishedByHYTTC580TemperaturesDevices) data.getValue());
                    break;
                case "StatusDataPublishedByAutochangerSeneca":
                    updateAutochangerTemperatures((StatusDataPublishedByAutochangerSeneca) data.getValue());
                    break;
                default:
            }
        }

        private void updateCarouselBrakes(StatusDataPublishedByCarouselBrakes s) {
            caTableModel.setValueAt(formatter.format(s.getTemperature1()), carouselRow.BayD.ordinal(), tempCol);
            caTableModel.setValueAt(formatter.format(s.getTemperature2()), carouselRow.BayI.ordinal(), tempCol);
            caTableModel.setValueAt(formatter.format(s.getTemperature3()), carouselRow.BayN.ordinal(), tempCol);
            caTableModel.setValueAt(formatter.format(s.getTemperature4()), carouselRow.BayX.ordinal(), tempCol);
            caTableModel.fireTableDataChanged();
        }

        private void updateTTC580Temperatures(StatusDataPublishedByHYTTC580TemperaturesDevices s) {
            for (int i=0; i<noOfSockets; i++ ) {
                caTableModel.setValueAt(formatter.format(s.getSocketTemperature(i+1)), carouselRow.Socket1.ordinal()+i, tempCol);
            }
            caTableModel.setValueAt(formatter.format(s.getAverageClampsTemperature()), carouselRow.AverageClamps.ordinal(), tempCol);
            caTableModel.setValueAt(formatter.format(s.getInternalTemperature()), carouselRow.TTC580.ordinal(), tempCol);
            caTableModel.fireTableDataChanged();
        }

        private void updateAutochangerTemperatures(StatusDataPublishedByAutochangerSeneca s) {
            for (int i=0; i<noOfAutochangerTemperatures; i++ ) {
                acTableModel.setValueAt(formatter.format(s.getTemperature(i)), StatusDataPublishedByAutochangerSeneca.TempId.LINEAR_RAIL_MOTOR_X_PLUS.ordinal()+i, tempCol);
            }
            acTableModel.fireTableDataChanged();
        }
    }

}
