
package org.lsst.ccs.subsystems.fcs.carousel.ui;

import java.awt.Color;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.event.ActionEvent;
import java.io.Serializable;
import java.util.Map;
import java.util.logging.Logger;
import javax.swing.BorderFactory;
import javax.swing.JButton;
import javax.swing.JFrame;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JComboBox;
import javax.swing.SwingUtilities;
import org.lsst.ccs.bus.data.ConfigurationInfo;
import static org.lsst.ccs.subsystems.fcs.FCSCst.NO_FILTER;
import static org.lsst.ccs.subsystems.fcs.FCSCst.AT_STANDBY;
import static org.lsst.ccs.subsystems.fcs.FCSCst.NOT_AT_STANDBY;
import org.lsst.ccs.subsystems.fcs.StatusDataPublishedByAccelerometer;
import org.lsst.ccs.subsystems.fcs.StatusDataPublishedByCarousel;
import org.lsst.ccs.subsystems.fcs.StatusDataPublishedByCarouselBrakes;
import org.lsst.ccs.subsystems.fcs.StatusDataPublishedByCarouselSocket;
import org.lsst.ccs.subsystems.fcs.StatusDataPublishedByEPOSController;
import static org.lsst.ccs.subsystems.fcs.carousel.ui.PLCCarouselPanel.POWSAV;
import org.lsst.ccs.subsystems.fcs.ui.commons.AcceleroPanel;
import org.lsst.ccs.subsystems.fcs.ui.commons.DigitalSwitch;
import org.lsst.ccs.subsystems.fcs.ui.commons.Tools;

import static org.lsst.ccs.subsystems.fcs.ui.commons.Tools.BIG_FONT;
import static org.lsst.ccs.subsystems.fcs.ui.commons.Tools.NICE_PINK;
import static org.lsst.ccs.subsystems.fcs.ui.commons.Tools.NONE;
import static org.lsst.ccs.subsystems.fcs.ui.commons.Tools.TIRETS;
import static org.lsst.ccs.subsystems.fcs.ui.commons.Tools.UNKNOWN_STATE;
import static org.lsst.ccs.subsystems.fcs.ui.commons.Tools.ZERO_VALUE;
import static org.lsst.ccs.subsystems.fcs.ui.commons.Tools.greenColor;
import static org.lsst.ccs.subsystems.fcs.ui.commons.Tools.insets_std;

/**
 * This view represents the carousel in a general design.
 * @author virieux
 */
public class CarouselGeneralViewPanel extends JPanel {
    private static final long serialVersionUID = 3579024322081964467L;
    private static final Logger FCSLOG = Logger.getLogger(CarouselGeneralViewPanel.class.getName());
    private InterfaceCarouselGUI subs;

    private final String pl = "Position_";

    /*General Informations Panel (Configuration parameters)*/
    private final JPanel generalInfoPanel = new JPanel();
    private final JLabel profileVelocityLabel = new JLabel();
    private final JLabel profileAccelerationLabel = new JLabel();
    private final JLabel profileDecelerationLabel = new JLabel();
    private final JLabel rotationTimeoutLabel = new JLabel();

    /*Actual parameters (dynamic parameters */
    private final JPanel actualParametersPanel = new JPanel();
    private final JLabel actualPositionLabel = new JLabel();
    private final JLabel estimatedPositionLabel = new JLabel();
    private final JLabel posLabel = new JLabel("Position: ");
    private final JLabel deltaPositionLabel = new JLabel();
    private final JLabel positionSensorTypeLabel = new JLabel();
    private final JLabel actualSpeedLabel = new JLabel();
    private final JLabel actualCurrentLabel = new JLabel();
    private final JLabel standbyLabel = new JLabel();
    private final DigitalSwitch standbySwitch = new DigitalSwitch();
    private final JLabel socketAtStandbyLabel = new JLabel();
    private final JLabel filterAtStandbyLabel = new JLabel();
    private final JLabel powerSaveLabel = new JLabel();

    /*Carousel Controller Panel */
    private final JPanel controllerPanel = new JPanel();
    private final DigitalSwitch controllerSwitch = new DigitalSwitch();
    private final JLabel eposStateLabel = new JLabel();

    /*Control Panel*/
    private final JPanel controlPanel = new JPanel();
    private final JButton updatePositionButton = new JButton("Update Position");
    private final JButton rotateButton = new JButton("RotateSocketToStandby");
    private final JButton slowModeButton = new JButton("SetSlowMode");
    private final JButton fastModeButton = new JButton("SetFastMode");
    private final String[] socketList = {
        "        ", "Socket 1", "Socket 2", "Socket 3", "Socket 4", "Socket 5"};
    private final JComboBox<String> socketMenu = new JComboBox<>(socketList);
    private final JLabel socketLabel = new JLabel("Choose destination socket: ");
    /* socket to rotate at Standby */
    private int socketID;

    /* a table of sockets with standby position */
    private final CarouselSocketTablePanel socketsPanel = new CarouselSocketTablePanel();

    /* Accelerometer panel */
    private final AcceleroPanel acceleroPanel = new AcceleroPanel();

    /* Brakes panel */
    private final BrakesPanel brakesPanel = new BrakesPanel();

    /* Clamps temperature panel */
    private final JPanel averageTemperaturePanel = new JPanel();
    private final JLabel averageTemperatureLabel = new JLabel();

    public CarouselGeneralViewPanel() {
        initComponents();
    }

    protected void setSubsystem(InterfaceCarouselGUI subs) {
        this.subs = subs;
    }

    private void setDefaultValues() {
        profileVelocityLabel.setText(ZERO_VALUE);
        profileAccelerationLabel.setText(ZERO_VALUE);
        profileDecelerationLabel.setText(ZERO_VALUE);
        rotationTimeoutLabel.setText(ZERO_VALUE);
        actualPositionLabel.setText(TIRETS);
        estimatedPositionLabel.setText(TIRETS);
        deltaPositionLabel.setText(TIRETS);
        actualSpeedLabel.setText(ZERO_VALUE);
        actualCurrentLabel.setText(ZERO_VALUE);
        standbyLabel.setText(NOT_AT_STANDBY);
        standbySwitch.setColor(Color.gray);
        socketAtStandbyLabel.setText(NONE);
        filterAtStandbyLabel.setText(NO_FILTER);
        controllerSwitch.setColor(Color.gray);
        eposStateLabel.setText(UNKNOWN_STATE);
        socketMenu.setSelectedIndex(0);
        averageTemperatureLabel.setText(ZERO_VALUE);
    }

    protected void initializeGui(ConfigurationInfo configInfo) {
        socketsPanel.initializeGui(configInfo);
    }

    protected void update(StatusDataPublishedByCarousel s) {
        FCSLOG.finer(() -> getName() + " updating from carousel data:");
        SwingUtilities.invokeLater(new Update(s));
    }

    protected void update(StatusDataPublishedByCarouselBrakes s) {
        FCSLOG.finer(() -> getName() + " updating from carousel brakes data:");
        brakesPanel.update(s);
    }

    protected void update(StatusDataPublishedByEPOSController s) {
        FCSLOG.finer(() -> getName() + " updating from carousel controller data:");
        SwingUtilities.invokeLater(new UpdateController(s));
        brakesPanel.updateController(s);
    }

    protected void update(StatusDataPublishedByAccelerometer s) {
        FCSLOG.finer(() -> getName() + " updating from accelerometer data:");
        acceleroPanel.updateAccelero(s);
    }

    protected void update(StatusDataPublishedByCarouselSocket s) {
        FCSLOG.finer(() -> getName() + " updating from socket data:");
        socketsPanel.updateSocket(s);
    }

    public void updateFromSensorList(Map<String, Serializable> sMap) {
        SwingUtilities.invokeLater(new UpdateFromSensorList(sMap));
    }

    /**
     * A Runnable class to update this panel from an object StatusDataPublishedByCarousel.
     */
    public class Update implements Runnable {
        private final StatusDataPublishedByCarousel s;

        public Update(StatusDataPublishedByCarousel s) {
            this.s = s;
        }

        @Override
        public void run() {
            /*Update of actual parameters Panel*/
            actualPositionLabel.setText(Integer.toString(s.getPosition()));
            String sensorType;
            switch (s.getPositionSensorType() & 0xF) {
                case 4:
                    //when no rotating sensorType = Ssi
                    sensorType = "Ssi";
                    estimatedPositionLabel.setText(TIRETS);
                    break;
                case 8:
                    //during rotation sensorType = Inc.
                    sensorType = "Inc";
                    estimatedPositionLabel.setText(Integer.toString(s.getEstimatedPosition()));
                    break;
                default:
                    sensorType = Integer.toString(s.getPositionSensorType());
                    estimatedPositionLabel.setText(TIRETS);
                    break;
            }
            positionSensorTypeLabel.setText(sensorType);
            posLabel.setText(pl + sensorType);
            rotationTimeoutLabel.setText(Long.toString(s.getRotationTimeout()));
            averageTemperatureLabel.setText(Long.toString(s.getMeanClampsTemperature()));

            // display information that make sense only if carousel is stopped at STANDBY.
            if (s.isAtStandby() && !s.isMoving()) {
                deltaPositionLabel.setText(Integer.toString(s.getDeltaPositionAtStandby()));
                standbyLabel.setText(AT_STANDBY);
                standbySwitch.setColor(greenColor);
                socketAtStandbyLabel.setText(s.getSocketAtStandbyName());
                socketAtStandbyLabel.setFont(BIG_FONT);
                //if not filter at standby s.getFilterAtStandbyName() == NO_FILTER
                if (s.isEmptyAtStandby()) {
                    filterAtStandbyLabel.setText(s.getFilterAtStandbyName());
                } else {
                    filterAtStandbyLabel.setText(s.getFilterAtStandbyName() + "/" + s.getFilterAtStandbyId());
                }
                filterAtStandbyLabel.setFont(BIG_FONT);

            } else {
                deltaPositionLabel.setText(TIRETS);
                standbyLabel.setText(NOT_AT_STANDBY);
                standbySwitch.setColor(Color.orange);
                filterAtStandbyLabel.setText(TIRETS);
                if (s.isMoving()) {
                    socketAtStandbyLabel.setText(TIRETS);
                } else {
                    //socketName could be "NO_SOCKET_AT_STANDBY" or "ERROR_READING_ID"
                    socketAtStandbyLabel.setText(s.getSocketAtStandbyName());
                }
            }
        }

    }

    /**
     * A Runnable to update controller panel with data published by carousel
     * controller.
     */
    public class UpdateController implements Runnable {

        private final StatusDataPublishedByEPOSController s;

        private UpdateController(StatusDataPublishedByEPOSController s) {
            this.s = s;
        }

        @Override
        public void run() {
            if (s.isBooted() && s.getState() != null) {
                eposStateLabel.setText(s.getState().name());
                controllerSwitch.setColor(s.getState().getColor());
                actualSpeedLabel.setText(Double.toString(s.getVelocity()));
                actualCurrentLabel.setText(Double.toString(s.getCurrent()));
                profileVelocityLabel.setText(Long.toString(s.getProfileVelocity()));
                profileAccelerationLabel.setText(Long.toString(s.getProfileAcceleration()));
                profileDecelerationLabel.setText(Long.toString(s.getProfileDeceleration()));
            } else if (!s.isBooted()) {
                eposStateLabel.setText("not booted");
                controllerSwitch.setColor(Color.RED);
            }
        }
    }

    /**
     * To update this panel from a map of sensors : name, value.
     */
    private class UpdateFromSensorList implements Runnable {

        private final Map<String, Serializable> sensorMap;

        public UpdateFromSensorList(Map<String, Serializable> sensorMap) {
            this.sensorMap = sensorMap;
        }

        @Override
        public void run() {
            if (sensorMap.containsKey(POWSAV)) {
                boolean power_save = (int) sensorMap.get(POWSAV) == 1;
                if (power_save) {
                    powerSaveLabel.setText("ON");
                } else {
                    powerSaveLabel.setText("OFF");
                }
            }
        }
    }



    // <editor-fold defaultstate="collapsed" desc="Form Description">
    private void initComponents() {
        setName("carouselGeneralViewPanel");

        /**
         ******************************************************************************
         ** Buttons initialization
         * *****************************************************************************
         */
        updatePositionButton.addActionListener((ActionEvent evt) -> {
            updatePositionButtonActionPerformed(evt);
        });

        slowModeButton.addActionListener((ActionEvent evt) -> {
            slowModeButtonActionPerformed(evt);
        });

        fastModeButton.addActionListener((ActionEvent evt) -> {
            fastModeButtonActionPerformed(evt);
        });

        rotateButton.addActionListener((ActionEvent evt) -> {
            rotateButtonActionPerformed(evt);
        });

        /**
         ******************************************************************************
         ** end of Buttons initialization
         * *****************************************************************************
         */


        setDefaultValues();
        /**
         ******************************************************************************
         ** General Informations Panel
         * *****************************************************************************
         */
        generalInfoPanel.setBorder(BorderFactory.createTitledBorder("General Information"));
        generalInfoPanel.setLayout(new GridBagLayout());
        GridBagConstraints gbc = new GridBagConstraints();
        gbc.insets = insets_std;
        //first column
        gbc.gridx = 0;
        gbc.gridy = 0;
        gbc.anchor = GridBagConstraints.LINE_START;

        generalInfoPanel.add(new JLabel("ProfileVelocity:"), gbc);
        gbc.gridy++;
        generalInfoPanel.add(new JLabel("ProfileAcceleration:"), gbc);
        gbc.gridy++;
        generalInfoPanel.add(new JLabel("ProfileDeceleration:"), gbc);
        gbc.gridy++;
        generalInfoPanel.add(new JLabel("RotationTimeout:"), gbc);

        //second column
        gbc.gridx++;
        gbc.gridy = 0;
        gbc.anchor = GridBagConstraints.LINE_END;
        generalInfoPanel.add(profileVelocityLabel, gbc);
        gbc.gridy++;
        generalInfoPanel.add(profileAccelerationLabel, gbc);
        gbc.gridy++;
        generalInfoPanel.add(profileDecelerationLabel, gbc);
        gbc.gridy++;
        generalInfoPanel.add(rotationTimeoutLabel, gbc);

        /**
         ******************************************************************************
         ** end of General Informations Panel
         * *****************************************************************************
         */

        /**
         ******************************************************************************
         ** Current position Panel
         * *****************************************************************************
         */
        actualParametersPanel.setBorder(BorderFactory.createLineBorder(Color.yellow));
        actualParametersPanel.setLayout(new GridBagLayout());
        GridBagConstraints gbc1 = new GridBagConstraints();
        gbc1.insets = insets_std;
        //first column
        gbc1.gridx = 0;
        gbc1.gridy = 0;
        gbc1.anchor = GridBagConstraints.LINE_START;
        actualParametersPanel.add(posLabel, gbc1);
        gbc1.gridy++;
        actualParametersPanel.add(new JLabel("Estimated Position: "), gbc1);
        gbc1.gridy++;
        actualParametersPanel.add(new JLabel("Delta Position: "), gbc1);
        gbc1.gridy++;
        actualParametersPanel.add(new JLabel("P. Sensor Type: "), gbc1);
        gbc1.gridy++;
        actualParametersPanel.add(new JLabel("Speed: "),gbc1);
        gbc1.gridy++;
        actualParametersPanel.add(new JLabel("Current: "), gbc1);
        gbc1.gridy++;
        actualParametersPanel.add(new JLabel("PowerSave: "), gbc1);
        gbc1.gridy++;
        actualParametersPanel.add(standbyLabel,gbc1);
        gbc1.gridy++;
        actualParametersPanel.add(new JLabel("Socket at STANDBY: "),gbc1);
        gbc1.gridy++;
        actualParametersPanel.add(new JLabel("Filter at STANDBY: "),gbc1);
        //second column
        gbc1.gridx++;
        gbc1.gridy = 0;
        gbc1.anchor = GridBagConstraints.LINE_END;
        actualParametersPanel.add(actualPositionLabel, gbc1);
        gbc1.gridy++;
        actualParametersPanel.add(estimatedPositionLabel, gbc1);
        gbc1.gridy++;
        actualParametersPanel.add(deltaPositionLabel, gbc1);
        gbc1.gridy++;
        actualParametersPanel.add(positionSensorTypeLabel, gbc1);
        gbc1.gridy++;
        actualParametersPanel.add(actualSpeedLabel, gbc1);
        gbc1.gridy++;
        actualParametersPanel.add(actualCurrentLabel, gbc1);
        gbc1.gridy++;
        actualParametersPanel.add(powerSaveLabel, gbc1);
        gbc1.gridy++;
        standbySwitch.setColor(Color.gray);
        actualParametersPanel.add(standbySwitch,gbc1);
        gbc1.gridy++;
        actualParametersPanel.add(socketAtStandbyLabel,gbc1);
        gbc1.gridy++;
        actualParametersPanel.add(filterAtStandbyLabel,gbc1);

        /**
         ******************************************************************************
         ** end of Current position Panel
         * *****************************************************************************
         */
        /**
         ******************************************************************************
         ** Carousel Controller Panel
         * *****************************************************************************
         */
        controllerPanel.setBorder(BorderFactory.createLineBorder(Color.MAGENTA, WIDTH + 5));
        controllerPanel.setLayout(new GridBagLayout());
        gbc = new GridBagConstraints();
        gbc.insets = insets_std;
        gbc.anchor = GridBagConstraints.LINE_START;
        gbc.gridx = 0;
        gbc.gridy = 0;
        controllerPanel.add(new JLabel("Controller status :"), gbc);
        gbc.gridy++;
        gbc.anchor = GridBagConstraints.CENTER;
        controllerSwitch.setSize(40);
        gbc.gridheight = 2;
        controllerPanel.add(controllerSwitch, gbc);
        gbc.gridheight = 1;
        gbc.gridx++;
        gbc.gridy++;
        controllerPanel.add(eposStateLabel, gbc);
        /**
         ******************************************************************************
         ** end of Carousel Controller Panel
         * *****************************************************************************
         */

        /**
         ******************************************************************************
         ** Control Panel
         * *****************************************************************************
         */
        controlPanel.setLayout(new GridBagLayout());
        GridBagConstraints gbc4 = new GridBagConstraints();
        gbc4.insets = insets_std;
        gbc4.anchor = GridBagConstraints.FIRST_LINE_START;
        gbc4.fill = GridBagConstraints.HORIZONTAL;
        gbc4.gridx = 0;
        gbc4.gridy = 0;
        gbc4.gridwidth = 2;
        controlPanel.add(updatePositionButton, gbc4);
        gbc4.gridy++;
        controlPanel.add(slowModeButton, gbc4);
        gbc4.gridy++;
        controlPanel.add(fastModeButton, gbc4);
        gbc4.gridy++;
        gbc4.anchor = GridBagConstraints.EAST;
        gbc4.gridwidth = 1;
        controlPanel.add(socketLabel, gbc4);
        gbc4.gridx++;
        gbc4.anchor = GridBagConstraints.WEST;
        gbc4.fill = GridBagConstraints.NONE;
        controlPanel.add(socketMenu, gbc4);
        gbc4.gridy++;
        gbc4.gridx = 0;
        gbc4.gridwidth = 2;
        gbc4.fill = GridBagConstraints.HORIZONTAL;
        gbc4.anchor = GridBagConstraints.FIRST_LINE_START;
        controlPanel.add(rotateButton, gbc4);
        /**
         ******************************************************************************
         ** end of Control Panel
         * *****************************************************************************
         */

        averageTemperaturePanel.setLayout(new GridBagLayout());
        gbc = new GridBagConstraints();
        averageTemperaturePanel.setBorder(BorderFactory.createTitledBorder(BorderFactory.createLineBorder(NICE_PINK), "Clamps Temperature"));
        gbc.insets = insets_std;
        gbc.anchor = GridBagConstraints.LINE_START;
        gbc.gridx = 0;
        gbc.gridy = 0;
        averageTemperaturePanel.add(new JLabel("Average clamps temperature in mV:"), gbc);
        gbc.gridx++;
        gbc1.anchor = GridBagConstraints.LINE_END;
        averageTemperaturePanel.add(averageTemperatureLabel, gbc);

        /**
         ******************************************************************************
         ** Whole Panel
         * *****************************************************************************
         * /**
         * *****************************************************************************
         * * Whole Panel
         * *****************************************************************************
         */
        /*General Information Panel*/
        setBorder(Tools.getGeneralPanelTitle("Carousel Overview"));
        setLayout(new java.awt.GridBagLayout());
        gbc = new GridBagConstraints();
        gbc.insets = insets_std;
        gbc.anchor = GridBagConstraints.CENTER;

        gbc.gridx = 0;
        gbc.gridy = 0;
        add(generalInfoPanel, gbc);
        gbc.gridx++;
        add(controllerPanel, gbc);

        gbc.gridx = 0;
        gbc.gridy++;
        add(actualParametersPanel, gbc);
        gbc.gridx++;
        add(controlPanel, gbc);

        gbc.gridx = 0;
        gbc.gridy++;
        add(socketsPanel, gbc);

        gbc.gridy++;
        add(brakesPanel, gbc);

        gbc.gridy++;
        add(averageTemperaturePanel, gbc);

        gbc.gridy++;
        add(acceleroPanel, gbc);

        /**
         ******************************************************************************
         ** end of Whole Panel
         * *****************************************************************************
         */
    }

    private void updatePositionButtonActionPerformed(ActionEvent evt) {
        subs.updatePosition();
    }

    private void slowModeButtonActionPerformed(ActionEvent evt) {
        subs.setSlowMode();
    }

    private void fastModeButtonActionPerformed(ActionEvent evt) {
        subs.setFastMode();
    }

    private void rotateButtonActionPerformed(ActionEvent evt) {
        int selectedSocketNb = socketMenu.getSelectedIndex();
        if (selectedSocketNb != 0) {
            this.socketID = selectedSocketNb;
            subs.rotate(socketID);
            socketMenu.setSelectedIndex(0);
        }
    }


    /**
     * Reset Panel to default values when subsystem is shutdown.
     */
    public void resetPanel() {
        setDefaultValues();
        socketsPanel.resetPanel();
        acceleroPanel.resetPanel();
        brakesPanel.resetPanel();
    }

    @Override
    public String toString() {
        return "Overview";
    }

    /**
     * Just to test the Panel.
     * @param argv
     */
    public static void main(String[] argv) {

        CarouselGeneralViewPanel d = new CarouselGeneralViewPanel();
        JFrame frame = new JFrame("Test");
        frame.setContentPane(d);
        frame.pack();
        frame.setVisible(true);
        frame.setDefaultCloseOperation(JFrame.EXIT_ON_CLOSE);

    }

}
