package org.lsst.ccs.subsystems.fcs.loader.ui;

import java.awt.Color;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.event.ActionEvent;
import java.io.Serializable;
import java.util.Map;
import javax.swing.BorderFactory;
import javax.swing.JButton;
import javax.swing.JFrame;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JProgressBar;
import javax.swing.SwingUtilities;
import org.lsst.ccs.bus.data.ConfigurationInfo;
import static org.lsst.ccs.subsystems.fcs.FCSCst.LOADER_CARRIER_NAME;
import static org.lsst.ccs.subsystems.fcs.FCSCst.LOADER_NAME;
import static org.lsst.ccs.subsystems.fcs.FCSCst.UNKNOWN;
import org.lsst.ccs.subsystems.fcs.StatusDataPublishedByEPOSController;
import org.lsst.ccs.subsystems.fcs.StatusDataPublishedByLoaderCarrier;
import org.lsst.ccs.subsystems.fcs.StatusDataPublishedByLoaderClamp;
import static org.lsst.ccs.subsystems.fcs.loader.ui.PLCLoaderPanel.AF0;
import static org.lsst.ccs.subsystems.fcs.loader.ui.PLCLoaderPanel.AF0B;
import static org.lsst.ccs.subsystems.fcs.loader.ui.PLCLoaderPanel.AF3;
import static org.lsst.ccs.subsystems.fcs.loader.ui.PLCLoaderPanel.AF3B;
import static org.lsst.ccs.subsystems.fcs.loader.ui.PLCLoaderPanel.AP2;
import static org.lsst.ccs.subsystems.fcs.loader.ui.PLCLoaderPanel.AP2B;
import static org.lsst.ccs.subsystems.fcs.loader.ui.PLCLoaderPanel.DEFAULT;
import static org.lsst.ccs.subsystems.fcs.loader.ui.PLCLoaderPanel.DOCKED;
import static org.lsst.ccs.subsystems.fcs.loader.ui.PLCLoaderPanel.DOCKEDB;
import static org.lsst.ccs.subsystems.fcs.loader.ui.PLCLoaderPanel.FILTERPRESENCE;
import static org.lsst.ccs.subsystems.fcs.loader.ui.PLCLoaderPanel.FILTERPRESENCEB;
import static org.lsst.ccs.subsystems.fcs.loader.ui.PLCLoaderPanel.KENG;
import static org.lsst.ccs.subsystems.fcs.loader.ui.PLCLoaderPanel.KENGB;
import static org.lsst.ccs.subsystems.fcs.loader.ui.PLCLoaderPanel.KLOCK;
import static org.lsst.ccs.subsystems.fcs.loader.ui.PLCLoaderPanel.KLOCKB;
import static org.lsst.ccs.subsystems.fcs.loader.ui.PLCLoaderPanel.RELAYCARRIER;
import static org.lsst.ccs.subsystems.fcs.loader.ui.PLCLoaderPanel.RELAYHOOKS;
import static org.lsst.ccs.subsystems.fcs.loader.ui.PLCLoaderPanel.SAFETYBELT;
import org.lsst.ccs.subsystems.fcs.ui.commons.DiamondSwitch;
import org.lsst.ccs.subsystems.fcs.ui.commons.DigitalSwitch;
import org.lsst.ccs.subsystems.fcs.ui.commons.Tools;
import static org.lsst.ccs.subsystems.fcs.ui.commons.Tools.TIRETS;
import static org.lsst.ccs.subsystems.fcs.ui.commons.Tools.ZERO_VALUE;
import static org.lsst.ccs.subsystems.fcs.ui.commons.Tools.greenColor;
import static org.lsst.ccs.subsystems.fcs.ui.commons.Tools.insets_big;
import static org.lsst.ccs.subsystems.fcs.ui.commons.Tools.insets_std;

/**
 * A view to display a global view of the loader.
 *
 * @author virieux
 */
public class LoaderGeneralViewPanel extends JPanel {

    private static final long serialVersionUID = 8834267363875256916L;
    private InterfaceLoaderGUI subs;

    private final JLabel loaderDockedLabel = new JLabel("LOADER DOCKED");
    private final JPanel fakeSensorsPanel = new JPanel();
    private final JPanel controlPanel = new JPanel();
    private final String fullPathCarrierName = LOADER_NAME + "/" + LOADER_CARRIER_NAME;

    public static final String AF1 = "acAF1s";
    public static final String AF1B = "acAF1b";

    /* camera protection panel */
    private final JPanel cpsPanel = new JPanel();
    private final JPanel keysPanel = new JPanel();
    private static final DigitalSwitch DOCKEDSWITCH = new DigitalSwitch(DOCKED, DOCKEDB);
    private static final DigitalSwitch AP2SWITCH = new DigitalSwitch(AP2, AP2B);
    private static final DigitalSwitch AF0SWITCH = new DigitalSwitch(AF0, AF0B);
    private static final DigitalSwitch AF1SWITCH = new DigitalSwitch(AF1, AF1B);
    private static final DigitalSwitch AF3SWITCH = new DigitalSwitch(AF3, AF3B);
    private static final DigitalSwitch NOLOCKOUTSWITCH = new DigitalSwitch(KLOCKB, KLOCK);
    private static final DigitalSwitch ENGINEERINGSWITCH = new DigitalSwitch(KENG, KENGB);
    private final DiamondSwitch carrierRelaySwitch = new DiamondSwitch(RELAYCARRIER);
    private final DiamondSwitch hooksRelaySwitch = new DiamondSwitch(RELAYHOOKS);
    private final DiamondSwitch safetyBeltPresenceSwitch = new DiamondSwitch(SAFETYBELT);
    private final DiamondSwitch noDefaultStatusSwitch = new DiamondSwitch(DEFAULT);


    /* Clamp Panel*/
    private final JPanel clampPanel = new JPanel();
    private static final DigitalSwitch FILTERPRESENCESWITCH = new DigitalSwitch(FILTERPRESENCE, FILTERPRESENCEB);
    private final JLabel clampHomingLabel = new JLabel();
    private final JLabel clampStateLabel = new JLabel();
    private final JLabel clampPositionLabel = new JLabel();
    private final JLabel forceStatusLabel = new JLabel();
    private final DigitalSwitch clampControllerSwitch = new DigitalSwitch();

    /* Carrier*/
    private final JPanel carrierPanel = new JPanel();
    private final JLabel carrierPositionLabel = new JLabel();
    private final JLabel carrierStatusLabel = new JLabel();
    private final JLabel carrierSpeedLabel = new JLabel();
    private final JProgressBar carrierPositionProgressBar = new JProgressBar(1, 0, 100);
    private final DigitalSwitch carrierControllerSwitch = new DigitalSwitch();

    //control panel
    private final JButton loadFilterButton = new JButton("loadFilterIntoBenchOrStorageBox");
    private final JButton unloadFilterButton = new JButton("unloadFilterFromBenchOrStorageBox");
    private final JButton moveFilterStoHButton = new JButton("moveFilterToHandoff");
    private final JButton moveFilterHtoSButton = new JButton("moveFilterToStorage");
    private final JButton openClampAndMoveEmptyToSButton = new JButton("openClampAndMoveEmptyToStorage");
    private final JButton moveEmptyToHandCloseButton = new JButton("moveEmptyToHandoffAndClose");

    private final DigitalSwitch[] complementarySwitches = new DigitalSwitch[] {AF0SWITCH,
            AF3SWITCH, AP2SWITCH, AF1SWITCH, ENGINEERINGSWITCH,
            NOLOCKOUTSWITCH
    };

    private final DigitalSwitch[] redondantSwitches = new DigitalSwitch[] {
            DOCKEDSWITCH, FILTERPRESENCESWITCH
    };

    private final DiamondSwitch[] diamondSwitchesGreenGray = new DiamondSwitch[] {
        carrierRelaySwitch, hooksRelaySwitch,
        safetyBeltPresenceSwitch,
    };
    // End of variables declaration


    /**
     * A Runnable to update this panel from data published on the status bus
     * from the Loader Clamp.
     */
    private class UpdateClamp implements Runnable {

        private final StatusDataPublishedByLoaderClamp s;

        public UpdateClamp(StatusDataPublishedByLoaderClamp status) {
            this.s = status;
        }

        @Override
        public void run() {

            if (s.isHomingDone()) {
                clampHomingLabel.setText("DONE");
            } else {
                clampHomingLabel.setText(TIRETS);
            }
            Color clampStateColor = s.getClampState().getColor();
            clampStateLabel.setText(s.getClampState().getText());
            clampStateLabel.setForeground(clampStateColor);
            forceStatusLabel.setText(s.getForceStatus().toString());
            //controller status
            clampControllerSwitch.setColor(s.isControllerInError() ? Color.RED : greenColor);

        }
    }

    /**
     * A Runnable to update this panel from carrier data
     */
    private class UpdateCarrier implements Runnable {

        private final StatusDataPublishedByLoaderCarrier s;

        public UpdateCarrier(StatusDataPublishedByLoaderCarrier s) {
            this.s = s;
        }

        @Override
        public void run() {

            if (s.isAtStorage()) {
                carrierStatusLabel.setText("STORAGE");

            } else if (s.isAtEngaged()) {
                carrierStatusLabel.setText("ENGAGED");

            } else if (s.isAtHandoff()) {
                carrierStatusLabel.setText("HANDOFF");

            } else {
                carrierStatusLabel.setText("IN_TRAVEL");
            }
            //controller status
            carrierControllerSwitch.setColor(s.isControllerInError() ? Color.RED : greenColor);

        }

    }

    private class UpdateFromSensorList implements Runnable {

        private final Map<String, Serializable> sensorMap;

        public UpdateFromSensorList(Map<String, Serializable> sensorMap) {
            this.sensorMap = sensorMap;
        }

        @Override
        public void run() {

            /* redondant sensors : value 0 (GRAY) or 1 (GREEN) but can be in error (RED) */
            for (DigitalSwitch ds : redondantSwitches) {
                ds.updateColorRedondantSensor(sensorMap);
            }

            /* ComplementarySensors*/
            for (DigitalSwitch ds : complementarySwitches) {
                ds.updateColorComplementary(sensorMap);
            }

            /* DigitalSensor for status signals : can only be 0 or 1 */
            for (DiamondSwitch ds : diamondSwitchesGreenGray) {
                ds.updateColorGreenGray(sensorMap);
            }

            noDefaultStatusSwitch.updateColorDefaultRedGreen(sensorMap);
        }

    }

    /**
     * A Runnable to update controller panel with data published by
     * loaderCarrier controller.
     */
    public class UpdateCarrierController implements Runnable {

        private final StatusDataPublishedByEPOSController s;

        private UpdateCarrierController(StatusDataPublishedByEPOSController s) {
            this.s = s;
        }

        @Override
        public void run() {
            if (s.isBooted() && s.getState() != null) {
                carrierPositionLabel.setText(Integer.toString(s.getPosition()));
                carrierSpeedLabel.setText(Integer.toString(s.getVelocity()));
                /* - : to give the illusion of the real motion of carrier*/
                carrierPositionProgressBar.setValue(-s.getPosition());
                carrierPositionProgressBar.setForeground(Color.PINK);
            }
        }
    }

    /**
     * A Runnable to update controller panel with data published by loaderClamp
     * controller.
     */
    public class UpdateClampController implements Runnable {

        private final StatusDataPublishedByEPOSController s;

        private UpdateClampController(StatusDataPublishedByEPOSController s) {
            this.s = s;
        }

        @Override
        public void run() {
            if (s.isBooted() && s.getState() != null) {
                clampPositionLabel.setText(Integer.toString(s.getPosition()));
            }
        }
    }

    /**
     * A constructor without any argument for tests.
     */
    public LoaderGeneralViewPanel() {
        initComponents();
    }

    /**
     * A constructor with a LoaderGUI.
     *
     * @param subs
     */
    public LoaderGeneralViewPanel(InterfaceLoaderGUI subs) {
        this.subs = subs;
        initComponents();
    }

    public void setSubsystem(InterfaceLoaderGUI subs) {
        this.subs = subs;
    }

    /**
     * Initialize this Panel from Configuration Info data. TODO : initialize
     * timeouts for commands loUnloadFilter and loLoadFilter ?
     *
     * @param configInfo
     */
    public void initializeGui(ConfigurationInfo configInfo) {
        SwingUtilities.invokeLater(new GuiInitialization(configInfo));
    }


    /**
     * To update this Panel when the loader clamp publishes on the bus.
     *
     * @param status
     */
    public void updateClamp(StatusDataPublishedByLoaderClamp status) {
        SwingUtilities.invokeLater(new UpdateClamp(status));
    }

    /**
     * to update this panel from carrier data
     *
     * @param status
     */
    public void updateCarrier(StatusDataPublishedByLoaderCarrier status) {
        SwingUtilities.invokeLater(new UpdateCarrier(status));
    }

    public void updateFromSensorList(Map<String, Serializable> sensorMap) {
        SwingUtilities.invokeLater(new UpdateFromSensorList(sensorMap));
    }

    public void updateCarrierController(StatusDataPublishedByEPOSController status) {
        SwingUtilities.invokeLater(new UpdateCarrierController(status));
    }

    public void updateClampController(StatusDataPublishedByEPOSController status) {
        SwingUtilities.invokeLater(new UpdateClampController(status));
    }

    /**
     * A Runnable to initialize this Panel from Configuration data.
     *
     */
    private class GuiInitialization implements Runnable {

        private final ConfigurationInfo configInfo;

        public GuiInitialization(ConfigurationInfo configInfo) {
            this.configInfo = configInfo;
        }

        @Override
        public void run() {
            //Retrieve the configuration data for my name.
            Map<String, String> configForCarrier = configInfo.getCurrentValuesFor(fullPathCarrierName);
            int handoffPosition = Tools.toInt(configForCarrier, "handoffPosition");
            int storagePosition = Tools.toInt(configForCarrier, "storagePosition");
            carrierPositionProgressBar.setMinimum(-handoffPosition);
            carrierPositionProgressBar.setMaximum(-storagePosition);
            carrierPositionProgressBar.setValue(0);
        }
    }


    /**
     * To initialize all the components of this Panel.
     */
    // <editor-fold defaultstate="collapsed" desc="Form Description">
    private void initComponents() {
        setBorder(BorderFactory.createTitledBorder(BorderFactory.createLineBorder(new Color(255, 255, 0)), "LOADER"));
        setForeground(new Color(204, 204, 255));
        setLayout(new GridBagLayout());

        /**
         * ** loader clamp Panel **
         */
        JPanel clampControllerPanel = new JPanel();
        clampControllerPanel.setBorder(BorderFactory.createEtchedBorder());
        clampControllerPanel.setLayout(new GridBagLayout());
        GridBagConstraints gbc = new GridBagConstraints();
        gbc.insets = insets_std;
        gbc.gridx = 0;
        gbc.gridy = 0;
        gbc.anchor = GridBagConstraints.CENTER;
        clampControllerPanel.add(new JLabel("Controller Status"), gbc);
        gbc.gridy++;
        clampControllerPanel.add(clampControllerSwitch, gbc);
        clampControllerSwitch.setSize(40);

        JPanel filterPresencePanel = new JPanel();
        filterPresencePanel.setBorder(BorderFactory.createEtchedBorder());
        filterPresencePanel.setLayout(new GridBagLayout());
        gbc = new GridBagConstraints();
        gbc.insets = insets_std;
        gbc.gridx = 0;
        gbc.gridy = 0;
        gbc.anchor = GridBagConstraints.CENTER;
        filterPresencePanel.add(new JLabel("Filter Presence"), gbc);
        gbc.gridy++;
        filterPresencePanel.add(FILTERPRESENCESWITCH, gbc);
        FILTERPRESENCESWITCH.setSize(40);

        clampPanel.setBorder(BorderFactory.createTitledBorder("Loader Hooks"));
        clampPanel.setLayout(new GridBagLayout());
        gbc = new GridBagConstraints();
        gbc.insets = insets_std;
        gbc.gridx = 0;
        gbc.gridy = 0;
        gbc.anchor = GridBagConstraints.LINE_START;
        clampPanel.add(new JLabel("Homing:"), gbc);
        gbc.gridx++;
        gbc.anchor = GridBagConstraints.LINE_END;
        clampPanel.add(clampHomingLabel, gbc);
        gbc.gridx--;

        gbc.gridy++;
        gbc.anchor = GridBagConstraints.LINE_START;
        clampPanel.add(clampControllerPanel, gbc);
        gbc.gridx++;
        clampPanel.add(filterPresencePanel, gbc);
        gbc.gridx--;
        gbc.gridy++;
        gbc.anchor = GridBagConstraints.LINE_START;
        int y = gbc.gridy;
        clampPanel.add(new JLabel("Hooks Status"), gbc);
        gbc.gridy++;
        clampPanel.add(new JLabel("Hooks Position"), gbc);
        gbc.gridy++;
        clampPanel.add(new JLabel("Clamped Force Status"), gbc);
        gbc.gridy = y;
        gbc.gridx++;
        gbc.anchor = GridBagConstraints.CENTER;
        clampPanel.add(clampStateLabel, gbc);
        gbc.gridy++;
        gbc.anchor = GridBagConstraints.LINE_END;
        clampPanel.add(clampPositionLabel, gbc);
        gbc.gridy++;
        gbc.anchor = GridBagConstraints.CENTER;
        clampPanel.add(forceStatusLabel, gbc);


        /**
         * ** end of loader clamp Panel **
         */

        /**
         * ** loader carrier Panel **
         */
        JPanel controllerPanel = new JPanel();
        controllerPanel.setBorder(BorderFactory.createEtchedBorder());
        controllerPanel.setLayout(new GridBagLayout());

        gbc = new GridBagConstraints();
        gbc.insets = insets_std;
        gbc.gridx = 0;
        gbc.gridy = 0;
        gbc.anchor = GridBagConstraints.CENTER;
        controllerPanel.add(new JLabel("Controller Status"), gbc);
        gbc.gridy++;
        controllerPanel.add(carrierControllerSwitch, gbc);
        carrierControllerSwitch.setSize(50);

        carrierPanel.setBorder(BorderFactory.createTitledBorder("Loader Carrier"));
        carrierPanel.setLayout(new GridBagLayout());
        gbc = new GridBagConstraints();
        gbc.insets = insets_std;
        gbc.gridx = 0;
        gbc.gridy = 0;
        gbc.anchor = GridBagConstraints.LINE_START;
        carrierPanel.add(new JLabel("Carrier Position"), gbc);
        gbc.gridy++;
        carrierPanel.add(new JLabel("Carrier Status"), gbc);
        gbc.gridy++;
        carrierPanel.add(new JLabel("Carrier Speed"), gbc);
        gbc.gridy++;
        int y2 = gbc.gridy;
        carrierPanel.add(controllerPanel, gbc);

        gbc.gridy = 0;
        gbc.gridx++;
        gbc.anchor = GridBagConstraints.LINE_END;
        carrierPanel.add(carrierPositionLabel, gbc);
        gbc.gridy++;
        carrierPanel.add(carrierStatusLabel, gbc);
        gbc.gridy++;
        carrierPanel.add(carrierSpeedLabel, gbc);

        gbc.gridy = y2;
        gbc.gridx = 1;
        gbc.gridheight = 2;
        gbc.anchor = GridBagConstraints.CENTER;
        carrierPositionProgressBar.setBorderPainted(true);
        carrierPositionProgressBar.setBackground(greenColor);
        carrierPositionProgressBar.setStringPainted(true);
        carrierPanel.add(carrierPositionProgressBar, gbc);

        /**
         * ** end of loader carrier Panel **
         */

        /**
         ******************************************************************************
         ** Control Panel
         * *****************************************************************************
         */
        loadFilterButton.addActionListener((ActionEvent evt) -> {
            loadFilterActionPerformed(evt);
        });

        unloadFilterButton.addActionListener((ActionEvent evt) -> {
            unloadFilterActionPerformed(evt);
        });

        moveFilterStoHButton.addActionListener((ActionEvent evt) -> {
            moveFilterStoHActionPerformed(evt);
        });

        moveFilterHtoSButton.addActionListener((ActionEvent evt) -> {
            moveFilterHtoSActionPerformed(evt);
        });

        openClampAndMoveEmptyToSButton.addActionListener((ActionEvent evt) -> {
            openClampAndMoveEmptyToStoSActionPerformed(evt);
        });

        moveEmptyToHandCloseButton.addActionListener((ActionEvent evt) -> {
            moveEmptyToHandCloseActionPerformed(evt);
        });

        controlPanel.setLayout(new GridBagLayout());
        GridBagConstraints gbc0 = new GridBagConstraints();
        gbc0.insets = insets_std;
        gbc0.gridx = 0;
        gbc0.gridy = 0;
        gbc0.fill = GridBagConstraints.HORIZONTAL;
        controlPanel.add(loadFilterButton, gbc0);
        gbc0.gridy++;
        controlPanel.add(unloadFilterButton, gbc0);
        gbc0.gridy++;
        controlPanel.add(moveFilterStoHButton, gbc0);
        gbc0.gridy++;
        controlPanel.add(moveFilterHtoSButton, gbc0);
        gbc0.gridy++;
        controlPanel.add(openClampAndMoveEmptyToSButton, gbc0);
        gbc0.gridy++;
        controlPanel.add(moveEmptyToHandCloseButton, gbc0);
        //end of Control Panel

        //Fake Sensors Panel
        fakeSensorsPanel.setBorder(BorderFactory.createLineBorder(Color.BLUE, WIDTH + 5));
        fakeSensorsPanel.setLayout(new GridBagLayout());
        GridBagConstraints gbc1 = new GridBagConstraints();
        gbc1.insets = insets_std;
        gbc1.gridx = 0;
        gbc1.gridy = 0;
        gbc1.fill = GridBagConstraints.HORIZONTAL;
        fakeSensorsPanel.add(loaderDockedLabel, gbc1);

        gbc1.gridx++;
        fakeSensorsPanel.add(DOCKEDSWITCH, gbc1);
        //end Fake Sensors Panel

        //CameraProtectionPanel with status coming from autochanger
        cpsPanel.setBorder(BorderFactory.createLineBorder(Color.MAGENTA, WIDTH + 5));
        cpsPanel.setLayout(new GridBagLayout());
        gbc0 = new GridBagConstraints();
        gbc0.gridx = 0;
        gbc0.gridy = 0;
        gbc0.anchor = GridBagConstraints.WEST;
        gbc0.insets = insets_std;
        cpsPanel.add(new JLabel("AP2 (AC at handoff)"), gbc0);
        gbc0.gridy++;
        cpsPanel.add(new JLabel("AF0 (AC latches opened)"), gbc0);
        gbc0.gridy++;
        cpsPanel.add(new JLabel("AF1 (filter detected in AC)"), gbc0);
        gbc0.gridy++;
        cpsPanel.add(new JLabel("AF3 (Filter Engaged and Latches Closed)"), gbc0);

        gbc0.gridx = 1;
        gbc0.gridy = 0;
        gbc0.anchor = GridBagConstraints.EAST;
        cpsPanel.add(AP2SWITCH, gbc0);
        gbc0.gridy++;
        cpsPanel.add(AF0SWITCH, gbc0);
        gbc0.gridy++;
        cpsPanel.add(AF1SWITCH, gbc0);
        gbc0.gridy++;
        cpsPanel.add(AF3SWITCH, gbc0);

        //keys Panel : key lock/permit, engineering key ...
        keysPanel.setBorder(BorderFactory.createLineBorder(Color.MAGENTA, WIDTH + 5));
        keysPanel.setLayout(new GridBagLayout());
        gbc0 = new GridBagConstraints();
        gbc0.gridx = 0;
        gbc0.gridy = 0;
        gbc0.anchor = GridBagConstraints.WEST;
        gbc0.insets = insets_std;
        keysPanel.add(new JLabel("No Lockout"), gbc0);
        gbc0.gridy++;
        keysPanel.add(new JLabel("Engineering key"), gbc0);
        gbc0.gridy++;
        keysPanel.add(new JLabel("Carrier relay"), gbc0);
        gbc0.gridy++;
        keysPanel.add(new JLabel("Hooks relay"), gbc0);
        gbc0.gridy++;
        keysPanel.add(new JLabel("Safety belt presence"), gbc0);
        gbc0.gridy++;
        keysPanel.add(new JLabel("No Default Status"), gbc0);

        gbc0.gridx = 1;
        gbc0.gridy++;
        gbc0.gridy = 0;
        gbc0.anchor = GridBagConstraints.EAST;
        keysPanel.add(NOLOCKOUTSWITCH, gbc0);
        gbc0.gridy++;
        keysPanel.add(ENGINEERINGSWITCH, gbc0);
        gbc0.gridy++;
        keysPanel.add(carrierRelaySwitch, gbc0);
        gbc0.gridy++;
        keysPanel.add(hooksRelaySwitch, gbc0);
        gbc0.gridy++;
        keysPanel.add(safetyBeltPresenceSwitch, gbc0);
        gbc0.gridy++;
        keysPanel.add(noDefaultStatusSwitch, gbc0);

        //Whole Panel
        gbc = new GridBagConstraints();
        setBorder(Tools.getGeneralPanelTitle("Loader Overview"));
        setLayout(new GridBagLayout());
        gbc.anchor = GridBagConstraints.BASELINE_LEADING;
        gbc.insets = insets_big;
        gbc.gridx = 0;
        gbc.gridy = 0;
        add(fakeSensorsPanel, gbc);
        gbc.gridy++;
        add(cpsPanel, gbc);
        gbc.gridx++;
        add(keysPanel, gbc);

        gbc.gridx--;
        gbc.gridy++;
        add(clampPanel, gbc);
        gbc.gridx++;
        add(carrierPanel, gbc);
        gbc.gridx--;
        gbc.gridy++;
        add(controlPanel, gbc);


        setDefaultValues();
    }

    private void setDefaultValues() {
        DOCKEDSWITCH.setColor(Color.gray);
        AP2SWITCH.setColor(Color.gray);
        AF0SWITCH.setColor(Color.gray);
        AF1SWITCH.setColor(Color.gray);
        AF3SWITCH.setColor(Color.gray);
        NOLOCKOUTSWITCH.setColor(Color.gray);
        ENGINEERINGSWITCH.setColor(Color.gray);
        carrierRelaySwitch.setColor(Color.gray);
        hooksRelaySwitch.setColor(Color.gray);
        safetyBeltPresenceSwitch.setColor(Color.gray);
        noDefaultStatusSwitch.setColor(Color.gray);
        clampHomingLabel.setText("NOT DONE");
        clampHomingLabel.setForeground(greenColor);
        FILTERPRESENCESWITCH.setColor(Color.gray);
        clampControllerSwitch.setColor(Color.gray);
        clampStateLabel.setText(UNKNOWN);
        clampPositionLabel.setText(ZERO_VALUE);
        forceStatusLabel.setText(UNKNOWN);
        carrierPositionLabel.setText(ZERO_VALUE);
        carrierStatusLabel.setText(UNKNOWN);
        carrierControllerSwitch.setColor(Color.gray);
    }

    /**
     * Reset components to default values.
     */
    public void resetPanel() {
        setDefaultValues();
    }

    private void loadFilterActionPerformed(ActionEvent evt) {
        subs.loLoadFilterIntoBenchOrStorageBox();
    }

    private void unloadFilterActionPerformed(ActionEvent evt) {
        subs.loUnloadFilterFromBenchOrStorageBox();
    }

    private void moveFilterStoHActionPerformed(ActionEvent evt) {
        subs.loMoveFilterStoH();
    }

    private void moveFilterHtoSActionPerformed(ActionEvent evt) {
        subs.loMoveFilterHtoS();
    }

    private void moveEmptyToHandCloseActionPerformed(ActionEvent evt) {
        subs.loMoveEmptyToHandClose();
    }

    private void openClampAndMoveEmptyToStoSActionPerformed(ActionEvent evt) {
        subs.loOpenClampAndMoveEmptyToS();
    }


    /**
     * To be displayed on the GUI.
     *
     * @return
     */
    @Override
    public String toString() {
        return "Overview";
    }

    /**
     * Just for tests.
     *
     * @param args
     */
    public static void main(String[] args) {
        LoaderGeneralViewPanel d = new LoaderGeneralViewPanel();
        JFrame frame = new JFrame("LoaderGeneralViewPanel");
        frame.setContentPane(d);
        frame.pack();
        frame.setVisible(true);
        frame.setDefaultCloseOperation(JFrame.EXIT_ON_CLOSE);
    }
}
