package org.lsst.ccs.subsystems.fcs.ui.commons;

import java.awt.Color;
import java.awt.Dimension;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import javax.swing.BorderFactory;
import javax.swing.JFrame;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.JTable;
import javax.swing.SwingUtilities;
import javax.swing.table.DefaultTableModel;
import javax.swing.table.TableColumn;
import javax.swing.table.TableRowSorter;
import org.lsst.ccs.bus.data.ConfigurationInfo;
import org.lsst.ccs.bus.data.KeyValueData;
import org.lsst.ccs.bus.data.KeyValueDataList;
import static org.lsst.ccs.subsystems.fcs.FCSCst.PLUTOGATEWAY_NB_BYTES;
import org.lsst.ccs.subsystems.fcs.StatusDataPublishedByPlutoGateway;
import static org.lsst.ccs.subsystems.fcs.ui.commons.GatherPanel.FCSLOG;
import static org.lsst.ccs.subsystems.fcs.ui.commons.Tools.NICE_PINK;
import static org.lsst.ccs.subsystems.fcs.ui.commons.Tools.insets_std;

/**
 * A Panel to display data from a PlutoGateway.
 *
 * @author virieux
 */
public class PlutoGatewayPanel extends JPanel {
    private static final long serialVersionUID = -7222037685600465352L;
    
    private static final Dimension PREFERED_SIZE = new Dimension(400,300);

    /**
     * Variables declaration *
     */
    private final String myName;
    private final String fullPathGatewayName;
    private final String gatewayName;
    
    //general informations panel
    private final CanOpenDevicePanel  generalInfoPanel;
    private final JPanel valuesPanel;
    JScrollPane jScrollPane = new JScrollPane();
    JTable valuesTable = new JTable();
    
    //my sensors list
    private final JPanel sensorsPanel;
    private final JTable sensorsTable = new JTable();
    private final JScrollPane sensorsJScrollPane  = new JScrollPane();
    /* list of short names of sensors plugged on my gateway */
    protected List<String> sensorNames;
    

    /**
     * A Runnable to update this panel from data published on the STATUS bus
     * by the plutoGateway.
     */
    class UpdatePlutoGateway implements Runnable {

        private StatusDataPublishedByPlutoGateway s;

        public UpdatePlutoGateway(StatusDataPublishedByPlutoGateway status) {
            this.s = status;
        }

        @Override
        public void run() {
            FCSLOG.fine(" UpdatePlutoGateway for gateway=" + fullPathGatewayName);
            //General Informations Panel
            generalInfoPanel.updateCanOpenDevice(fullPathGatewayName, s);
            
            /* values panel*/
            DefaultTableModel valuesTableModel = (DefaultTableModel) valuesTable.getModel();
            int rowNB = s.getIntValues().length;
            for (int i = 0; i < rowNB; i++) {
                int value = s.getIntValues()[i];
                String bin = String.format("%016d", Long.parseLong(Integer.toBinaryString(value)));
                valuesTableModel.setValueAt(value, i, 1);
                valuesTableModel.setValueAt("  0b" + bin, i, 2);
            }
            valuesTableModel.fireTableDataChanged();
        }
    }

    /**
     * A Runnable to update this panel from data published on the STATUS bus by
     * a sensor.
     */
    class UpdateSensor implements Runnable {

        private KeyValueDataList kvdl;

        public UpdateSensor(KeyValueDataList kvdl) {
            this.kvdl = kvdl;
        }

        @Override
        public void run() {
            for (KeyValueData data : kvdl) {
                String sensorName = data.getKey();
                DefaultTableModel sensorsTableModel = (DefaultTableModel) sensorsTable.getModel();
                int rowNB = sensorsTableModel.getRowCount();
                for (int i = 0; i < rowNB; i++) {
                    if ((sensorsTableModel.getValueAt(i, 0)).equals(sensorName) && !sensorName.contains("inclinometer")) {
                        //Inclinometer publishes a double (inclinaison) not an int (voltage)
                        sensorsTableModel.setValueAt((int)data.getValue(), i, 3);
                    } else if ((sensorsTableModel.getValueAt(i, 0)).equals(sensorName) && sensorName.contains("inclinometer")) {
                        sensorsTableModel.setValueAt((double)data.getValue(), i, 3);
                    }
                }
                sensorsTableModel.fireTableDataChanged();
            }
        }
    }

    /**
     * Creates new form PlutoGatewayPanel with a name and a general informations panel.
     * @param fullPathGatewayName
     */
    public PlutoGatewayPanel(String fullPathGatewayName) {
        this.fullPathGatewayName = fullPathGatewayName;
        this.generalInfoPanel = new CanOpenDevicePanel(fullPathGatewayName);
        gatewayName = Tools.getShortComponentName(fullPathGatewayName);
        myName = gatewayName + "Panel";
        valuesPanel = new JPanel();
        sensorsPanel = new JPanel();
        
        initComponents();
    }
    
    public void setSensorNames(List<String> sensorNamesList) {
        sensorNames = new ArrayList<>();
        for (String fullPathSensorName : sensorNamesList) {
            sensorNames.add(Tools.getShortComponentName(fullPathSensorName));
        }
    }

    
    /**
     * Initialize the PLutoGatewayPanel from configuration info.
     * @param configInfo 
     */
    public void initializeGui(ConfigurationInfo configInfo) {
        generalInfoPanel.initializeGui(configInfo);
        /* values panel*/
        DefaultTableModel valuesTableModel = (DefaultTableModel) valuesTable.getModel();
        /* loop on byteNumero*/
        for (int i = 0; i < PLUTOGATEWAY_NB_BYTES; i++) {
            Object[] tab = {i, 0, 0};
            valuesTableModel.addRow(tab);
        }
        valuesTableModel.fireTableDataChanged();
    }

    /**
     * Initialize sensors table from a list of names of sensors which are
     * plugged on this plutoGateway.
     *
     * The names of sensors are full path names.
     *
     * All sensors implement SensorPluggedOnDevice interface. A sensor can be an
     * instance of DigitalSensor, ForceSensor, Inclinometer.
     *
     * @param configInfo
     * @param sensorList
     */
    public void initializeGuiWithSensorsList(ConfigurationInfo configInfo, List<String> sensorList) {
        
        setSensorNames(sensorList);

        DefaultTableModel sensorsTableModel = (DefaultTableModel) sensorsTable.getModel();
        for (String fullPathSensorName : sensorList) {
            FCSLOG.info("retrieving configInfo for " + fullPathSensorName);
            Map<String, String> sensorConfig = configInfo.getCurrentValuesFor(fullPathSensorName);
            String sensorName = Tools.getShortComponentName(fullPathSensorName);
            int byteNumero = Integer.parseInt(sensorConfig.get("byteNumero"));
            String inputNumeroS = sensorConfig.get("inputNumero");
            /* sensors which are coded on one byte, as ForceSensor, Inclinometer, don't have a parameter inputNumero.*/
            if (inputNumeroS == null) {
                Object[] tab = {sensorName, byteNumero};
                sensorsTableModel.addRow(tab);
            } else {
                Object[] tab = {sensorName, byteNumero, Integer.parseInt(inputNumeroS)};
                sensorsTableModel.addRow(tab);
            }
        }
        sensorsTableModel.fireTableDataChanged();
    }
    
    /**
     * updates PlutoGatewayPanel from data published by a plutoGateway.
     * @param deviceToUpdateName
     * @param status 
     */
    public void updatePlutoGateway(String deviceToUpdateName, StatusDataPublishedByPlutoGateway status) {
        if (gatewayName.equals(deviceToUpdateName)) {
            generalInfoPanel.updateCanOpenDevice(deviceToUpdateName, status);
            SwingUtilities.invokeLater(new UpdatePlutoGateway(status));
        }
    }

    /**
     * Update PlutoGateway panel from data published by a sensor if this sensor
     * is plugged on my plutoGateway.
     *
     * @param kvdl
     */
    public void updateSensor(KeyValueDataList kvdl) {
        SwingUtilities.invokeLater(new UpdateSensor(kvdl));
    }
    

    @SuppressWarnings("unchecked")
    // <editor-fold defaultstate="collapsed" desc="Form description">                          
    private void initComponents() {
        
        /**
         ******************************************************************************
         ** Values Panel
         * *****************************************************************************
         */
        initValuesPanel();

        /**
         ******************************************************************************
         ** end of Values Panel
         * *****************************************************************************
         */
        initSensorsPanel();
        
        /**
         ******************************************************************************
         ** Whole Panel
         * *****************************************************************************
         */
        setBorder(BorderFactory.createLineBorder(Color.GREEN));
        //setForeground(new java.awt.Color(204, 204, 255));
        setLayout(new GridBagLayout());
        GridBagConstraints gbc = new GridBagConstraints();
        gbc.insets = insets_std;
        gbc.anchor = GridBagConstraints.CENTER;

        gbc.gridx = 0;
        gbc.gridy = 0;
        add(generalInfoPanel, gbc);
        gbc.gridy++;
        add(valuesPanel, gbc);
        
        /* add sensors panels*/
        gbc.gridy++;
        int b = 0;
        add(sensorsPanel, gbc);
                
                

        /**
         ******************************************************************************
         ** end of Whole Panel
         * *****************************************************************************
         */
    }

    public void initValuesPanel() {
        valuesTable.setModel(new DefaultTableModel(
                new Object[][]{},
                new String[]{
                    "Byte", "Decimal Value", "Binaire Value"
                }
        ) {
            private static final long serialVersionUID = 1L;
            private final Class[] types = new Class[]{
                Integer.class, Integer.class, String.class
            };
            private final boolean[] canEdit = new boolean[]{
                false, false, false
            };

            @Override
            public Class getColumnClass(int columnIndex) {
                return types[columnIndex];
            }

            @Override
            public boolean isCellEditable(int rowIndex, int columnIndex) {
                return canEdit[columnIndex];
            }
        });
        valuesTable.setName("valuesTable_" + gatewayName);
        TableColumn column;
        for (int i = 0; i < 3; i++) {
            column = valuesTable.getColumnModel().getColumn(i);
            if (i == 0) {
                column.setPreferredWidth(25); //first column is smaller
            }
            if (i == 2) {
                column.setPreferredWidth(150); //third column is bigger
            }
        }
        jScrollPane.setViewportView(valuesTable);
        jScrollPane.setPreferredSize(PREFERED_SIZE);

        valuesPanel.setBorder(BorderFactory.createLineBorder(NICE_PINK));
        valuesPanel.setLayout(new GridBagLayout());
        GridBagConstraints gbc0 = new GridBagConstraints();
        gbc0.anchor = GridBagConstraints.NORTHWEST;
        gbc0.insets = insets_std;
        gbc0.gridx = 0;
        gbc0.gridy = 0;
        valuesPanel.add(jScrollPane, gbc0);
    }
    
    public void initSensorsPanel() {
        
        DefaultTableModel sensorDefaultModel = new DefaultTableModel(
                new Object[][]{},
                new String[]{
                    "Sensor Name", "Byte", "Bit", "Value"
                }
        ) {
            private static final long serialVersionUID = 1L;
            private final Class[] types = new Class[]{
                String.class, Integer.class, Integer.class, Integer.class
            };
            private final boolean[] canEdit = new boolean[]{
                false, false, false, false
            };

            @Override
            public Class getColumnClass(int columnIndex) {
                return types[columnIndex];
            }

            @Override
            public boolean isCellEditable(int rowIndex, int columnIndex) {
                return canEdit[columnIndex];
            }
        };
        sensorsTable.setName("sensorsTable_" + gatewayName);
        sensorsTable.setModel(sensorDefaultModel);
        sensorsTable.setRowSorter(new TableRowSorter(sensorDefaultModel));
        TableColumn column;
        for (int i = 0; i < 2; i++) {
            column = sensorsTable.getColumnModel().getColumn(i);
            if (i == 0) {
                column.setPreferredWidth(PREFERED_SIZE.width * 2 / 3); //first column is bigger
            }
        }
        sensorsJScrollPane.setViewportView(sensorsTable);
        sensorsJScrollPane.setPreferredSize(PREFERED_SIZE);
        sensorsPanel.setBorder(BorderFactory.createTitledBorder("Table of sensors plugged on " + gatewayName));
        sensorsPanel.setLayout(new GridBagLayout());
        GridBagConstraints gbc = new GridBagConstraints();
        gbc.anchor = GridBagConstraints.NORTHWEST;
        gbc.insets = insets_std;
        gbc.gridx = 0;
        gbc.gridy = 0;
        JLabel jl = new JLabel("Click on the top of a column to sort this table.");
        sensorsPanel.add(jl, gbc);
        gbc.gridy++;
        sensorsPanel.add(sensorsJScrollPane, gbc);
    }
    
    public void resetPanel() {
        FCSLOG.info(myName + " resetting panels");
        generalInfoPanel.resetPanel();
        Tools.resetTable(valuesTable);
        Tools.resetTable(sensorsTable);
    }

    @Override
    public String toString() {
        if (fullPathGatewayName == null) {
            return "plutoGateway";
        } else {
            return Tools.getShortComponentName(fullPathGatewayName);
        }
    }

    public static void main(String[] argv) {
        JFrame frame = new JFrame("Pluto Gateway Panel");
        PlutoGatewayPanel p = new PlutoGatewayPanel("my_plutoGateway");
        frame.setContentPane(p);
        frame.pack();
        frame.setVisible(true);
        frame.setDefaultCloseOperation(JFrame.EXIT_ON_CLOSE);
    }

}
