
package org.lsst.ccs.subsystems.fcs.ui.commons;

import java.awt.Color;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.util.Map;
import javax.swing.BorderFactory;
import javax.swing.JFrame;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.SwingUtilities;
import org.lsst.ccs.bus.data.ConfigurationInfo;
import static org.lsst.ccs.subsystems.fcs.FCSCst.UNKNOWN;
import org.lsst.ccs.subsystems.fcs.StatusDataPublishedByHardware;
import static org.lsst.ccs.subsystems.fcs.ui.commons.Tools.BIG_FONT;
import static org.lsst.ccs.subsystems.fcs.ui.commons.Tools.getShortComponentName;
import static org.lsst.ccs.subsystems.fcs.ui.commons.Tools.insets_std;

/**
 * This Panel displays data from a CANopen hardware.
 * For example, this panel is displayed at the Top of the EPOSControllerPanel to display the general
 * information of the EPOS controller.
 * @author virieux
 */
public class CanOpenDevicePanel extends JPanel {
    private static final long serialVersionUID = -7222037685600465352L;

    /**
     * Variables declaration *
     */
    private final String deviceName;
    private final String publishedDataKey;
    //general informations panel
    private JPanel generalInfoPanel;
    private final JLabel nLabel = new JLabel("Name: ");
    private final JLabel snLabel = new JLabel("Serial number: ");
    private final JLabel nIDhexaLabel = new JLabel("CANopen nodeID (hexa): ");
    private final JLabel nIDdecLabel = new JLabel("CANopen nodeID (decimal): ");
    private final JLabel bLabel = new JLabel("booted: ");
    private final JLabel iLabel = new JLabel("initialized: ");
    private final JLabel nameLabel = new JLabel("Unknown");
    private final JLabel serialLabel = new JLabel("XXXXXXXXX");
    private final JLabel nodeIDhexaLabel = new JLabel("1A");
    private final JLabel nodeIDdecLabel = new JLabel("26");

    private final JLabel bootedLabel = new JLabel("false");
    private final JLabel initLabel = new JLabel("false");

    /**
     * A Runnable to initialize this Panel from the configuration data.
     */
    private class GuiInitialization implements Runnable {

        private final ConfigurationInfo configInfo;


        public GuiInitialization(ConfigurationInfo configInfo)
        {
            this.configInfo = configInfo;
        }

        @Override
        public void run() {
            //General Informations Panel
            Map<String, String> config = configInfo.getCurrentValuesFor(publishedDataKey);
            serialLabel.setText(config.get("serialNB"));
            String nodeID = config.get("nodeID");
            nodeIDdecLabel.setText(nodeID);
            nodeIDhexaLabel.setText("0x"+Integer.toHexString(Integer.parseInt(nodeID)));
        }
    }

    /**
     * A Runnable to update this Panel when a hardware publishes on the Status Bus.
     */
    private class UpdateCanOpenDevice implements Runnable {

        private final StatusDataPublishedByHardware s;

        public UpdateCanOpenDevice(StatusDataPublishedByHardware status) {
            this.s = status;
        }

        @Override
        public void run() {
            //General Informations Panel
            bootedLabel.setText(Boolean.toString(s.isBooted()));
            initLabel.setText(Boolean.toString(s.isInitialized()));
        }
    }

    /**
     * To create a new CanOpenDevicePanel with a device name.
     * @param deviceName
     */
    public CanOpenDevicePanel(String deviceName) {
        this.deviceName = getShortComponentName(deviceName);
        publishedDataKey = deviceName;
        initComponents();
    }

    /**
     * To initialize this panel from the data coming from the CANopen device in the configuration system.
     * @param configInfo
     */
    public void initializeGui(ConfigurationInfo configInfo) {
        SwingUtilities.invokeLater(new GuiInitialization(configInfo));
    }

    /**
     * Update the Panel from the data published on the STATUS bus by a CANopen hardware.
     * @param deviceToUpdateName
     * @param status
     */
    public void updateCanOpenDevice(String deviceToUpdateName, StatusDataPublishedByHardware status) {
        if (deviceName.equals(getShortComponentName(deviceToUpdateName))) {
            SwingUtilities.invokeLater(new UpdateCanOpenDevice(status));
        }
    }


    private void initComponents() {

        nameLabel.setText(publishedDataKey);

        /**
         ******************************************************************************
         ** General Informations Panel
         * *****************************************************************************
         */
        generalInfoPanel = new JPanel();
        generalInfoPanel.setBorder(BorderFactory.createTitledBorder("General Informations"));
        generalInfoPanel.setLayout(new GridBagLayout());
        GridBagConstraints gbc3 = new GridBagConstraints();
        gbc3.insets = insets_std;
        //first colomn
        gbc3.gridx = 0;
        gbc3.gridy = 0;
        gbc3.anchor = GridBagConstraints.LINE_START;

        generalInfoPanel.add(nLabel, gbc3);
        gbc3.gridy++;
        generalInfoPanel.add(snLabel, gbc3);
        gbc3.gridy++;
        generalInfoPanel.add(nIDhexaLabel, gbc3);
        gbc3.gridy++;
        generalInfoPanel.add(nIDdecLabel, gbc3);
        gbc3.gridy++;
        generalInfoPanel.add(bLabel, gbc3);
        gbc3.gridy++;
        generalInfoPanel.add(iLabel, gbc3);
        //second colomn
        gbc3.gridx++;
        gbc3.gridy = 0;
        gbc3.anchor = GridBagConstraints.LINE_END;
        generalInfoPanel.add(nameLabel, gbc3);
        nameLabel.setFont(BIG_FONT);
        gbc3.gridy++;
        generalInfoPanel.add(serialLabel, gbc3);
        gbc3.gridy++;
        generalInfoPanel.add(nodeIDhexaLabel, gbc3);
        gbc3.gridy++;
        generalInfoPanel.add(nodeIDdecLabel, gbc3);
        gbc3.gridy++;
        generalInfoPanel.add(bootedLabel, gbc3);
        gbc3.gridy++;
        generalInfoPanel.add(initLabel, gbc3);
        /**
         ******************************************************************************
         ** End of General Informations Panel
         * *****************************************************************************
         */

        /**
         ******************************************************************************
         ** Whole Panel
         * *****************************************************************************
         */
        setBorder(javax.swing.BorderFactory.createLineBorder(Color.GREEN));
        //setForeground(new java.awt.Color(204, 204, 255));
        setLayout(new java.awt.GridBagLayout());
        GridBagConstraints gbc = new GridBagConstraints();
        gbc.insets = insets_std;
        gbc.anchor = GridBagConstraints.NORTHWEST;

        gbc.gridx = 0;
        gbc.gridy = 0;
        add(generalInfoPanel, gbc);

        /**
         ******************************************************************************
         ** end of Whole Panel
         * *****************************************************************************
         */
    }

    public void resetPanel() {
        serialLabel.setText(UNKNOWN);
        nodeIDdecLabel.setText("0");
        nodeIDhexaLabel.setText("0x0");
        bootedLabel.setText("false");
        initLabel.setText("false");
    }

    @Override
    public String toString() {
        if (deviceName == null) {
            return "Unamed CANopen Device Panel";

        } else {
            int ind = deviceName.lastIndexOf("/");            
            return ind >=0 ? deviceName.substring(ind+1) : deviceName;
        }
    }

    public static void main(String[] argv) {
        JFrame frame = new JFrame("CANOpen Device Panel");
        CanOpenDevicePanel p = new CanOpenDevicePanel("My so cude CANopen device");
        frame.setContentPane(p);
        frame.pack();
        frame.setVisible(true);
        frame.setDefaultCloseOperation(JFrame.EXIT_ON_CLOSE);
    }

}
