
package org.lsst.ccs.subsystems.fcs.ui;

import java.awt.Color;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.event.ActionEvent;
import java.io.Serializable;
import java.util.Map;
import java.util.stream.Collectors;
import javax.swing.BorderFactory;
import javax.swing.JButton;
import javax.swing.JFrame;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JTextField;
import javax.swing.SwingUtilities;
import org.lsst.ccs.bus.data.KeyValueData;
import org.lsst.ccs.bus.data.KeyValueDataList;
import static org.lsst.ccs.subsystems.fcs.FCSCst.AC_PLC_DATA_KEY;
import static org.lsst.ccs.subsystems.fcs.FCSCst.AT_STANDBY;
import static org.lsst.ccs.subsystems.fcs.FCSCst.CA_PLC_DATA_KEY;
import static org.lsst.ccs.subsystems.fcs.FCSCst.FCSLOG;
import static org.lsst.ccs.subsystems.fcs.FCSCst.LOADER_PLC_DATA_KEY;
import static org.lsst.ccs.subsystems.fcs.FCSCst.NOT_AT_STANDBY;
import static org.lsst.ccs.subsystems.fcs.FCSCst.NO_FILTER;
import static org.lsst.ccs.subsystems.fcs.FCSCst.UNKNOWN;
import org.lsst.ccs.subsystems.fcs.FcsEnumerations;
import org.lsst.ccs.subsystems.fcs.StatusDataPublishedByAutoChangerTrucks;
import org.lsst.ccs.subsystems.fcs.StatusDataPublishedByAutochangerThreeClamps;
import org.lsst.ccs.subsystems.fcs.StatusDataPublishedByAutochangerTwoLatches;
import org.lsst.ccs.subsystems.fcs.StatusDataPublishedByCarousel;
import org.lsst.ccs.subsystems.fcs.StatusDataPublishedByEPOSController;
import org.lsst.ccs.subsystems.fcs.StatusDataPublishedByLoaderCarrier;
import org.lsst.ccs.subsystems.fcs.StatusDataPublishedByLoaderClamp;
import static org.lsst.ccs.subsystems.fcs.loader.ui.PLCLoaderPanel.FILTERPRESENCE;
import static org.lsst.ccs.subsystems.fcs.loader.ui.PLCLoaderPanel.FILTERPRESENCEB;
import org.lsst.ccs.subsystems.fcs.ui.commons.DigitalSwitch;
import static org.lsst.ccs.subsystems.fcs.ui.commons.Tools.BIG_FONT;
import static org.lsst.ccs.subsystems.fcs.ui.commons.Tools.TIRETS;
import static org.lsst.ccs.subsystems.fcs.ui.commons.Tools.UNKNOWN_STATE;
import static org.lsst.ccs.subsystems.fcs.ui.commons.Tools.ZERO_VALUE;
import static org.lsst.ccs.subsystems.fcs.ui.commons.Tools.greenColor;
import static org.lsst.ccs.subsystems.fcs.ui.commons.Tools.insets_std;

/**
 *
 * @author virieux
 */
public class FcsGeneralViewPanel extends JPanel {
    private static final long serialVersionUID = 1L;
    private static final String NONE = "NONE";

    private InterfaceFcsGUI subs;
    
    private final String title = "FCS OVERVIEW";

    
    private JPanel autochangerPanel;
    private JPanel carouselPanel;
    private JPanel loaderPanel;
    private JPanel loaderControlPanel;
    private JPanel controlPanel;
    
    /*autochanger information*/
    private final JLabel filterONLINELabel = new JLabel(NONE);
    private final JLabel filterOnTrucksLabel = new JLabel(NONE);
    private final JLabel onlineClampsStateLabel = new JLabel("UNKNOWN");
    private final JLabel latchesStateLabel = new JLabel("UNKNOWN");
    private final JLabel trucksPositionLabel = new JLabel("UNKNOWN");
    private final JLabel trucksPositionNameLabel = new JLabel("UNKNOWN");

    /* carousel information */
    private final JLabel actualPositionLabel = new JLabel();
    private final JLabel actualSpeedLabel = new JLabel();
    private final JLabel actualCurrentLabel = new JLabel();
    private final JLabel standbyLabel = new JLabel();
    private final DigitalSwitch standbySwitch = new DigitalSwitch();
    private final JLabel socketAtStandbyLabel = new JLabel();
    private final JLabel ioModuleStatusLabel = new JLabel();
    private final JLabel filterAtStandbyLabel = new JLabel();
    private final JLabel caClampsStateLabel = new JLabel();
    private final JLabel deltaPositionLabel = new JLabel();

    /* loader information */
    private final JLabel loaderClampHomingLabel = new JLabel();
    private final JLabel filterOnLoaderLabel = new JLabel();
    private final JLabel loaderClampStateLabel = new JLabel();
    private final JLabel loaderClampPositionLabel = new JLabel();
    private final JLabel loaderCarrierPositionLabel = new JLabel();
    private final JLabel loaderCarrierPositionNameLabel = new JLabel();

    /* status information */
    private final FcsCameraProtectionPanel cameraStatusPanel = new FcsCameraProtectionPanel();

    /* control panel */
    private final JButton setFilterButton = new JButton("set filter");
    private final JButton loadFilterButton = new JButton("load filter into camera");
    private final JButton unloadFilterButton = new JButton("unload filter from camera");
    private final JButton connectLoaderButton = new JButton("connect loader");
    private final JButton disconnectLoaderButton = new JButton("disconnect loader");
    private final JTextField filterField = new JTextField(2);
    private final JLabel filterLabel = new JLabel("filter ID to set:");

    /* filter ID to set */
    private int filterID;

    public FcsGeneralViewPanel() {
        initComponents();
    }

    public void setSubs(InterfaceFcsGUI subs) {
        this.subs = subs;
    }


    void updateFromStatusData(KeyValueData data) {
        if (data.getValue() instanceof StatusDataPublishedByAutochangerThreeClamps) {
            updateClamps((StatusDataPublishedByAutochangerThreeClamps) data.getValue());
        } else if (data.getValue() instanceof StatusDataPublishedByAutochangerTwoLatches) {
            updateLatches((StatusDataPublishedByAutochangerTwoLatches) data.getValue());
        } else if (data.getValue() instanceof StatusDataPublishedByAutoChangerTrucks) {
            updateTrucks((StatusDataPublishedByAutoChangerTrucks) data.getValue());
        } else if (data.getValue() instanceof StatusDataPublishedByCarousel) {
            updateCarousel((StatusDataPublishedByCarousel) data.getValue());
        } else if (data.getValue() instanceof StatusDataPublishedByLoaderClamp) {
            updateLoaderClamp((StatusDataPublishedByLoaderClamp) data.getValue());
        } else if (data.getValue() instanceof StatusDataPublishedByLoaderCarrier) {
            updateLoaderCarrier((StatusDataPublishedByLoaderCarrier) data.getValue());
        } else if (data.getValue() instanceof KeyValueDataList
                && LOADER_PLC_DATA_KEY.equals(data.getKey())) {
            cameraStatusPanel.updateLoaderFromSensorList((KeyValueDataList) data.getValue());
            updateFromSensorList((KeyValueDataList) data.getValue());
        } else if (data.getValue() instanceof KeyValueDataList
                && AC_PLC_DATA_KEY.equals(data.getKey())) {
            cameraStatusPanel.updateAutochangerFromSensorList((KeyValueDataList) data.getValue());
        } else if (data.getValue() instanceof KeyValueDataList
                && CA_PLC_DATA_KEY.equals(data.getKey())) {
            cameraStatusPanel.updateCarouselFromSensorList((KeyValueDataList) data.getValue());
        }
    }

    void updateCarousel(StatusDataPublishedByCarousel s) {
        FCSLOG.debug(getName() + " updating from carousel data:");
        SwingUtilities.invokeLater(new UpdateCarousel(s));
    }

    void updateController(StatusDataPublishedByEPOSController s) {
        FCSLOG.debug(getName() + " updating from carousel controller data:");
        SwingUtilities.invokeLater(new UpdateController(s));
    }

    void updateCarrierController(StatusDataPublishedByEPOSController s) {
        FCSLOG.debug(getName() + " updating from loader carrier controller data:");
        SwingUtilities.invokeLater(new UpdateCarrierController(s));
    }

    void updateClampController(StatusDataPublishedByEPOSController s) {
        FCSLOG.debug(getName() + " updating from loader clamp controller data:");
        SwingUtilities.invokeLater(new UpdateClampController(s));
    }

    void updateTrucks(StatusDataPublishedByAutoChangerTrucks s) {
        SwingUtilities.invokeLater(new UpdateTrucks(s));
    }

    void updateLatches(StatusDataPublishedByAutochangerTwoLatches s) {
        SwingUtilities.invokeLater(new UpdateLatches(s));
    }

    void updateClamps(StatusDataPublishedByAutochangerThreeClamps s) {
        SwingUtilities.invokeLater(new UpdateOnlineClamps(s));
    }

    void updateLoaderClamp(StatusDataPublishedByLoaderClamp s) {
        SwingUtilities.invokeLater(new UpdateLoaderClamp(s));
    }

    void updateLoaderCarrier(StatusDataPublishedByLoaderCarrier s) {
        SwingUtilities.invokeLater(new UpdateLoaderCarrier(s));
    }

    void updateFromSensorList(KeyValueDataList kvdl) {
        SwingUtilities.invokeLater(new UpdateFromSensorList(kvdl));
    }
    /**
     * A Runnable class to update carousel panel from an object
     * StatusDataPublishedByCarousel.
     */
    public class UpdateCarousel implements Runnable {

        private final StatusDataPublishedByCarousel s;

        public UpdateCarousel(StatusDataPublishedByCarousel s) {
            this.s = s;
        }

        @Override
        public void run() {
            actualPositionLabel.setText(Integer.toString(s.getPosition()));

            if (s.isAtStandby() && !s.isMoving()) {
                standbyLabel.setText(AT_STANDBY);
                standbySwitch.setColor(greenColor);
                socketAtStandbyLabel.setText(s.getSocketAtStandbyName());
                socketAtStandbyLabel.setFont(BIG_FONT);
                //if not filter at standby s.getFilterAtStandbyName() == NO_FILTER
                if (s.isEmptyAtStandby()) {
                    filterAtStandbyLabel.setText(s.getFilterAtStandbyName());
                } else {
                    filterAtStandbyLabel.setText(s.getFilterAtStandbyName() + "/" + s.getFilterAtStandbyId());
                }
                filterAtStandbyLabel.setFont(BIG_FONT);
                caClampsStateLabel.setText(s.getClampsStateAtStandby().toString());
                caClampsStateLabel.setForeground(s.getClampsStateAtStandby().getColor());
                ioModuleStatusLabel.setText(s.getIOStatusAtStandby().toString());
                ioModuleStatusLabel.setForeground(s.getIOStatusAtStandby().getColor());
                deltaPositionLabel.setText(Integer.toString(s.getDeltaPositionAtStandby()));

            } else {
                standbyLabel.setText(NOT_AT_STANDBY);
                standbySwitch.setColor(Color.orange);
                if (s.isMoving()) {
                    socketAtStandbyLabel.setText(TIRETS);
                } else {
                    //socketName could be "NO_SOCKET_AT_STANDBY" or "ERROR_READING_ID"
                    socketAtStandbyLabel.setText(s.getSocketAtStandbyName());
                }
                filterAtStandbyLabel.setText(TIRETS);
                caClampsStateLabel.setText(TIRETS);
                ioModuleStatusLabel.setText(TIRETS);
            }
        }
    }

    /**
     * A Runnable to update controller panel with data published by carousel
     * controller.
     */
    public class UpdateController implements Runnable {

        private final StatusDataPublishedByEPOSController s;

        public UpdateController(StatusDataPublishedByEPOSController s) {
            this.s = s;
        }

        @Override
        public void run() {
            if (s.isBooted() && s.getState() != null) {
                actualSpeedLabel.setText(Double.toString(s.getVelocity()));
                actualCurrentLabel.setText(Double.toString(s.getCurrent()));
            }
        }
    }

    /**
     * A Runnable to update panel with data published by carrier controller.
     */
    public class UpdateCarrierController implements Runnable {

        private final StatusDataPublishedByEPOSController s;

        public UpdateCarrierController(StatusDataPublishedByEPOSController s) {
            this.s = s;
        }

        @Override
        public void run() {
            if (s.isBooted() && s.getState() != null) {
                loaderCarrierPositionLabel.setText(Integer.toString(s.getPosition()));
                loaderCarrierPositionNameLabel.setForeground(greenColor);
            }
        }
    }

    /**
     * A Runnable to update panel with data published by clamp controller.
     */
    public class UpdateClampController implements Runnable {

        private final StatusDataPublishedByEPOSController s;

        public UpdateClampController(StatusDataPublishedByEPOSController s) {
            this.s = s;
        }

        @Override
        public void run() {
            if (s.isBooted() && s.getState() != null) {
                loaderClampPositionLabel.setText(Integer.toString(s.getPosition()));
            }
        }
    }

    /**
     * A Runnable to initialize trucks data from status data published on the
     * STATUS bus by autochanger trucks.
     */
    private class UpdateTrucks implements Runnable {

        private StatusDataPublishedByAutoChangerTrucks s;

        public UpdateTrucks(StatusDataPublishedByAutoChangerTrucks status) {
            this.s = status;
        }

        @Override
        public void run() {
            trucksPositionLabel.setText(String.valueOf(s.getDriverPosition()));
            trucksPositionNameLabel.setForeground(greenColor);
            if (s.isAtHandoff()) {
                trucksPositionNameLabel.setText("HANDOFF");
            } else if (s.isAtOnline()) {
                trucksPositionNameLabel.setText("ONLINE");
            } else if (s.isAtStandby()) {
                trucksPositionNameLabel.setText("STANDBY");
            } else {
                trucksPositionNameLabel.setText("IN_TRAVEL");
                trucksPositionNameLabel.setForeground(Color.YELLOW);
            }
        }

    }

    /**
     * A Runnable to update latches data from status data published on the
     * STATUS bus by autochanger latches.
     */
    class UpdateLatches implements Runnable {

        private StatusDataPublishedByAutochangerTwoLatches s;

        public UpdateLatches(StatusDataPublishedByAutochangerTwoLatches status) {
            this.s = status;
        }

        @Override
        public void run() {
            latchesStateLabel.setText(s.getLockStatus().toString());
            latchesStateLabel.setForeground(s.getLockStatus().getColor());
            filterOnTrucksLabel.setFont(BIG_FONT);
            if (s.getFilterPresenceStatus().equals(FcsEnumerations.FilterPresenceStatus.ENGAGED)) {
                filterOnTrucksLabel.setText(s.getFilterName() + "/" + s.getFilterId());
            } else {
                filterOnTrucksLabel.setText("No Filter");
            }
        }
    }

    /**
     * A Runnable to update ONLINE clamps data from status data published on the
     * STATUS bus by autochanger ONLINE clamps.
     */
    class UpdateOnlineClamps implements Runnable {

        private StatusDataPublishedByAutochangerThreeClamps s;

        public UpdateOnlineClamps(StatusDataPublishedByAutochangerThreeClamps status) {
            this.s = status;
        }

        @Override
        public void run() {
            onlineClampsStateLabel.setText(s.getLockStatus().toString());
            onlineClampsStateLabel.setForeground(s.getLockStatus().getColor());
        }
    }

    /**
     * A Runnable class to update loader panel from an object
     * StatusDataPublishedByLoaderClamp.
     */
    public class UpdateLoaderClamp implements Runnable {

        private final StatusDataPublishedByLoaderClamp s;

        public UpdateLoaderClamp(StatusDataPublishedByLoaderClamp s) {
            this.s = s;
        }

        @Override
        public void run() {
            /*Update of loader Panel*/
            if (s.isHomingDone()) {
                loaderClampHomingLabel.setText("DONE");
                loaderClampHomingLabel.setForeground(greenColor);
            } else {
                loaderClampHomingLabel.setText(TIRETS);
                loaderClampHomingLabel.setForeground(greenColor);
            }
            loaderClampStateLabel.setText(s.getClampState().name());
            loaderClampStateLabel.setForeground(s.getClampState().getColor());
        }
    }

    /**
     * A Runnable class to update loader panel from an object
     * StatusDataPublishedByLoaderCarrier.
     */
    public class UpdateLoaderCarrier implements Runnable {

        private final StatusDataPublishedByLoaderCarrier s;

        public UpdateLoaderCarrier(StatusDataPublishedByLoaderCarrier s) {
            this.s = s;
        }

        @Override
        public void run() {
            /*Update of loader Panel*/

            if (s.isAtStorage()) {
                loaderCarrierPositionNameLabel.setText("STORAGE");
            } else if (s.isAtHandoff()) {
                loaderCarrierPositionNameLabel.setText("HANDOFF");
            } else if (s.isAtEngaged()) {
                loaderCarrierPositionNameLabel.setText("ENGAGED");
            } else {
                loaderCarrierPositionNameLabel.setText("IN_TRAVEL");
                loaderCarrierPositionNameLabel.setForeground(Color.YELLOW);
            }
        }
    }

    
    /**
     * A Runnable class to update FcsGeneralView from a KeyValueDataList of sensors.
     */
    public class UpdateFromSensorList implements Runnable {
        
        private final KeyValueDataList kvdl;

        public UpdateFromSensorList(KeyValueDataList kvdl) {
            this.kvdl = kvdl;
        }

        @Override
        public void run() {
            /* sMap is the map obtained from kvdl : key= (String) sensorName, value = (int) sensorValue */
            Map<String, Serializable> sMap = 
                    kvdl.getListOfKeyValueData().stream().collect(Collectors.toMap(KeyValueData::getKey, KeyValueData::getValue));
            
            if (1 == (int) sMap.get(FILTERPRESENCE) && 1 == (int) sMap.get(FILTERPRESENCEB)) {
                /* a filter is on loader*/
                filterOnLoaderLabel.setText("YES");
                filterOnLoaderLabel.setForeground(greenColor);
            } else {
                filterOnLoaderLabel.setText(NONE);
                filterOnLoaderLabel.setForeground(Color.gray);
            }
        }
    
    }

    
    

    // <editor-fold defaultstate="collapsed" desc="Form Description">                          
    private void initComponents() {

        /**
         ******************************************************************************
         ** Buttons initialization
         * *****************************************************************************
         */
        setFilterButton.addActionListener((ActionEvent evt) -> {
            setFilterButtonActionPerformed(evt);
        });

        loadFilterButton.addActionListener((ActionEvent evt) -> {
            loadFilterButtonActionPerformed(evt);
        });

        unloadFilterButton.addActionListener((ActionEvent evt) -> {
            unloadFilterButtonActionPerformed(evt);
        });

        connectLoaderButton.addActionListener((ActionEvent evt) -> {
            connectLoaderButtonActionPerformed(evt);
        });

        disconnectLoaderButton.addActionListener((ActionEvent evt) -> {
            disconnectLoaderButtonActionPerformed(evt);
        });
        
        /**
         ******************************************************************************
         ** Autochanger Panel
         * *****************************************************************************
         */
        autochangerPanel = new JPanel();
        autochangerPanel.setBorder(BorderFactory.createTitledBorder("autochanger"));
        autochangerPanel.setLayout(new GridBagLayout());
        GridBagConstraints gbc1 = new GridBagConstraints();
        gbc1.insets = insets_std;
        //first colomn
        gbc1.gridx = 0;
        gbc1.gridy = 0;
        gbc1.anchor = GridBagConstraints.LINE_START;
        autochangerPanel.add(new JLabel("Filter ONLINE: "),gbc1);
        gbc1.gridy++;
        autochangerPanel.add(new JLabel("Filter on trucks: "), gbc1);
        gbc1.gridy++;
        autochangerPanel.add(new JLabel("ONLINE clamps state: "),gbc1); 
        gbc1.gridy++;
        autochangerPanel.add(new JLabel("Latches state: "),gbc1);        
        gbc1.gridy++;
        autochangerPanel.add(new JLabel("Trucks Position: "),gbc1); 
        //second colomn
        gbc1.gridx++;
        gbc1.gridy = 0;
        gbc1.anchor = GridBagConstraints.LINE_END;
        autochangerPanel.add(filterONLINELabel, gbc1);
        gbc1.gridy++;
        autochangerPanel.add(filterOnTrucksLabel, gbc1);
        gbc1.gridy++;
        autochangerPanel.add(onlineClampsStateLabel, gbc1);
        gbc1.gridy++;
        autochangerPanel.add(latchesStateLabel, gbc1);        
        gbc1.gridy++;
        autochangerPanel.add(trucksPositionNameLabel, gbc1);
        gbc1.gridx++;
        autochangerPanel.add(trucksPositionLabel, gbc1);
         /**
         ******************************************************************************
         ** Carousel Panel
         * *****************************************************************************
         */
        carouselPanel = new JPanel();
        carouselPanel.setBorder(BorderFactory.createLineBorder(Color.yellow));
        carouselPanel.setBorder(BorderFactory.createTitledBorder("carousel"));
        carouselPanel.setLayout(new GridBagLayout());
        gbc1 = new GridBagConstraints();
        gbc1.insets = insets_std;
        //first colomn
        gbc1.gridx = 0;
        gbc1.gridy = 0;
        gbc1.anchor = GridBagConstraints.LINE_START;
        carouselPanel.add(new JLabel("Position: "), gbc1);
        gbc1.gridy++;
        carouselPanel.add(new JLabel("Delta Position: "), gbc1);
        gbc1.gridy++;
        carouselPanel.add(new JLabel("Speed: "), gbc1);
        gbc1.gridy++;
        carouselPanel.add(new JLabel("Current: "), gbc1);
        gbc1.gridy++;
        carouselPanel.add(standbyLabel, gbc1);
        gbc1.gridy++;
        carouselPanel.add(new JLabel("Socket at STANDBY: "), gbc1);
        gbc1.gridy++;
        carouselPanel.add(new JLabel("Filter at STANDBY: "), gbc1);
        gbc1.gridy++;
        carouselPanel.add(new JLabel("IO Status: "), gbc1);
        gbc1.gridy++;
        carouselPanel.add(new JLabel("Clamps Status: "), gbc1);
        //second colomn
        gbc1.gridx++;
        gbc1.gridy = 0;
        gbc1.anchor = GridBagConstraints.LINE_END;
        carouselPanel.add(actualPositionLabel, gbc1);
        gbc1.gridy++;
        carouselPanel.add(deltaPositionLabel, gbc1);
        gbc1.gridy++;
        carouselPanel.add(actualSpeedLabel, gbc1);
        gbc1.gridy++;
        carouselPanel.add(actualCurrentLabel, gbc1);
        gbc1.gridy++;
        standbySwitch.setColor(Color.gray);
        carouselPanel.add(standbySwitch, gbc1);
        gbc1.gridy++;
        carouselPanel.add(socketAtStandbyLabel, gbc1);
        gbc1.gridy++;
        carouselPanel.add(filterAtStandbyLabel, gbc1);
        gbc1.gridy++;
        carouselPanel.add(ioModuleStatusLabel, gbc1);
        gbc1.gridy++;
        carouselPanel.add(caClampsStateLabel, gbc1);
         /**
         ******************************************************************************
         ** Loader Panel
         * *****************************************************************************
         */
        loaderPanel = new JPanel();
        loaderPanel.setBorder(BorderFactory.createTitledBorder("loader"));
        loaderPanel.setLayout(new GridBagLayout());
        gbc1 = new GridBagConstraints();
        gbc1.insets = insets_std;
        //first colomn
        gbc1.gridx = 0;
        gbc1.gridy = 0;
        gbc1.anchor = GridBagConstraints.LINE_START;
        loaderPanel.add(new JLabel("Clamp Homing: "), gbc1);
        gbc1.gridy++;
        loaderPanel.add(new JLabel("Filter on loader: "), gbc1);
        gbc1.gridy++;
        loaderPanel.add(new JLabel("Clamp state: "), gbc1);
        gbc1.gridy++;
        loaderPanel.add(new JLabel("Clamp position: "), gbc1);
        gbc1.gridy++;
        loaderPanel.add(new JLabel("Carrier position: "), gbc1);

        //second colomn
        gbc1.gridx++;
        gbc1.gridy = 0;
        gbc1.anchor = GridBagConstraints.LINE_END;
        loaderPanel.add(loaderClampHomingLabel, gbc1);
        gbc1.gridy++;
        loaderPanel.add(filterOnLoaderLabel, gbc1);
        gbc1.gridy++;
        loaderPanel.add(loaderClampStateLabel, gbc1);
        gbc1.gridy++;
        loaderPanel.add(loaderClampPositionLabel, gbc1);
        gbc1.gridy++;
        loaderPanel.add(loaderCarrierPositionNameLabel, gbc1);
        gbc1.gridx++;
        loaderPanel.add(loaderCarrierPositionLabel, gbc1);

        /**
         ******************************************************************************
         ** Loader Control Panel
         * *****************************************************************************
         */
        loaderControlPanel = new JPanel();
        loaderControlPanel.setBorder(BorderFactory.createTitledBorder("loader commands"));
        loaderControlPanel.setLayout(new GridBagLayout());
        gbc1 = new GridBagConstraints();
        gbc1.insets = insets_std;
        gbc1.gridx = 0;
        gbc1.gridy = 0;
        loaderControlPanel.add(connectLoaderButton, gbc1);
        gbc1.gridy++;
        loaderControlPanel.add(disconnectLoaderButton, gbc1);
        gbc1.gridy = 0;
        gbc1.gridx++;
        loaderControlPanel.add(loadFilterButton, gbc1);
        gbc1.gridy++;
        loaderControlPanel.add(unloadFilterButton, gbc1);

        /**
         ******************************************************************************
         ** Control Panel
         * *****************************************************************************
         */
        controlPanel = new JPanel();
        controlPanel.setLayout(new GridBagLayout());
        gbc1 = new GridBagConstraints();
        gbc1.insets = insets_std;
        gbc1.gridx = 0;
        gbc1.gridy = 0;
        controlPanel.add(filterLabel, gbc1);
        gbc1.gridx++;
        controlPanel.add(filterField, gbc1);
        gbc1.gridx++;
        controlPanel.add(setFilterButton, gbc1);
        gbc1.gridx = 0;
        gbc1.gridy = 1;
        gbc1.gridwidth = 3;
        controlPanel.add(loaderControlPanel, gbc1);


        /**
         ******************************************************************************
         ** Whole Panel
         * *****************************************************************************
         */
        /*General Information Panel*/
        setBorder(BorderFactory.createTitledBorder(title));
        setLayout(new java.awt.GridBagLayout());
        GridBagConstraints gbc = new GridBagConstraints();
        gbc.insets = insets_std;
        gbc.anchor = GridBagConstraints.BASELINE_LEADING;

        gbc.gridx = 0;
        gbc.gridy = 0;
        gbc.gridwidth = 3;
        add(cameraStatusPanel, gbc);

        gbc.gridy++;
        gbc.gridwidth = 1;
        gbc.gridheight = 2;
        add(carouselPanel, gbc);

        gbc.gridheight = 1;

        gbc.gridx++;
        add(autochangerPanel, gbc);
        
        gbc.gridx++;
        add(loaderPanel, gbc);

        gbc.gridx = 1;
        gbc.gridy++;
        add(controlPanel, gbc);
        /**
         ******************************************************************************
         ** end of Whole Panel
         * *****************************************************************************
         */
        setDefaultValues();
    }

    /**
     * set default values for components color and labels.
     */
    private void setDefaultValues() {
        /*carousel*/
        actualPositionLabel.setText(ZERO_VALUE);
        actualSpeedLabel.setText(ZERO_VALUE);
        actualCurrentLabel.setText(ZERO_VALUE);
        standbyLabel.setText(NOT_AT_STANDBY);
        standbySwitch.setColor(Color.gray);
        socketAtStandbyLabel.setText(NONE);
        filterAtStandbyLabel.setText(NO_FILTER);
        caClampsStateLabel.setText(UNKNOWN_STATE);
        ioModuleStatusLabel.setText(UNKNOWN_STATE);

        /*autochanger*/
        trucksPositionLabel.setText(ZERO_VALUE);
        trucksPositionNameLabel.setText(UNKNOWN_STATE);
        onlineClampsStateLabel.setText(UNKNOWN_STATE);
        latchesStateLabel.setText(UNKNOWN_STATE);

        /* loader */
        loaderClampHomingLabel.setText(UNKNOWN);
        filterOnLoaderLabel.setText(UNKNOWN);
        loaderClampStateLabel.setText(UNKNOWN);
        loaderClampPositionLabel.setText(ZERO_VALUE);
        loaderCarrierPositionNameLabel.setText(UNKNOWN);
        loaderCarrierPositionLabel.setText(ZERO_VALUE);
    }

    /**
     * Reset Panels to default values.
     */
    public void resetPanel() {
        setDefaultValues();
        cameraStatusPanel.resetPanel();
    }
    
    @Override
    public String toString() {
        return title;
    }

    private void setFilterButtonActionPerformed(ActionEvent evt) {
        filterID = Integer.parseInt(filterField.getText());
        subs.setFilter(filterID);
    }

    private void loadFilterButtonActionPerformed(ActionEvent evt) {
        subs.loadFilter();
    }

    private void unloadFilterButtonActionPerformed(ActionEvent evt) {
        subs.unloadFilter();
    }

    private void connectLoaderButtonActionPerformed(ActionEvent evt) {
        subs.connectLoader();
    }

    private void disconnectLoaderButtonActionPerformed(ActionEvent evt) {
        subs.disconnectLoader();
    }


    /**
     * This main launches an GUI for the whole subsystem "fcs". It's used only
     * for tests.
     *
     * @param args the command line arguments
     */
    public static void main(String[] args) {

        FcsGeneralViewPanel d = new FcsGeneralViewPanel();
        JFrame frame = new JFrame("FCS General View Panel");
        frame.setContentPane(d);
        frame.pack();
        frame.setVisible(true);
        frame.setDefaultCloseOperation(JFrame.EXIT_ON_CLOSE);
    }
    
}
