package org.lsst.ccs.subsystems.fcs.ui.commons;

import java.lang.reflect.InvocationTargetException;
import java.util.List;
import java.util.Map;
import javax.swing.JPanel;
import org.lsst.ccs.bus.data.ConfigurationInfo;
import org.lsst.ccs.bus.data.KeyValueData;
import org.lsst.ccs.bus.messages.StatusConfigurationInfo;
import org.lsst.ccs.bus.messages.StatusMessage;
import org.lsst.ccs.bus.messages.StatusSubsystemData;
import org.lsst.ccs.messaging.StatusMessageListener;
import org.lsst.ccs.subsystems.fcs.FCSCst;
import org.lsst.ccs.subsystems.fcs.StatusDataPublishedByHardware;
import static org.lsst.ccs.subsystems.fcs.ui.commons.Tools.getShortComponentName;
import org.lsst.ccs.utilities.logging.Logger;

/**
 * This class is implemented by all the FCS GUIs. 
 * It's goal is to manage the common stuff for all the subsystems :
 * initialization and update of the panels for the hardware panels  
 * the panels where the CANopen devices are displayed.
 * The subsystemPanel is handled by the GeneralPanel.
 * 
 * @author virieux
 */
public abstract class GatherPanel extends JPanel implements StatusMessageListener {

    protected static final Logger FCSLOG = FCSCst.FCSLOG;
    private static final long serialVersionUID = 3948843130488926276L;
    protected CanOpenHardwareListPanel canOpenHardwareListPanel;

    /**
     * Creates a TopPanel with a tcpProxy name.
     * @param tcpProxyName
     */
    public GatherPanel(String tcpProxyName) {
        //CANopen devices panel
        canOpenHardwareListPanel = new CanOpenHardwareListPanel(tcpProxyName);
        setName("gatherPanel");
    }
    
    public void setHardwareNames(List<String> hardwareNames) {
        FCSLOG.info("setHardwareNames with names=" + hardwareNames);
        canOpenHardwareListPanel.setHardwareNames(hardwareNames);
    }

    public CanOpenHardwareListPanel getCanOpenHardwareListPanel() {
        return canOpenHardwareListPanel;
    }
    
    /**
     * Update GUI when a StatusConfigurationInfo message is received on the status bus. 
     * @param configInfo 
     */
    abstract public void updateFromConfigurationInfo(ConfigurationInfo configInfo);
    
    /**
     * Update GUI when a KeyValueData message is received on the status bus.
     * @param data 
     */
    abstract public void updateFromStatusData(KeyValueData data);

    /**
     * What to do when we receive data coming from our destination.
     * @param msg
     */
    @Override
    public void onStatusMessage(StatusMessage msg) {
        
        if (msg instanceof StatusConfigurationInfo) {
            FCSLOG.debug(getName() + " read data StatusConfigurationInfo");
            updateFromConfigurationInfo(((StatusConfigurationInfo) msg).getConfigurationInfo());
            
        } else {
            // Safe cast : filtering ensures we only receive StatusSubsystemData messages
            StatusSubsystemData d = (StatusSubsystemData)msg;
            KeyValueData data = d.getObject();
            FCSLOG.debug(getName() + " read data dataType:" + data.getKey());

            updateFromStatusData(data);
        }
    }
    


    /**
     * Update the canOpenHardwareListPanel from the data published by a piece of hardware on the STATUS bus.
     * @param deviceToUpdateName
     * @param statusDataPublishedByHardware 
     */
    public void updateCanOpenHardwarePanel(String deviceToUpdateName, StatusDataPublishedByHardware statusDataPublishedByHardware) {
        this.canOpenHardwareListPanel.updateCanOpenHardwarePanel(getShortComponentName(deviceToUpdateName), statusDataPublishedByHardware);
    }
    
    /**
     * Update the canOpenHardwareListPanel from number of booted nodes.
     * @param bootedNodesNb 
     */
    public void updateCanOpenHardwarePanel(int bootedNodesNb) {
        this.canOpenHardwareListPanel.updateCanOpenHardwarePanel(bootedNodesNb);
    }
    
    /**
     * 
     * @param configInfo
     * @param hardwareList
     */
    public void initializeGUIWithConfigInfo(ConfigurationInfo configInfo, List<String> hardwareList) {
        this.canOpenHardwareListPanel.initializeGui(configInfo, hardwareList);
    }


    /**
     * To update GUI with LOADER hardware list.
     * This method has to be overridden for wholeFCS GUI.
     * @param configInfo
     * @param hardwareList 
     * @throws java.lang.InterruptedException 
     * @throws java.lang.reflect.InvocationTargetException 
     */
    public void initializeGUIWithLoaderConfigInfo(ConfigurationInfo configInfo, List<String> hardwareList)
        throws InterruptedException, InvocationTargetException{
        /* To be overridden for whole FCS GUI in FcsGatherPanel*/
    }

    /**
     * This method is to retrieve the configuration information for the filters from the ConfigurationInfo
     * and the list of filter names. 
     * It's empty 
     * @param configInfo 
     * @param filterNamesList 
     * @throws java.lang.InterruptedException 
     * @throws java.lang.reflect.InvocationTargetException 
     */
    public void initializeGUIWithFiltersInfo(ConfigurationInfo configInfo, List<String> filterNamesList)
            throws InterruptedException, InvocationTargetException {  
        /* When the TopPanel contains Panels which display filters, the body of this method should exists.*/
    }

    /**
     *
     * @param controllersMap
     */
    public void initializeGUIWithControllersMap(Map<String, String> controllersMap) {
        /* To be overridden by subclasses */
    }
    
    public void initializeGuiWithAutochangerSensorsList(ConfigurationInfo configInfo, List<String> sensorsList) {
        /* To be overridden by subclasses */
    }
    
    public void initializeGuiWithLoaderSensorsList(ConfigurationInfo configInfo, List<String> sensorsList) {
        /* To be overridden by subclasses */
    }
    
    public boolean isMyHardware(String hardwareName) {
        FCSLOG.info("hardwareNameList");
        FCSLOG.info("hardwareName");
        return canOpenHardwareListPanel.getHardwareNames().contains(hardwareName);
    }
    
    /**
     * This methods reset panels with initial text and color.
     */
    public void resetPanel() {
        this.canOpenHardwareListPanel.resetPanel();
    }


}
