package org.lsst.ccs.subsystems.fcs.carousel.ui;

import java.awt.Color;
import java.awt.Dimension;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.util.Map;
import javax.swing.BorderFactory;
import javax.swing.JFrame;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.SwingUtilities;
import org.lsst.ccs.bus.data.ConfigurationInfo;
import static org.lsst.ccs.subsystems.fcs.FCSCst.FCSLOG;
import static org.lsst.ccs.subsystems.fcs.FCSCst.NO_FILTER;
import static org.lsst.ccs.subsystems.fcs.FCSCst.AT_STANDBY;
import static org.lsst.ccs.subsystems.fcs.FCSCst.CA_CLAMPXMINUSCONTROLLER_NAME;
import static org.lsst.ccs.subsystems.fcs.FCSCst.CA_CLAMPXPLUSCONTROLLER_NAME;
import static org.lsst.ccs.subsystems.fcs.FCSCst.NOT_AT_STANDBY;
import static org.lsst.ccs.subsystems.fcs.FCSCst.UNKNOWN;
import org.lsst.ccs.subsystems.fcs.StatusDataPublishedByCarousel;
import org.lsst.ccs.subsystems.fcs.StatusDataPublishedByCarouselClamp;
import org.lsst.ccs.subsystems.fcs.StatusDataPublishedByCarouselSocket;
import org.lsst.ccs.subsystems.fcs.StatusDataPublishedByEPOSController;
import org.lsst.ccs.subsystems.fcs.ui.commons.DigitalSwitch;
import org.lsst.ccs.subsystems.fcs.ui.commons.FilterPanel;
import static org.lsst.ccs.subsystems.fcs.ui.commons.Tools.UNKNOWN_STATE;
import static org.lsst.ccs.subsystems.fcs.ui.commons.Tools.BIG_FONT;
import static org.lsst.ccs.subsystems.fcs.ui.commons.Tools.TIRETS;
import static org.lsst.ccs.subsystems.fcs.ui.commons.Tools.UNKNOWN_VALUE;
import static org.lsst.ccs.subsystems.fcs.ui.commons.Tools.greenColor;
import static org.lsst.ccs.subsystems.fcs.ui.commons.Tools.insets_std;
import static org.lsst.ccs.subsystems.fcs.ui.commons.Tools.VERY_BIG_FONT;
import static org.lsst.ccs.subsystems.fcs.ui.commons.Tools.ZERO_VALUE;

/**
 * A Panel for a carousel socket.
 * There are 2 carousel clamps on a socket.
 * @author virieux
 */
public class CarouselSocketPanel extends JPanel {
    private static final long serialVersionUID = 6352712511682293959L;

    private InterfaceCarouselGUI subs;
    private final String socketName;
    private final String clampXminusName;
    private final String clampXplusName;

    private CarouselClampPanel clampXminusPanel;
    private CarouselClampPanel clampXplusPanel;
    private FilterPanel filterPanel;

    /*Clamp State Panel*/
    private JPanel clampsStatePanel;
    private final JLabel clampsState = new JLabel(UNKNOWN_STATE);
    private final JLabel clampsStateLabel = new JLabel("Clamps State :");
    private final JLabel xMinusLabel = new JLabel("X-");
    private final JLabel xMinusLabel2 = new JLabel("X-");
    private final JLabel xPlusLabel = new JLabel("X+");
    private final JLabel xPlusLabel2 = new JLabel("X+");

    /*General Information Panel*/
    private JPanel generalInfoPanel;
    private final JLabel nLabel = new JLabel("Name: ");
    private final JLabel nameLabel = new JLabel("socketXXX");
    private final JLabel positionOnCarouselLabel = new JLabel("0");
    private final JLabel sLabel = new JLabel("Carousel position at STANDBY: ");
    private final JLabel standbyPositionLabel = new JLabel(UNKNOWN_VALUE);

    /*Actual Position Panel*/
    private JPanel actualPositionPanel;
    private final JLabel actualPositionLabel = new JLabel(UNKNOWN_VALUE);
    private final JLabel deltaPositionLabel = new JLabel(ZERO_VALUE);
    private final JLabel standbyLabel = new JLabel(NOT_AT_STANDBY);
    private final DigitalSwitch standbySwitch = new DigitalSwitch();
    private final JLabel ioModuleStatusLabel = new JLabel();

    /**
     *
     * Creates a CarouselSocketPanel with no parameter.
     * Initializes fields socketName, clampXminusName and clampXplusName.
     * Just for tests.
     */
    public CarouselSocketPanel() {
        socketName = "socket99";
        clampXminusName = "clampXminus99";
        clampXplusName = "clampXplus99";
        initComponents();
    }

    /**
     * Creates a CarouselSocketPanel with a socketName.
     * @param socketName
     */
    public CarouselSocketPanel(String socketName) {
        this.socketName = socketName;
        nameLabel.setText(socketName);

        /*To find the name of the clamps needed to initialize the clampPanels.*/
        char num = socketName.charAt(socketName.length()-1);
        clampXminusName = socketName+"/clampXminus" + num;
        clampXplusName = socketName+"/clampXplus" + num;
        clampXminusPanel = new CarouselClampPanel(clampXminusName);
        clampXplusPanel = new CarouselClampPanel(clampXplusName);
        initComponents();
    }

    public void setSubsystem(InterfaceCarouselGUI subs) {
        this.subs = subs;
    }



    /**
     * Initialize the GUI with the configuration data.
     *
     * @param configInfo
     */
    public void initializeGui(ConfigurationInfo configInfo) {
        Map<String, String> config = configInfo.getCurrentValuesFor(socketName);
        positionOnCarouselLabel.setText(config.get("positionOnCarousel"));
        standbyPositionLabel.setText(config.get("standbyPosition"));
        clampXminusPanel.initializeGui(configInfo);
        clampXplusPanel.initializeGui(configInfo);
    }

    public void updateSocket(StatusDataPublishedByCarouselSocket s) {
        SwingUtilities.invokeLater(new UpdateSocket(s));
    }

    public void updateSocketFromCarousel(StatusDataPublishedByCarousel s) {
        SwingUtilities.invokeLater(new UpdateSocketFromCarousel(s));
    }
    
    public void updateController(String controllerName, StatusDataPublishedByEPOSController s) {
        switch (controllerName) {
            case CA_CLAMPXMINUSCONTROLLER_NAME:
                clampXminusPanel.updateController(s);
                break;

            case CA_CLAMPXPLUSCONTROLLER_NAME:
                clampXplusPanel.updateController(s);
                break;
        }
    }
    

    /**
     * A Runnable to updateSocket this panel from a StatusDataPublishedByCarouselSocket.
     */
    public class UpdateSocket implements Runnable {
        private StatusDataPublishedByCarouselSocket s;

        public UpdateSocket(StatusDataPublishedByCarouselSocket s) {
            this.s = s;
        }

        @Override
        public void run() {
            if (s.isAtStandby() && !s.isCarouselMoving()) {
                standbyLabel.setText(AT_STANDBY);
                standbySwitch.setColor(greenColor);
                deltaPositionLabel.setText(Integer.toString(s.getDeltaPosition()));
            } else {
                standbyLabel.setText(NOT_AT_STANDBY);
                standbySwitch.setColor(Color.orange);
                deltaPositionLabel.setText(TIRETS);
                //the following lines are ok if controllers publish regulary otherwise
                //controller panel is not updated.
//                clampXminusPanel.resetControllerPanel();
//                clampXplusPanel.resetControllerPanel();
            }
            clampsState.setText(s.getClampState().toString());
            clampsState.setForeground(s.getClampState().getColor());
            filterPanel.setIsFilterInHouse(!s.isEmpty(), s.getFilterName(), s.getFilterID());
            ioModuleStatusLabel.setText(s.getIOStatus().toString());
            ioModuleStatusLabel.setForeground(s.getIOStatus().getColor());
        }
    }

    /**
     * A Runnable to update this panel from a StatusDataPublishedByCarousel.
     */
    public class UpdateSocketFromCarousel implements Runnable {
        private StatusDataPublishedByCarousel s;

        public UpdateSocketFromCarousel(StatusDataPublishedByCarousel s) {
            this.s = s;
        }

        @Override
        public void run() {
            actualPositionLabel.setText(Integer.toString(s.getPosition()));
            clampXminusPanel.updateMinLocked(s.getMinLocked());
            clampXplusPanel.updateMinLocked(s.getMinLocked());
        }
    }


    /**
     * Update the socket panel from data published by a carousel clamp.
     * It can be a clampXminus or a clampXplus.
     * @param clampName
     * @param s
     */
    public void update(String clampName, StatusDataPublishedByCarouselClamp s) {
        //TODO when the filter goes to the autochanger this has to be changed.
        if (clampName.contains("Xminus")) {
            clampXminusPanel.updateClamp(s);
        } else if (clampName.contains("Xplus")) {
            clampXplusPanel.updateClamp(s);
        } else {
            FCSLOG.error("Error in clamps name on STATUS bus. Unknown clamp name: " + clampName);
        }
    }


    // <editor-fold defaultstate="collapsed" desc="Form description">
    private void initComponents() {
        setName(socketName+"Panel");

        /**
         * X- and X+ label
         */
        xMinusLabel.setFont(VERY_BIG_FONT); // NOI18N
        xMinusLabel2.setFont(VERY_BIG_FONT); // NOI18N
        xPlusLabel2.setFont(VERY_BIG_FONT); // NOI18N
        xPlusLabel.setFont(VERY_BIG_FONT); // NOI18N
        /**
         ******************************************************************************
         ** General Informations Panel
         * *****************************************************************************
         */
        generalInfoPanel = new JPanel();
        generalInfoPanel.setBorder(BorderFactory.createTitledBorder("General Informations"));
        generalInfoPanel.setLayout(new GridBagLayout());
        GridBagConstraints gbc3 = new GridBagConstraints();
        gbc3.insets = insets_std;
        //first colomn
        gbc3.gridx = 0;
        gbc3.gridy = 0;
        gbc3.anchor = GridBagConstraints.LINE_START;

        generalInfoPanel.add(nLabel, gbc3);
        gbc3.gridy++;

        generalInfoPanel.add(sLabel, gbc3);
        //second colomn
        gbc3.gridx++;
        gbc3.gridy = 0;
        gbc3.anchor = GridBagConstraints.LINE_END;
        generalInfoPanel.add(nameLabel, gbc3);
        nameLabel.setFont(BIG_FONT);
        gbc3.gridy++;
        generalInfoPanel.add(standbyPositionLabel, gbc3);
        /**
         ******************************************************************************
         ** end of General Informations Panel
         * *****************************************************************************
         */

        /**
         ******************************************************************************
         ** Actual Position Panel
         * *****************************************************************************
         */
        actualPositionPanel = new JPanel();
        actualPositionPanel.setBorder(BorderFactory.createLineBorder(Color.yellow));
        actualPositionPanel.setLayout(new GridBagLayout());
        actualPositionPanel.setMinimumSize(new Dimension(150,50));
        GridBagConstraints gbc1 = new GridBagConstraints();
        gbc1.insets = insets_std;
        //first colomn
        gbc1.gridx = 0;
        gbc1.gridy = 0;
        gbc1.anchor = GridBagConstraints.LINE_START;
        actualPositionPanel.add(new JLabel("Actual position: "), gbc1);
        gbc1.gridy++;
        actualPositionPanel.add(new JLabel("Delta position: "), gbc1);
        gbc1.gridy++;
        actualPositionPanel.add(standbyLabel, gbc1);
        gbc1.gridy++;
        actualPositionPanel.add(new JLabel("IO Status:"),gbc1);
        //second colomn
        gbc1.gridx++;
        gbc1.gridy = 0;
        gbc1.anchor = GridBagConstraints.LINE_END;
        actualPositionPanel.add(actualPositionLabel, gbc1);
        gbc1.gridy++;
        actualPositionPanel.add(deltaPositionLabel, gbc1);
        gbc1.gridy++;
        standbySwitch.setColor(Color.gray);
        actualPositionPanel.add(standbySwitch,gbc1);
        gbc1.gridy++;
        actualPositionPanel.add(ioModuleStatusLabel,gbc1);

        /**
         ******************************************************************************
         ** end of Actual Position Panel
         * *****************************************************************************
         */

        /**
         ******************************************************************************
         ** Clamps State Panel
         * *****************************************************************************
         */
        clampsStatePanel = new JPanel();
        clampsState.setFont(BIG_FONT); // NOI18N
        clampsState.setBorder(BorderFactory.createEtchedBorder(new Color(255, 255, 255), new Color(255, 255, 255)));
        clampsStatePanel.setLayout(new GridBagLayout());
        GridBagConstraints gbc2 = new GridBagConstraints();
        gbc2.insets = insets_std;
        gbc2.gridx = 0;
        gbc2.gridy = 0;
        clampsStatePanel.add(clampsStateLabel, gbc2);
        gbc2.gridx++;
        clampsStatePanel.add(clampsState, gbc2);
        /**
         ******************************************************************************
         ** end of Clamps State Panel
         * *****************************************************************************
         */
         /**
         ******************************************************************************
         ** Clamps Panel
         * *****************************************************************************
         */
        JPanel clampsPanel = new JPanel();
        clampsPanel.setLayout(new GridBagLayout());
        GridBagConstraints gbc = new GridBagConstraints();
        gbc.insets = insets_std;
        gbc.gridx = 0;
        gbc.gridy = 0;
       /*Clamps State  Panel with X- and X+*/
        gbc.anchor = GridBagConstraints.WEST;
        clampsPanel.add(xMinusLabel, gbc);

        gbc.gridx++;
        gbc.anchor = GridBagConstraints.WEST;
        clampsPanel.add(clampsStatePanel, gbc);

        gbc.anchor = GridBagConstraints.EAST;
        gbc.gridx++;
        clampsPanel.add(xPlusLabel, gbc);

        /*Clamps Panel and Filter Panel*/
        gbc.gridx=0;
        gbc.gridy++;
        clampsPanel.add(clampXminusPanel, gbc);
        gbc.gridx++;
        clampsPanel.add(clampXplusPanel, gbc);

        /*Last line with X- and X+*/
        gbc.anchor = GridBagConstraints.WEST;
        gbc.gridx = 0;
        gbc.gridy++;
        clampsPanel.add(xMinusLabel2, gbc);
        gbc.anchor = GridBagConstraints.EAST;
        gbc.gridx++;
        gbc.gridx++;
        clampsPanel.add(xPlusLabel2, gbc);

        /**
         ******************************************************************************
         ** end of Clamps Panel
         * *****************************************************************************
         */

         /**
         ******************************************************************************
         ** Filter Panel
         * *****************************************************************************
         */
        filterPanel = new FilterPanel();
        filterPanel.setSize(100);
        filterPanel.setIsFilterInHouse(false, NO_FILTER, 0);
        /**
         ******************************************************************************
         ** end of Filter Panel
         * *****************************************************************************
         */


        /**
         ******************************************************************************
         ** Whole Panel
         * *****************************************************************************
         */
        setBorder(BorderFactory.createLineBorder(Color.PINK));
        setLayout(new GridBagLayout());
        gbc = new GridBagConstraints();
        gbc.insets = insets_std;
        gbc.anchor = GridBagConstraints.NORTHWEST;

        gbc.gridx = 0;
        gbc.gridy = 0;
        add(generalInfoPanel, gbc);

        gbc.gridx++;
        add(actualPositionPanel);

        gbc.gridx++;
        gbc.anchor = GridBagConstraints.CENTER;
        add(filterPanel, gbc);

        gbc.gridx = 0;
        gbc.gridy++;
        gbc.gridwidth = 3;
        add(clampsPanel,gbc);


        /**
         ******************************************************************************
         ** end of Whole Panel
         * *****************************************************************************
         */
        setDefaultValues();

    }

    /**
     * Reset Panel to display default values when subsystem is halted.
     */
    public void resetPanel() {
        setDefaultValues();
        clampXminusPanel.resetPanel();
        clampXplusPanel.resetPanel();
    }

    private void setDefaultValues() {
        clampsState.setText(UNKNOWN_STATE);
        clampsState.setForeground(Color.darkGray);
        actualPositionLabel.setText(UNKNOWN_VALUE);
        deltaPositionLabel.setText(ZERO_VALUE);
        standbyLabel.setText(UNKNOWN);
        standbySwitch.setColor(Color.darkGray);
        ioModuleStatusLabel.setText(UNKNOWN);
        ioModuleStatusLabel.setForeground(Color.darkGray);
        filterPanel.setIsFilterInHouse(false, NO_FILTER, 0);
    }

    @Override
    public String toString() {
        if (socketName == null) return "Unamed socket";
        int ind = socketName.lastIndexOf("/");            
        return ind >=0 ? socketName.substring(ind+1) : socketName;
    }

    /**
     * Just to test the Panel.
     * @param argv
     */
    public static void main(String[] argv) {

        CarouselSocketPanel d = new CarouselSocketPanel();
        JFrame frame = new JFrame("Test");
        frame.setContentPane(d);
        frame.pack();
        frame.setVisible(true);
        frame.setDefaultCloseOperation(JFrame.EXIT_ON_CLOSE);

    }
}
