package org.lsst.ccs.subsystems.fcs.ui.commons;

import org.lsst.ccs.subsystems.fcs.StatusDataPublishedByEPOSController;
import org.lsst.ccs.subsystems.fcs.ui.ControllersViewPanel;

import javax.swing.BorderFactory;
import javax.swing.BoxLayout;
import javax.swing.JButton;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JSeparator;
import javax.swing.JTextField;
import javax.swing.SwingConstants;
import javax.swing.SwingUtilities;
import java.awt.BorderLayout;
import java.awt.CardLayout;
import java.awt.Color;
import java.awt.Dimension;
import java.awt.FlowLayout;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.util.HashSet;
import java.util.Set;
import java.util.logging.Logger;

import static org.lsst.ccs.subsystems.fcs.ui.commons.Tools.greenColor;

public class SingleControllerPanel extends JPanel {
    private static final Logger FCSLOG = Logger.getLogger(EPOSControllerPanel.class.getName());
    private static final String UNDEFINED = "UNDEFINED";
    private static final String NO_ERROR = "no error";
    private static final String ACTIVATED = "Activated";
    private static final String RELEASED = "Released";
    private final String fullPathControllerName;
    private final String controllerName;
    private final ControllersViewPanel controllersView; // used for updating the error listing
    private final InterfaceGeneralGUI subs;
    private final boolean hasBrakeAttached;
    private final Set<Integer> errorCodeTracker = new HashSet<>();
    private JTextField eposMode;  // modeLabel in EPOSControllerPanel
    private JTextField eposState; // enableLabel
    private JLabel brakeState; // brakeLabel
    private JTextField errorStatus; // errorRegisterLabel
    private JTextField errorCount;
    private DigitalSwitch eposStateIcon; // enableSwitch
    private DigitalSwitch errorIcon; // faultSwitch
    private JButton enable;
    private JButton disable;
    private JButton reset;
    private boolean isControllerEnabled;

    public SingleControllerPanel(String controllerName, String systemName, String fullPathControllerName, InterfaceGeneralGUI subs, ControllersViewPanel controllersView, boolean hasBrakesAttached) {
        this.controllerName = controllerName;
        this.fullPathControllerName = fullPathControllerName;
        this.subs = subs;
        this.controllersView = controllersView;
        this.hasBrakeAttached = hasBrakesAttached;
        this.isControllerEnabled = true;
        initComponents(systemName);
    }

    private void resetActionPerformed() {
        FCSLOG.info("[SingleControllerPanel] reset from " + fullPathControllerName);
        subs.sendCommandSwingWorker("faultReset", 1000, fullPathControllerName);
    }

    private void deactivateActionPerformed() {
        FCSLOG.info("[SingleControllerPanel] deactivate from " + fullPathControllerName);
        subs.sendCommandSwingWorker("goToSwitchOnDisabled", 1000, fullPathControllerName);
    }

    private void activateActionPerformed() {
        FCSLOG.info("[SingleControllerPanel] activate from " + fullPathControllerName);
        subs.sendCommandSwingWorker("goToOperationEnable", 1000, fullPathControllerName);
    }

    public void updateController(StatusDataPublishedByEPOSController status) {
        if (isControllerEnabled) { // todo might be problematic if loader is put online (must have a previous step enabling the controller)
            SwingUtilities.invokeLater(new UpdateController(status));
        }
    }

    @Override
    public void setEnabled(boolean b) {
        // the panel is updated only if its state changes
        if (isControllerEnabled!=b) {
            enableButtons(b);
            isControllerEnabled = b;
        }
    }

    private void enableButtons(boolean b) {
        enable.setEnabled(b);
        disable.setEnabled(b);
        reset.setEnabled(b);
    }

    /*
    GUI initialisation
     */
    private VerticalTextComponent createSystemTag(String name) {
        return new VerticalTextComponent(name);
    }

    private JPanel createEposModePanel() {
        JPanel pane = new JPanel();
        pane.setLayout(new FlowLayout(FlowLayout.LEFT, 0, 0));
        pane.add(new JLabel("EPOS mode:"));
        eposMode = new JTextField(UNDEFINED);
        eposMode.setEditable(false);
        pane.add(eposMode);
        return pane;
    }

    private JPanel createEposStatePanel() {
        JPanel pane = new JPanel();
        pane.setLayout(new FlowLayout(FlowLayout.LEFT, 0, 0));
        eposStateIcon = new DigitalSwitch();
        eposState = new JTextField(UNDEFINED);
        eposState.setEditable(false);
        pane.add(eposStateIcon);
        pane.add(eposState);
        return pane;
    }

    private JPanel createBrakeStatePanel(boolean attachedToController) {
        CardLayout cl = new CardLayout();
        JPanel pane = new JPanel(cl);

        JPanel cardWithBrakes = new JPanel();
        JPanel cardWithoutBrakes = new JPanel();
        cardWithBrakes.setLayout(new FlowLayout(FlowLayout.LEFT, 0, 0));
        cardWithBrakes.add(new JLabel("Brake:"));
        brakeState = new JLabel(UNDEFINED);
        cardWithBrakes.add(brakeState);
        cardWithBrakes.setAlignmentX(RIGHT_ALIGNMENT);

        pane.add(cardWithBrakes);
        pane.add(cardWithoutBrakes);

        if (attachedToController) {
            cl.first(pane);
        } else {
            cl.last(pane);
        }
        return pane;
    }

    private JPanel createErrorStatusPanel() {
        JPanel pane = new JPanel();
        pane.setLayout(new FlowLayout(FlowLayout.LEFT, 0, 0));
        errorIcon = new DigitalSwitch();
        errorStatus = new JTextField(NO_ERROR);
        errorStatus.setEditable(false);
        pane.add(errorIcon);
        pane.add(errorStatus);
        return pane;
    }

    private JPanel createErrorCountPanel() {
        JPanel pane = new JPanel();
        pane.setLayout(new FlowLayout(FlowLayout.LEFT, 0, 0));
        errorCount = new JTextField("0");
        errorCount.setEditable(false);
        pane.add(new JLabel("Number of errors:"));
        pane.add(errorCount);
        pane.add(reset = new JButton("Reset"));
        reset.setToolTipText("Command: faultReset");
        reset.addActionListener((java.awt.event.ActionEvent evt) -> resetActionPerformed());
        return pane;
    }

    private JPanel createCommandPanel() {
        JPanel pane = new JPanel();
        pane.add(enable = new JButton(Tools.formatButtonLabelWhenLinkedWithATooltip("Enable")));
        pane.add(disable = new JButton(Tools.formatButtonLabelWhenLinkedWithATooltip("Disable")));
        enable.setToolTipText(Tools.formatTooltipText("goToOperationEnable", "Enable Controller"));
        disable.setToolTipText(Tools.formatTooltipText("goToSwitchOnDisabled", "Disable Controller (should be the default state)"));
        enable.addActionListener((java.awt.event.ActionEvent evt) -> activateActionPerformed());
        disable.addActionListener((java.awt.event.ActionEvent evt) -> deactivateActionPerformed());
        return pane;
    }

    private void initComponents(String subsystem) {
        JPanel controllerPanel = new JPanel();
        controllerPanel.setBorder(BorderFactory.createTitledBorder(BorderFactory.createLineBorder(VerticalTextComponent.getSubsystemColor(subsystem), 2), controllerName));
        controllerPanel.setLayout(new GridBagLayout());
        GridBagConstraints gbc = new GridBagConstraints();
        gbc.gridx = 0;
        gbc.gridy = 0;
        gbc.anchor = GridBagConstraints.FIRST_LINE_START;
        controllerPanel.add(createEposModePanel(), gbc);
        gbc.gridy++;
        controllerPanel.add(createEposStatePanel(), gbc);
        gbc.gridy++;
        gbc.anchor = GridBagConstraints.LINE_END;
        controllerPanel.add(createCommandPanel(), gbc);
        gbc.gridy++;
        gbc.anchor = GridBagConstraints.CENTER;
        JSeparator sep = new JSeparator(SwingConstants.HORIZONTAL); // default
        sep.setPreferredSize(new Dimension(25, 10));
        sep.setVisible(true);
        gbc.fill = GridBagConstraints.HORIZONTAL;

        controllerPanel.add(sep, gbc);
        gbc.gridy++;
        gbc.anchor = GridBagConstraints.LINE_START;
        controllerPanel.add(createErrorStatusPanel(), gbc);
        gbc.gridy++;
        controllerPanel.add(createErrorCountPanel(), gbc);
        gbc.gridy++;
        controllerPanel.add(createBrakeStatePanel(hasBrakeAttached), gbc);

        setLayout(new BoxLayout(this, BoxLayout.X_AXIS));
        add(createSystemTag(subsystem), BorderLayout.LINE_START);
        add(controllerPanel, BorderLayout.CENTER);
    }

    /**
     * A Runnable class to update this panel from data published on the STATUS bus
     * by an EPOS Controller.
     */
    private class UpdateController implements Runnable {
        private final StatusDataPublishedByEPOSController s;

        public UpdateController(StatusDataPublishedByEPOSController status) {
            this.s = status;
        }

        public boolean processErrorCode(int errorCode) {
            if (!errorCodeTracker.contains(errorCode)) {
                errorCodeTracker.add(errorCode);
                return true;
            } else {
                return false;
            }
        }

        public void clearErrors() {
            errorCodeTracker.clear();
        }

        @Override
        public void run() {
            if (s.getMode()!=null) {
                eposMode.setText(s.getMode().toString());
            }
            if (s.isInitialized()) {
                errorIcon.setColor(s.isInError() ? Color.RED:greenColor);
                errorStatus.setText(s.getErrorRegister());

                errorCount.setText(String.valueOf(s.getErrorHistoryNB()));
                if (s.getErrorHistoryNB() > 0) {
                    if (processErrorCode(s.getLastErrorCode())) {
                        controllersView.updateErrorListing(controllerName, s.getLastErrorCode(), s.getLastErrorName());
                    }
                } else {
                    clearErrors();
                    controllersView.clearErrorListing(controllerName);
                }
            } else {
                if (s.isBooted()) {
                    errorIcon.setColor(Color.ORANGE);
                    errorStatus.setText("Not initialized");
                    setEnabled(true);
                } else {
                    errorIcon.setColor(Color.RED);
                    errorStatus.setText("Not booted");
                    setEnabled(false);
                }
            }
            if (s.isBooted() && s.getState()!=null) {
                eposState.setText(s.getState().name());
                eposStateIcon.setColor(s.getState().getColor());
            } else if (!s.isBooted()) {
                eposState.setText("Not booted");
                eposStateIcon.setColor(Color.RED);
            }
            if (s.isBooted() && s.isControllerWithBrake() && s.isBrakeActivated()) {
                brakeState.setText(ACTIVATED);
            } else if (s.isControllerWithBrake() && !s.isBrakeActivated()) {
                brakeState.setText(RELEASED);
            } else {
                if (hasBrakeAttached) {
                    brakeState.setText("");
                }
            }
        }
    }
}
