
package org.lsst.ccs.subsystems.fcs.autochanger.ui;

import java.awt.Color;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.event.ActionEvent;
import java.util.Map;
import java.util.logging.Logger;
import javax.swing.BorderFactory;
import javax.swing.JButton;
import javax.swing.JFrame;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.SwingUtilities;
import org.lsst.ccs.bus.data.ConfigurationInfo;
import static org.lsst.ccs.subsystems.fcs.FCSCst.AC_LATCHXMINUSCONTROLLER_NAME;
import static org.lsst.ccs.subsystems.fcs.FCSCst.AC_LATCHXPLUSCONTROLLER_NAME;
import org.lsst.ccs.subsystems.fcs.FcsEnumerations;
import org.lsst.ccs.subsystems.fcs.FcsEnumerations.LockStatus;
import org.lsst.ccs.subsystems.fcs.StatusDataPublishedByAutochangerLatch;
import org.lsst.ccs.subsystems.fcs.StatusDataPublishedByEPOSController;
import org.lsst.ccs.subsystems.fcs.ui.commons.ControllerStatePanel;
import org.lsst.ccs.subsystems.fcs.ui.commons.DigitalSwitch;
import static org.lsst.ccs.subsystems.fcs.ui.commons.Tools.UNKNOWN_STATE;
import static org.lsst.ccs.subsystems.fcs.ui.commons.Tools.ZERO_VALUE;
import static org.lsst.ccs.subsystems.fcs.ui.commons.Tools.insets_big;
import static org.lsst.ccs.subsystems.fcs.ui.commons.Tools.insets_std;

/**
 *
 * @author virieux
 */
public class AutochangerLatchPanel extends JPanel {

    private static final long serialVersionUID = -3564360054429278810L;
    private static final Logger FCSLOG = Logger.getLogger(AutochangerLatchPanel.class.getName());

    // Variables declaration
    private InterfaceAutochangerGUI subs;

    private DigitalSwitch openSensorDS;
    private DigitalSwitch closeSensorDS;
    private DigitalSwitch filterEngagedSensorDS;
    private DigitalSwitch errorDS;

    private JPanel sensorsPanel;
    private String latchName;
    private JLabel latchNameLabel;
    private JLabel openLabel;
    private JLabel closeLabel;
    private JLabel errorLabel;
    private JLabel filterEngagedLabel;
    private JLabel lockStatus; //is it useful to display the lockStatus ?

    //Config Parameters Panel
    private JPanel paramPanel;
    private final JLabel motionTimeoutLabel = new JLabel();
    private final JLabel currentToOpenLabel = new JLabel();
    private final JLabel currentToCloseLabel = new JLabel();

    //Controller State Panel
    private final ControllerStatePanel controllerPanel;

    //Control panel
    private JPanel controlPanel;
    private JButton abortButton;
    private JButton closeButton;
    private JButton openButton;

    /**
     * A Runnable to initialize this Panel.
     */
    private class GuiInitialization implements Runnable {

        private final ConfigurationInfo configInfo;
        private final long timeout;

        public GuiInitialization(ConfigurationInfo configInfo, long timeout) {
            this.configInfo = configInfo;
            this.timeout = timeout;
        }

        @Override
        public void run() {
            //Retrieve the configuration data for my latchName.
            Map<String, String> configForName = configInfo.getCurrentValuesFor(latchName);
            FCSLOG.fine(() -> latchName + " configForName=" + configForName);

            //Config Panel
            motionTimeoutLabel.setText(Long.toString(timeout));
            String currentToOpenL = configForName.get("currentToOpen");
            currentToOpenLabel.setText(currentToOpenL);
            currentToCloseLabel.setText("-" + currentToOpenL);
        }
    }

    /**
     * A Runnable to update this Panel from data published on the STATUS bus from
     * a latch.
     * @param status
     */
    class UpdateLatch implements Runnable {

        private final StatusDataPublishedByAutochangerLatch s;

        public UpdateLatch(StatusDataPublishedByAutochangerLatch status) {
            this.s = status;
        }

        @Override
        public void run() {
            openSensorDS.updateColor(s.isUnlockSensorsInError(), s.getUnlockSensorValue());
            closeSensorDS.updateColor(s.isLockSensorsInError(), s.getLockSensorValue());
            filterEngagedSensorDS.updateColor(s.isFilterEngagedSensorsInError(), s.getFilterPresenceSensorValue());
            boolean inTravel = s.getLockStatus().equals(FcsEnumerations.LockStatus.INTRAVEL);
            boolean unknown = s.getLockStatus().equals(FcsEnumerations.LockStatus.UNKNOWN);
            if (s.isInError() || inTravel || unknown) {
                errorDS.setColor(s.getLockStatus().getColor());
                errorLabel.setText(s.getLockStatus().getText());
            } else {
                errorDS.setColor(LockStatus.NOERROR.getColor());
                errorLabel.setText(LockStatus.NOERROR.getText());
            }
            lockStatus.setText(s.getLockStatus().getText());
            lockStatus.setForeground(s.getLockStatus().getColor());
        }
    }

    // End of variables declaration
    public AutochangerLatchPanel(String latchName) {
        this.latchName = latchName;
        String controllerName = latchName.contains("Xminus") ? AC_LATCHXMINUSCONTROLLER_NAME : AC_LATCHXPLUSCONTROLLER_NAME;
        controllerPanel = new ControllerStatePanel(controllerName, false);
        initComponents();
    }

    public void setSubsystem(InterfaceAutochangerGUI subs) {
        this.subs = subs;
    }

    void setLatchName(String name) {
        latchNameLabel.setText(name);
    }

    /**
     * Initialize this Panel from configuration data.
     * @param configInfo
     * @param timeout
     */
    public void initializeGui(ConfigurationInfo configInfo, long timeout) {
        FCSLOG.info(() -> "Initializing panel for " + latchNameLabel + " timeout=" + timeout);
        SwingUtilities.invokeLater(new GuiInitialization(configInfo, timeout));
    }

    public void updateLatch(StatusDataPublishedByAutochangerLatch status) {
        SwingUtilities.invokeLater(new UpdateLatch(status));
    }

    public void updateController(StatusDataPublishedByEPOSController s) {
        controllerPanel.update(s);
    }

    private void closeButtonActionPerformed(ActionEvent evt) {
        subs.closeLatch(latchName);
    }

    private void openButtonActionPerformed(ActionEvent evt) {
        subs.openLatch(latchName);
    }

    private void abortButtonActionPerformed(ActionEvent evt) {
        subs.abortAction();
    }

    private void initComponents() {
        closeButton = new JButton("Close Latch");
        openButton = new JButton("Open Latch");
        abortButton = new JButton("Abort");

        closeButton.addActionListener((ActionEvent evt) -> {
            closeButtonActionPerformed(evt);
        });

        openButton.addActionListener((ActionEvent evt) -> {
            openButtonActionPerformed(evt);
        });

        abortButton.addActionListener((ActionEvent evt) -> {
            abortButtonActionPerformed(evt);
        });

        /**
         * *****************************************************************************
         ** Sensors Panel
         * *****************************************************************************
         */
        openSensorDS = new DigitalSwitch();
        closeSensorDS = new DigitalSwitch();
        filterEngagedSensorDS = new DigitalSwitch();
        errorDS = new DigitalSwitch();

        latchNameLabel = new JLabel("myLatch");
        openLabel = new JLabel("<html><P align=\"center\">OPEN<br>SENSOR</p></html>");
        closeLabel = new JLabel("<html><P align=\"center\">CLOSE<br>SENSOR</p></html>");
        errorLabel = new JLabel("ERROR");
        filterEngagedLabel = new JLabel("<html><P align=\"center\">FILTER<br>ENGAGED</p></html>");
        lockStatus = new JLabel();
        sensorsPanel = new JPanel();
        sensorsPanel.setBorder(BorderFactory.createEmptyBorder());
        sensorsPanel.setLayout(new GridBagLayout());
        GridBagConstraints gbcs = new GridBagConstraints();
        gbcs.insets = insets_std;
        gbcs.gridx = 0;
        gbcs.gridy = 0;
        sensorsPanel.add(latchNameLabel, gbcs);
        gbcs.gridy++;
        sensorsPanel.add(lockStatus, gbcs);

        gbcs.gridx = 1;
        gbcs.gridy = 0;
        sensorsPanel.add(openSensorDS, gbcs);
        gbcs.gridy++;
        sensorsPanel.add(openLabel, gbcs);

        gbcs.gridx = 2;
        gbcs.gridy = 0;
        sensorsPanel.add(closeSensorDS, gbcs);
        gbcs.gridy++;
        sensorsPanel.add(closeLabel, gbcs);

        gbcs.gridx = 3;
        gbcs.gridy = 0;
        sensorsPanel.add(filterEngagedSensorDS, gbcs);
        gbcs.gridy++;
        sensorsPanel.add(filterEngagedLabel, gbcs);

        gbcs.gridx = 4;
        gbcs.gridy = 0;
        sensorsPanel.add(errorDS, gbcs);
        gbcs.gridy++;
        sensorsPanel.add(errorLabel, gbcs);
        /**
         * *****************************************************************************
         ** end of Sensors Panel
         * *****************************************************************************
         */
        /**
         * *****************************************************************************
         ** Config Parameters Panel
         * *****************************************************************************
         */
        paramPanel = new JPanel();
        paramPanel.setBorder(BorderFactory.createLineBorder(Color.BLACK));
        paramPanel.setLayout(new GridBagLayout());
        GridBagConstraints gbcp = new GridBagConstraints();
        gbcp.insets = insets_std;
        //1st column
        gbcp.gridx = 0;
        gbcp.gridy = 0;
        gbcp.anchor = GridBagConstraints.LINE_START;
        paramPanel.add(new JLabel("timeout to open/close:"), gbcp);
        gbcp.gridy++;
        paramPanel.add(new JLabel("current to open:"), gbcp);
        gbcp.gridy++;
        paramPanel.add(new JLabel("current to close:"), gbcp);
        //2nd column
        gbcp.anchor = GridBagConstraints.LINE_END;
        gbcp.gridx = 1;
        gbcp.gridy = 0;
        paramPanel.add(motionTimeoutLabel, gbcp);
        gbcp.gridy++;
        paramPanel.add(currentToOpenLabel, gbcp);
        gbcp.gridy++;
        paramPanel.add(currentToCloseLabel, gbcp);
        /**
         * *****************************************************************************
         ** end of Config Parameters Panel
         * *****************************************************************************
         */

        /**
         ******************************************************************************
         ** Control Panel
         * *****************************************************************************
         */
        controlPanel = new JPanel();
        controlPanel.setLayout(new java.awt.GridBagLayout());
        GridBagConstraints gbc3 = new java.awt.GridBagConstraints();
        gbc3.anchor = GridBagConstraints.FIRST_LINE_START;
        gbc3.fill = GridBagConstraints.HORIZONTAL;
        gbc3.gridx = 0;
        gbc3.gridy = 0;
        controlPanel.add(closeButton, gbc3);
        gbc3.gridy++;
        controlPanel.add(openButton, gbc3);
        gbc3.gridy++;
        controlPanel.add(abortButton, gbc3);
        /**
         ******************************************************************************
         ** End of Control Panel
         * *****************************************************************************
         */

        /**
         * *****************************************************************************
         ** Whole Panel
         * *****************************************************************************
         */
        setDefaultValues();

        setBorder(BorderFactory.createEtchedBorder());
        setForeground(new Color(204, 204, 255));
        setLayout(new GridBagLayout());
        GridBagConstraints gbc = new GridBagConstraints();
        gbc.insets = insets_big;
        gbc.anchor = GridBagConstraints.CENTER;

        gbc.gridx = 0;
        gbc.gridy = 0;
        add(sensorsPanel, gbc);
        gbc.gridx++;
        add(controllerPanel, gbc);

        gbc.anchor = GridBagConstraints.WEST;
        gbc.gridx = 0;
        gbc.gridy++;
        add(paramPanel, gbc);

        gbc.gridx++;
        gbc.insets = insets_std;
        gbc.anchor = GridBagConstraints.CENTER;
        add(controlPanel, gbc);
    }

    private void setDefaultValues() {
        openSensorDS.setColor(Color.gray);
        closeSensorDS.setColor(Color.gray);
        filterEngagedSensorDS.setColor(Color.gray);
        errorDS.setColor(Color.gray);
        lockStatus.setText(UNKNOWN_STATE);
        motionTimeoutLabel.setText(ZERO_VALUE);
        currentToOpenLabel.setText(ZERO_VALUE);
        currentToCloseLabel.setText(ZERO_VALUE);
    }

    /**
     * Reset components to default values.
     */
    public void resetPanel() {
        setDefaultValues();
        controllerPanel.resetPanel();
    }

    @Override
    public String toString() {
        if (latchName == null) {
            return "Latch";
        } else {
            return latchName;
        }
    }

    public static void main(String[] argv) {

        AutochangerLatchPanel d = new AutochangerLatchPanel("myLatch");
        JFrame frame = new JFrame("Autochanger Latch Panel");
        frame.setContentPane(d);
        frame.pack();
        frame.setVisible(true);
        frame.setDefaultCloseOperation(JFrame.EXIT_ON_CLOSE);

    }

}
