
package org.lsst.ccs.subsystems.fcs.ui.commons;

import java.awt.Color;
import java.awt.Dimension;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.util.Map;
import javax.swing.BorderFactory;
import javax.swing.JButton;
import javax.swing.JFrame;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.SwingUtilities;
import org.lsst.ccs.bus.data.ConfigurationInfo;
import org.lsst.ccs.gconsole.plugins.commandbrowser.HintTextField;
import static org.lsst.ccs.subsystems.fcs.EPOSEnumerations.EposMode.CURRENT;
import static org.lsst.ccs.subsystems.fcs.EPOSEnumerations.EposMode.HOMING;
import static org.lsst.ccs.subsystems.fcs.EPOSEnumerations.EposMode.PROFILE_POSITION;
import static org.lsst.ccs.subsystems.fcs.FCSCst.FCSLOG;
import static org.lsst.ccs.subsystems.fcs.FCSCst.UNKNOWN;
import org.lsst.ccs.subsystems.fcs.StatusDataPublishedByEPOSController;
import org.lsst.ccs.subsystems.fcs.StatusDataPublishedByHardware;
import static org.lsst.ccs.subsystems.fcs.ui.commons.Tools.BIG_FONT;
import static org.lsst.ccs.subsystems.fcs.ui.commons.Tools.greenColor;
import static org.lsst.ccs.subsystems.fcs.ui.commons.Tools.insets_small;
import static org.lsst.ccs.subsystems.fcs.ui.commons.Tools.insets_std;
import org.lsst.ccs.utilities.conv.InputConversionEngine;
import static org.lsst.ccs.subsystems.fcs.ui.commons.Tools.NICE_PINK;
import static org.lsst.ccs.subsystems.fcs.ui.commons.Tools.getShortComponentName;

/**
 * A Panel to display data coming from an EPOS motor controller.
 *
 * @author virieux
 */
public class EPOSControllerPanel extends JPanel {

    private static final long serialVersionUID = 2482213337630248322L;
    private InterfaceGeneralGUI subs;
    private final String fullPathControllerName;
    private final String controllerName;

    /**
     * Variables declaration *
     */
    //general informations panel
    private final CanOpenDevicePanel generalInfoPanel;

    //Fault Panel
    private final JPanel faultPanel = new JPanel();
    private final DigitalSwitch faultSwitch = new DigitalSwitch();
    private final JLabel errorRegisterLabel = new JLabel();
    private final JLabel lastErrorHistoryLabel = new JLabel();

    //mode panel
    private final JPanel modePanel = new JPanel();
    private final JLabel modeLabel = new JLabel("UNKNOWN");

    //control Panel
    private final JPanel controlPanel = new JPanel();
    private final JButton checkFaultButton = new JButton("CheckFault");
    private final JButton faultResetButton = new JButton("FaultReset");
    private final JButton refreshButton = new JButton("Refresh");

    //change EPOS Parameter Panel
    private final JPanel changeParameterPanel = new JPanel();
    private final HintTextField keyArea = new HintTextField("String");
    private final HintTextField valueArea = new HintTextField("int");
    private final JLabel keyLabel = new JLabel("parameter name");
    private final JLabel valueLabel = new JLabel("parameter value");
    private final JButton changeParameterButton = new JButton("ChangeEPOSParameter");
    private final JButton saveParametersButton = new JButton("Save EPOSParameters in Controller CPU");

    //panel to display enable information and enable/disable buttons
    private final JPanel enablePanel = new JPanel();
    private final JLabel enableLabel = new JLabel();
    private final DigitalSwitch enableSwitch = new DigitalSwitch();
    private final JButton enableButton = new JButton("goToOperationEnable");
    private final JButton disableButton = new JButton("goToSwitchOnDisabled");
    private final JLabel brakeLabel = new JLabel();

    //panel for CURRENT mode parameters
    private final EPOSParametersPanel currentParamsPanel;
    //panel for PROFILE_POSITION mode parameters
    private final EPOSParametersPanel profile_positionParamsPanel;
    //panel for HOMING mode parameters
    private final EPOSParametersPanel homingParamsPanel;

    /**
     * A Runnable to initialize this Panel from data coming from the Configuration system.
     */
    private class GuiInitialization implements Runnable {

        private final ConfigurationInfo configInfo;

        public GuiInitialization(ConfigurationInfo configInfo) {
            this.configInfo = configInfo;
        }

        @Override
        public void run() {
            //General Informations Panel
            generalInfoPanel.initializeGui(configInfo);

            //Retrieve this controller configuration data
            Map<String, String> configForName = configInfo.getCurrentValuesFor(fullPathControllerName);
            FCSLOG.fine(fullPathControllerName + " configForName="+configForName);

            //panel for CURRENT mode parameters
            @SuppressWarnings("unchecked")
            Map<String, String> currentParamsMap = (Map<String, String>) InputConversionEngine.convertArgToType(configForName.get("paramsForCurrent"), Map.class);
            currentParamsPanel.initializeGUI(CURRENT.toString(), currentParamsMap);

            //panel for PROFILE_POSITION mode parameters
            @SuppressWarnings("unchecked")
            Map<String, String> profileParamsMap = (Map<String, String>) InputConversionEngine.convertArgToType(configForName.get("paramsForProfilePosition"), Map.class);
            profile_positionParamsPanel.initializeGUI(PROFILE_POSITION.toString(), profileParamsMap);

            //panel for HOMING mode parameters
            @SuppressWarnings("unchecked")
            Map<String, String> homingParamsMap = (Map<String, String>) InputConversionEngine.convertArgToType(configForName.get("paramsForHoming"), Map.class);
            homingParamsPanel.initializeGUI(HOMING.toString(), homingParamsMap);
        }
    }

    /**
     * A Runnable class to update this panel from data published on the STATUS bus
     * by an EPOS Controller.
     */
    private class UpdateController implements Runnable {

        private StatusDataPublishedByEPOSController s;

        public UpdateController(StatusDataPublishedByEPOSController status) {
            this.s = status;
        }

        @Override
        public void run() {
            //General Informations Panel
            generalInfoPanel.updateCanOpenDevice(fullPathControllerName, (StatusDataPublishedByHardware) s);

            //Mode Panel
            if (s.getMode() != null) {
                modeLabel.setText(s.getMode().toString());
            }

            if (s.isInitialized()) {
                //Fault Panel
                faultSwitch.setColor(s.isInError() ? Color.RED : greenColor);
                errorRegisterLabel.setText(s.getErrorRegister());
                //error History
                if (s.getErrorHistoryNB() == 0) {
                    lastErrorHistoryLabel.setText("no error in history");
                } else {
                    StringBuilder sb = new StringBuilder("<html>");
                    sb.append("<center>");
                    sb.append("Errors nb : ");
                    sb.append(s.getErrorHistoryNB());
                    sb.append("<br>");
                    sb.append("Last error");
                    sb.append("<br>");
                    sb.append("0x");
                    sb.append(Integer.toHexString(s.getLastErrorCode()));
                    sb.append("<br>");
                    sb.append(s.getLastErrorName());
                    sb.append("</center>");
                    sb.append("</html>");
                    lastErrorHistoryLabel.setText(sb.toString());
                }
            } else if (s.isBooted()){
                faultSwitch.setColor(Color.ORANGE);
                errorRegisterLabel.setText("Not initialized");
            } else {
                faultSwitch.setColor(Color.RED);
                errorRegisterLabel.setText("Not booted");
            }

            //enablePanel
            if (s.isBooted() && s.getState() != null) {
                enableLabel.setText(s.getState().name());
                enableSwitch.setColor(s.getState().getColor());
            } else if (!s.isBooted()) {
                enableLabel.setText("not booted");
                enableSwitch.setColor(Color.RED);
            }

            if (s.isBooted() && s.isControllerWithBrake() && s.isBrakeActivated()) {
                brakeLabel.setText("<HTML><CENTER>BRAKE<BR>ACTIVATED</CENTER></HTML>");
            } else if (s.isControllerWithBrake() && !s.isBrakeActivated()) {
                brakeLabel.setText("<HTML><CENTER>BRAKE<BR>RELEASED</CENTER></HTML>");
            } else {
                brakeLabel.setText("");
            }
        }
    }

    public EPOSControllerPanel(String fullPathControllerName) {
        this.fullPathControllerName = fullPathControllerName;
        this.generalInfoPanel = new CanOpenDevicePanel(fullPathControllerName);
        controllerName = getShortComponentName(fullPathControllerName);
        currentParamsPanel = new EPOSParametersPanel(controllerName);
        profile_positionParamsPanel = new EPOSParametersPanel(controllerName);
        homingParamsPanel = new EPOSParametersPanel(controllerName);
        initComponents();
    }

    /**
     * To initialize the subsystem for this EPOSControllerPanel.
     * @param newSubs
     */
    public void setSubsystem(InterfaceGeneralGUI newSubs) {
        this.subs = newSubs;
    }

    public void initializeGui(ConfigurationInfo configInfo) {
        SwingUtilities.invokeLater(new GuiInitialization(configInfo));
    }

    public void updateController(String name, StatusDataPublishedByEPOSController status) {
        if (controllerName.equals(name)) {
            SwingUtilities.invokeLater(new UpdateController(status));
        }
    }

    private void initComponents() {

        setDefaultValues();
        //Buttons
        checkFaultButton.addActionListener((java.awt.event.ActionEvent evt) -> {
            checkFaultActionPerformed(evt);
        });

        faultResetButton.addActionListener((java.awt.event.ActionEvent evt) -> {
            faultResetActionPerformed(evt);
        });

        refreshButton.addActionListener((java.awt.event.ActionEvent evt) -> {
            refreshActionPerformed();
        });

        changeParameterButton.addActionListener((java.awt.event.ActionEvent evt) -> {
            changeParameterActionPerformed();
        });

        saveParametersButton.addActionListener((java.awt.event.ActionEvent evt) -> {
            saveParametersActionPerformed();
        });

        enableButton.addActionListener((java.awt.event.ActionEvent evt) -> {
            enableActionPerformed();
        });

        disableButton.addActionListener((java.awt.event.ActionEvent evt) -> {
            disableActionPerformed();
        });

        /**
         ******************************************************************************
         ** Mode Panel
         * *****************************************************************************
         */
        modePanel.setBorder(BorderFactory.createTitledBorder(BorderFactory.createLineBorder(NICE_PINK),"EPOS mode"));
        modePanel.setPreferredSize(new Dimension(160, 50));
        modePanel.setLayout(new GridBagLayout());
        GridBagConstraints gbc2 = new GridBagConstraints();
        gbc2.insets = insets_std;
        //first column
        gbc2.gridx = 0;
        gbc2.gridy = 0;
        gbc2.anchor = GridBagConstraints.LINE_START;
        modeLabel.setFont(BIG_FONT);
        modePanel.add(modeLabel, gbc2);
        /**
         ******************************************************************************
         ** end of Mode Panel
         * *****************************************************************************
         */
        /**
         ******************************************************************************
         ** Fault Panel
         ******************************************************************************
         */
        faultPanel.setBorder(BorderFactory.createTitledBorder("Error"));
        faultPanel.setLayout(new GridBagLayout());
        GridBagConstraints gbc1 = new GridBagConstraints();
        gbc1.insets = insets_std;
        gbc1.gridx = 0;
        gbc1.gridy = 0;
        gbc1.anchor = GridBagConstraints.CENTER;
        faultPanel.add(faultSwitch, gbc1);
        gbc1.gridy++;
        faultSwitch.setColor(Color.GRAY);
        errorRegisterLabel.setFont(BIG_FONT);
        faultPanel.add(new JLabel("Error register"), gbc1);
        gbc1.gridy++;
        faultPanel.add(errorRegisterLabel, gbc1);
        gbc1.gridy++;
        faultPanel.add(lastErrorHistoryLabel, gbc1);
        /**
         ******************************************************************************
         ** end of Fault Panel
         * *****************************************************************************
         */
        /**
         ******************************************************************************
         ** Enable Panel
         ******************************************************************************
         */
        enablePanel.setBorder(BorderFactory.createTitledBorder(BorderFactory.createLineBorder(Color.BLUE),"Enable/Disable"));
        enablePanel.setLayout(new GridBagLayout());
        GridBagConstraints gbc5 = new GridBagConstraints();
        gbc5.insets = insets_std;
        gbc5.gridx = 0;
        gbc5.gridy = 0;
        enablePanel.add(enableSwitch, gbc5);
        gbc5.gridx++;
        enablePanel.add(enableLabel, gbc5);
        gbc5.gridx = 0;
        gbc5.gridy++;
        gbc5.insets = insets_small;
        enablePanel.add(enableButton, gbc5);
        gbc5.gridx++;
        enablePanel.add(brakeLabel, gbc5);
        gbc5.gridx = 0;
        gbc5.gridy++;
        enablePanel.add(disableButton, gbc5);

        /**
         ******************************************************************************
         ** End of Enable Panel
         ******************************************************************************
         */
        /**
         * ****************************************************************************
         ** Control Panel
         * *****************************************************************************
         */
        controlPanel.setLayout(new GridBagLayout());
        GridBagConstraints gbc4 = new GridBagConstraints();
        gbc4.anchor = GridBagConstraints.FIRST_LINE_START;
        gbc4.fill = GridBagConstraints.HORIZONTAL;
        gbc4.insets = insets_small;
        gbc4.gridx = 0;
        gbc4.gridy = 0;
        controlPanel.add(refreshButton, gbc4);
        gbc4.gridy++;
        controlPanel.add(checkFaultButton, gbc4);
        gbc4.gridy++;
        controlPanel.add(faultResetButton, gbc4);
        /**
         ******************************************************************************
         ** End of Control Panel
         ******************************************************************************
         */

        /**
         ******************************************************************************
         ** Change Parameter Panel
         ******************************************************************************
         */
        changeParameterPanel.setBorder(BorderFactory.createTitledBorder(BorderFactory.createLineBorder(NICE_PINK),
                "change EPOS parameters"));
        changeParameterPanel.setLayout(new GridBagLayout());
        GridBagConstraints gbc3 = new GridBagConstraints();
        gbc3.fill = GridBagConstraints.HORIZONTAL;
        gbc3.anchor = GridBagConstraints.LINE_START;
        gbc3.gridx = 0;
        gbc3.gridy = 0;
        changeParameterPanel.add(keyLabel, gbc3);
        gbc3.gridx++;
        keyArea.setColumns(20);
        gbc3.anchor = GridBagConstraints.LINE_END;
        changeParameterPanel.add(keyArea, gbc3);
        gbc3.gridx = 0;
        gbc3.gridy++;
        gbc3.anchor = GridBagConstraints.LINE_START;
        changeParameterPanel.add(valueLabel, gbc3);
        gbc3.gridx++;
        gbc3.anchor = GridBagConstraints.LINE_END;
        changeParameterPanel.add(valueArea, gbc3);
        gbc3.gridx = 0;
        gbc3.gridy++;
        gbc3.gridwidth = 2;
        gbc3.fill = GridBagConstraints.FIRST_LINE_END;
        gbc3.insets = insets_small;
        changeParameterPanel.add(changeParameterButton, gbc3);
        gbc3.gridy++;
        changeParameterPanel.add(saveParametersButton, gbc3);
        /**
         ******************************************************************************
         ** end of Change Parameter Panel
         ******************************************************************************
         */
        /**
         ******************************************************************************
         ** Whole Panel
         * *****************************************************************************
         */
        setBorder(javax.swing.BorderFactory.createLineBorder(Color.GREEN));
        setLayout(new java.awt.GridBagLayout());
        GridBagConstraints gbc = new GridBagConstraints();
        gbc.insets = insets_std;
        gbc.anchor = GridBagConstraints.NORTHWEST;

        gbc.gridx = 0;
        gbc.gridy = 0;
        gbc.gridwidth = 2;
        add(generalInfoPanel, gbc);

        gbc.gridx = 2;
        gbc.gridwidth = 1;
        add(faultPanel, gbc);

        gbc.gridx++;
        add(controlPanel, gbc);

        gbc.gridx = 0;
        gbc.gridy = 1;
        add(modePanel, gbc);
        gbc.gridy = 2;
        add(enablePanel, gbc);

        gbc.gridy = 1;
        gbc.gridx++;
        gbc.gridwidth = 3;
        gbc.gridheight = 2;
        gbc.fill = GridBagConstraints.HORIZONTAL;
        add(changeParameterPanel, gbc);

        gbc.gridx = 0;
        gbc.gridy = 4;
        gbc.gridwidth = 4;
        gbc.gridheight = 1;
        add(currentParamsPanel, gbc);
        gbc.gridy++;
        add(profile_positionParamsPanel, gbc);
        gbc.gridy++;
        add(homingParamsPanel, gbc);

        /**
         ******************************************************************************
         ** end of Whole Panel
         * *****************************************************************************
         */
    }

    private void setDefaultValues() {
        enableSwitch.setColor(Color.GRAY);
        faultSwitch.setColor(Color.GRAY);
        errorRegisterLabel.setText(UNKNOWN);
        lastErrorHistoryLabel.setText(UNKNOWN);
        enableLabel.setText(UNKNOWN);
        modeLabel.setText(UNKNOWN);
        brakeLabel.setText("");
    }

    /**
     * Reset to default values;
     */
    public void resetPanel() {
        setDefaultValues();
        generalInfoPanel.resetPanel();
        this.currentParamsPanel.resetPanel();
        this.homingParamsPanel.resetPanel();
        this.profile_positionParamsPanel.resetPanel();
    }

    private void checkFaultActionPerformed(java.awt.event.ActionEvent evt) {
        subs.sendCommandSwingWorker("checkFault", 1000, fullPathControllerName);
    }

    private void faultResetActionPerformed(java.awt.event.ActionEvent evt) {
        subs.sendCommandSwingWorker("faultReset", 1000, fullPathControllerName);
    }

    private void refreshActionPerformed() {
        subs.sendCommandSwingWorker("publishData", 1000, fullPathControllerName);
    }

    private void changeParameterActionPerformed() {
        if (valueArea.getText() == null ) throw new IllegalArgumentException(valueArea.getText() + " is not a valid value."
                + " Please enter a decimal value.");
        subs.sendCommandSwingWorker("changeEPOSParameter " + keyArea.getText() + " " + valueArea.getText(),
                1000, fullPathControllerName);
    }

    private void saveParametersActionPerformed() {
        subs.sendCommandSwingWorker("saveParameters", 1000, fullPathControllerName);
    }

    private void enableActionPerformed() {
        subs.sendCommandSwingWorker("goToOperationEnable", 1000, fullPathControllerName);
    }

    private void disableActionPerformed() {
        subs.sendCommandSwingWorker("goToSwitchOnDisabled", 1000, fullPathControllerName);
    }

    @Override
    public String toString() {
        if (fullPathControllerName == null) {
            return "Unamed Controller Panel";

        } else {
            int ind = fullPathControllerName.lastIndexOf("/");
            return ind >=0 ? fullPathControllerName.substring(ind+1) : fullPathControllerName;
        }
    }

    public static void main(String[] argv) {

        EPOSControllerPanel p = new EPOSControllerPanel("myController");

        JFrame frame = new JFrame("EPOS Controller");
        p.setDefaultValues();
        frame.setContentPane(p);
        frame.pack();
        frame.setVisible(true);
        frame.setDefaultCloseOperation(JFrame.EXIT_ON_CLOSE);
    }
}
