
package org.lsst.ccs.subsystems.fcs.autochanger.ui;

import java.awt.Color;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.event.ActionEvent;
import java.util.Map;
import javax.swing.BorderFactory;
import javax.swing.JButton;
import javax.swing.JFrame;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.SwingUtilities;
import org.lsst.ccs.bus.data.ConfigurationInfo;
import static org.lsst.ccs.subsystems.fcs.FCSCst.AC_CLAMPXMINUSCONTROLLER_NAME;
import static org.lsst.ccs.subsystems.fcs.FCSCst.AC_CLAMPXPLUSCONTROLLER_NAME;
import static org.lsst.ccs.subsystems.fcs.FCSCst.AC_CLAMPYMINUSCONTROLLER_NAME;
import static org.lsst.ccs.subsystems.fcs.FCSCst.FCSLOG;
import org.lsst.ccs.subsystems.fcs.FcsEnumerations.LockStatus;
import org.lsst.ccs.subsystems.fcs.StatusDataPublishedByAutochangerOnlineClamp;
import org.lsst.ccs.subsystems.fcs.StatusDataPublishedByEPOSController;
import org.lsst.ccs.subsystems.fcs.ui.commons.ControllerStatePanel;
import org.lsst.ccs.subsystems.fcs.ui.commons.DigitalSwitch;
import static org.lsst.ccs.subsystems.fcs.ui.commons.Tools.UNKNOWN_STATE;
import static org.lsst.ccs.subsystems.fcs.ui.commons.Tools.ZERO_VALUE;
import static org.lsst.ccs.subsystems.fcs.ui.commons.Tools.insets_big;
import static org.lsst.ccs.subsystems.fcs.ui.commons.Tools.insets_std;

/**
 *
 * @author virieux
 */
public class AutochangerOnlineClampPanel extends JPanel {

    private static final long serialVersionUID = 7563287373167394297L;
    private final String clampName;

    // Variables declaration
    private InterfaceAutochangerGUI subs;
    private DigitalSwitch closeSensorDS;
    private DigitalSwitch closeSensorCDS;
    private DigitalSwitch openSensorDS;
    private DigitalSwitch openSensorCDS;
    private DigitalSwitch errorDS;

    private JLabel clampNameLabel;
    private JLabel closeLabel;
    private JLabel openLabel;
    private JLabel errorLabel;

    private JLabel lockStatus;

    //Config Parameters Panel
    private JPanel paramPanel;
    private final JLabel initialCurrentToCloseLabel = new JLabel();
    private final JLabel finalCurrentToCloseLabel = new JLabel();
    private final JLabel currentToOpenLabel = new JLabel();
    private final JLabel currentToClampLabel = new JLabel();
    private final JLabel incrementCurrentToOpenLabel = new JLabel();
    private final JLabel incrementCurrentToClampLabel = new JLabel();
    private final JLabel incrementCurrentToCloseLabel = new JLabel();
    private final JLabel targetPositionToOpenLabel = new JLabel();
    private final JLabel targetPositionToCloseLabel = new JLabel();

    //Controller State Panel
    private final ControllerStatePanel controllerPanel;

    //control Panel
    private JPanel controlPanel;
    private JButton homingButton = new JButton("Open and Homing");
    private JButton openButton = new JButton("Open");
    private JButton closeButton = new JButton("Close");
    private JButton lockButton = new JButton("Lock");
    private JButton unlockButton = new JButton("Unlock");
    private JButton abortButton = new JButton("Abort");
    // End of variables declaration

    /* sent current panel*/
    private JPanel currentAndPositionPanel;
    private final JLabel homingLabel = new JLabel();
    private final JLabel sentCurrentLabel = new JLabel();

    /**
     *
     * A Runnable to initialize this Panel from configuration data.
     */
    private class GuiInitialization implements Runnable {

        private final ConfigurationInfo configInfo;

        public GuiInitialization(ConfigurationInfo configInfo) {
            this.configInfo = configInfo;
        }

        @Override
        public void run() {
            //Retrieve the configuration data for my name.
            Map<String, String> configForName = configInfo.getCurrentValuesFor(clampName);
            FCSLOG.fine(clampName + " configForName="+configForName);

            //Config Panel
            currentToOpenLabel.setText(configForName.get("currentToOpen"));
            initialCurrentToCloseLabel.setText(configForName.get("initialCurrentToClose"));
            finalCurrentToCloseLabel.setText(configForName.get("finalCurrentToClose"));
            currentToClampLabel.setText(configForName.get("currentToClamp"));
            incrementCurrentToOpenLabel.setText(configForName.get("incrementCurrentToOpen"));
            incrementCurrentToCloseLabel.setText(configForName.get("incrementCurrentToClose"));
            incrementCurrentToClampLabel.setText(configForName.get("incrementCurrentToClamp"));
            targetPositionToOpenLabel.setText(configForName.get("targetPositionToOpen"));
            targetPositionToCloseLabel.setText(configForName.get("targetPositionToClose"));
        }
    }


    /**
     *
     * A Runnable to Update this Panel from status data published on STATUS bus.
     */
    class UpdateClamp implements Runnable {

        private StatusDataPublishedByAutochangerOnlineClamp s;

        public UpdateClamp(StatusDataPublishedByAutochangerOnlineClamp status) {
            this.s = status;
        }

        @Override
        public void run() {
            lockStatus.setText(s.getLockStatus().getText());
            lockStatus.setForeground(s.getLockStatus().getColor());
            closeSensorDS.updateColor(s.isLockSensorInError(), s.getLockSensorValue());
            openSensorDS.updateColor(s.isUnlockSensorInError(), s.getUnlockSensorValue());
            boolean inTravel = s.getLockStatus().equals(LockStatus.INTRAVEL);
            boolean unknown = s.getLockStatus().equals(LockStatus.UNKNOWN);
            if (s.isInError() || inTravel || unknown) {
                errorDS.setColor(s.getLockStatus().getColor());
                errorLabel.setText(s.getLockStatus().getText());
            } else {
                errorDS.setColor(LockStatus.NOERROR.getColor());
                errorLabel.setText("<html><p align=\"center\">NO<br>ERROR</p></html>");
            }

            //current panel
            if (s.isHomingDone()) {
                homingLabel.setText("DONE");
                homingLabel.setForeground(Color.green);
            } else {
                homingLabel.setText("NOT DONE");
                homingLabel.setForeground(Color.red);
            }
            sentCurrentLabel.setText(String.valueOf(s.getSentCurrent()));
        }
    }

    /**
     * Create a new Panel with a clamp name.
     * @param clampName
     */
    public AutochangerOnlineClampPanel(String clampName) {
        this.clampName = clampName;
        controllerPanel = new ControllerStatePanel(getControllerName(clampName), true);
        initComponents();
    }

    private static String getControllerName(String clampName) {
        if (clampName.contains("Xminus")) {
            return AC_CLAMPXMINUSCONTROLLER_NAME;
        } else if (clampName.contains("Xplus")) {
            return AC_CLAMPXPLUSCONTROLLER_NAME;
        } else {
            return AC_CLAMPYMINUSCONTROLLER_NAME;
        }
    }

    public void setSubsystem(InterfaceAutochangerGUI subs) {
        this.subs = subs;
    }


    public void initializeGui(ConfigurationInfo configInfo) {
        SwingUtilities.invokeLater(new GuiInitialization(configInfo));
    }


    public void updateClamp(StatusDataPublishedByAutochangerOnlineClamp status) {
        SwingUtilities.invokeLater(new UpdateClamp(status));
    }

    public void updateController(StatusDataPublishedByEPOSController s) {
        controllerPanel.update(s);
    }

    private void initComponents() {

        /**
         ******************************************************************************
         ** Buttons initialization
         * *****************************************************************************
         */
        homingButton.addActionListener((ActionEvent evt) -> {
            homingButtonActionPerformed(evt);
        });

        openButton.addActionListener((ActionEvent evt) -> {
            openButtonActionPerformed(evt);
        });

        closeButton.addActionListener((ActionEvent evt) -> {
            closeButtonActionPerformed(evt);
        });

        lockButton.addActionListener((ActionEvent evt) -> {
            lockButtonActionPerformed(evt);
        });

        unlockButton.addActionListener((ActionEvent evt) -> {
            unlockButtonActionPerformed(evt);
        });

        abortButton.addActionListener((ActionEvent evt) -> {
            abortButtonActionPerformed(evt);
        });
        /**
         ******************************************************************************
         ** end of Buttons initialization
         * *****************************************************************************
         */

        closeSensorDS = new DigitalSwitch();
        closeSensorCDS = new DigitalSwitch();
        openSensorDS = new DigitalSwitch();
        openSensorCDS = new DigitalSwitch();
        errorDS = new DigitalSwitch();

        int ind = clampName.lastIndexOf("/");
        String clampNamePublished = ind >= 0 ? clampName.substring(ind + 1) : clampName;
        clampNameLabel = new JLabel(clampNamePublished);
        openLabel = new JLabel("<html><P align=\"center\">OPEN<br>SENSOR</p></html>");
        closeLabel = new JLabel("<html><P align=\"center\">CLOSE<br>SENSOR</p></html>");
        errorLabel = new JLabel();
        lockStatus = new JLabel();

        /**
         * *****************************************************************************
         ** Config Parameters Panel
         * *****************************************************************************
         */
        paramPanel = new JPanel();
        paramPanel.setBorder(BorderFactory.createLineBorder(Color.BLACK));
        paramPanel.setLayout(new GridBagLayout());
        GridBagConstraints gbcp = new GridBagConstraints();
        gbcp.insets = insets_std;
        //1st column
        gbcp.gridx = 0;
        gbcp.gridy = 0;
        gbcp.anchor = GridBagConstraints.LINE_START;
        paramPanel.add(new JLabel("currentToOpen:"), gbcp);
        gbcp.gridy++;
        paramPanel.add(new JLabel("initialCurrentToClose:"), gbcp);
        gbcp.gridy++;
        paramPanel.add(new JLabel("finalCurrentToClose:"), gbcp);
        gbcp.gridy++;
        paramPanel.add(new JLabel("currentToClamp:"), gbcp);
        gbcp.gridy++;
        paramPanel.add(new JLabel("positionOpen:"), gbcp);
        gbcp.gridy++;
        paramPanel.add(new JLabel("positionClose:"), gbcp);

        //2nd column
        gbcp.anchor = GridBagConstraints.LINE_END;
        gbcp.gridx = 1;
        gbcp.gridy = 0;
        paramPanel.add(currentToOpenLabel, gbcp);
        gbcp.gridy++;
        paramPanel.add(initialCurrentToCloseLabel, gbcp);
        gbcp.gridy++;
        paramPanel.add(finalCurrentToCloseLabel, gbcp);
        gbcp.gridy++;
        paramPanel.add(currentToClampLabel, gbcp);
        gbcp.gridy++;
        paramPanel.add(targetPositionToOpenLabel, gbcp);
        gbcp.gridy++;
        paramPanel.add(targetPositionToCloseLabel, gbcp);
        /**
         * *****************************************************************************
         ** end of Config Parameters Panel
         * *****************************************************************************
         */

        /**
         * *****************************************************************************
         ** Control Panel
         * *****************************************************************************
         */
        controlPanel = new JPanel();
        controlPanel.setLayout(new GridBagLayout());
        GridBagConstraints gbcc = new GridBagConstraints();
        gbcc.anchor = GridBagConstraints.FIRST_LINE_START;
        gbcc.fill = GridBagConstraints.HORIZONTAL;
        gbcc.gridx = 0;
        gbcc.gridy = 0;
        controlPanel.add(homingButton, gbcc);
        gbcc.gridy++;
        controlPanel.add(closeButton, gbcc);
        gbcc.gridy++;
        controlPanel.add(openButton, gbcc);
        gbcc.gridy++;
        controlPanel.add(lockButton, gbcc);
        gbcc.gridy++;
        controlPanel.add(unlockButton, gbcc);
        gbcc.gridy++;
        controlPanel.add(abortButton, gbcc);
        /**
         * *****************************************************************************
         ** end of Control Panel
         * *****************************************************************************
         */

        /**
         * *****************************************************************************
         ** Current and Position Panel
         * *****************************************************************************
         */
        currentAndPositionPanel = new JPanel();
        currentAndPositionPanel.setLayout(new GridBagLayout());
        gbcp = new GridBagConstraints();
        gbcp.insets = insets_std;
        //1st column
        gbcp.gridx = 0;
        gbcp.gridy = 0;
        gbcp.anchor = GridBagConstraints.LINE_START;
        currentAndPositionPanel.add(new JLabel("homing :"), gbcp);
        gbcp.gridy++;
        currentAndPositionPanel.add(new JLabel("current sent :"), gbcp);

        //2nd column
        gbcp.anchor = GridBagConstraints.LINE_END;
        gbcp.gridx = 1;
        gbcp.gridy = 0;
        currentAndPositionPanel.add(homingLabel, gbcp);
        gbcp.gridy++;
        currentAndPositionPanel.add(sentCurrentLabel, gbcp);
        /**
         * *****************************************************************************
         ** end of Current Panel
         * *****************************************************************************
         */

        /**
         * *****************************************************************************
         ** Whole Panel
         * *****************************************************************************
         */
        setDefaultValues();

        setBorder(BorderFactory.createEtchedBorder());
        setForeground(new Color(204, 204, 255));
        setLayout(new GridBagLayout());
        GridBagConstraints gbc = new GridBagConstraints();
        gbc.insets = insets_big;

        gbc.gridx = 0;
        gbc.gridy = 0;
        add(clampNameLabel, gbc);
        gbc.gridy++;
        add(lockStatus, gbc);

        gbc.gridx = 1;
        gbc.gridy = 0;
        add(openSensorDS, gbc);
        gbc.gridy++;
        add(openLabel, gbc);

        gbc.gridx = 2;
        gbc.gridy = 0;
        add(closeSensorDS, gbc);
        gbc.gridy++;
        add(closeLabel, gbc);

        gbc.gridx = 3;
        gbc.gridy = 0;
        add(errorDS, gbc);
        gbc.gridy++;
        add(errorLabel, gbc);

        gbc.gridx = 5;
        gbc.gridy = 0;
        gbc.gridheight = 2;
        add(controllerPanel, gbc);

        gbc.gridx = 0;
        gbc.gridy = 2;
        gbc.gridheight = 1;
        gbc.gridwidth = 2;
        add(currentAndPositionPanel, gbc);

        gbc.gridx = 0;
        gbc.gridy = 3;
        add(paramPanel, gbc);

        gbc.gridx = 5;
        gbc.gridy = 2;
        gbc.gridheight = 2;
        add(controlPanel, gbc);
    }

    private void setDefaultValues() {
        closeSensorDS.setColor(Color.gray);
        closeSensorCDS.setColor(Color.gray);
        openSensorDS.setColor(Color.gray);
        openSensorCDS.setColor(Color.gray);
        errorDS.setColor(Color.gray);
        errorLabel.setText("ERROR");
        lockStatus.setText(UNKNOWN_STATE);
        initialCurrentToCloseLabel.setText(ZERO_VALUE);
        finalCurrentToCloseLabel.setText(ZERO_VALUE);
        currentToOpenLabel.setText(ZERO_VALUE);
        currentToClampLabel.setText(ZERO_VALUE);
        sentCurrentLabel.setText(ZERO_VALUE);
        incrementCurrentToOpenLabel.setText(ZERO_VALUE);
        incrementCurrentToCloseLabel.setText(ZERO_VALUE);
        incrementCurrentToClampLabel.setText(ZERO_VALUE);
        targetPositionToOpenLabel.setText(ZERO_VALUE);
        targetPositionToCloseLabel.setText(ZERO_VALUE);
        homingLabel.setText("NOT DONE");
        homingLabel.setForeground(Color.red);
        controllerPanel.resetPanel();
    }

    /**
     * Reset components to default values.
     */
    public void resetPanel() {
        setDefaultValues();
    }

    private void homingButtonActionPerformed(ActionEvent evt) {
        subs.homingOnlineClamp(clampName);
    }

    private void openButtonActionPerformed(ActionEvent evt) {
        subs.openOnlineClamp(clampName);
    }

    private void closeButtonActionPerformed(ActionEvent evt) {
        subs.closeOnlineClamp(clampName);
    }

    private void lockButtonActionPerformed(ActionEvent evt) {
        subs.lockOnlineClamp(clampName);
    }

    private void unlockButtonActionPerformed(ActionEvent evt) {
        subs.unlockOnlineClamp(clampName);
    }

    private void abortButtonActionPerformed(ActionEvent evt) {
        subs.abortAction();
    }

    @Override
    public String toString() {
        if (clampName == null) {
            return "ONLINE clamp";
        } else {
            return clampName;
        }
    }

    public static void main(String[] argv) {

        AutochangerOnlineClampPanel d = new AutochangerOnlineClampPanel("clampName");
        JFrame frame = new JFrame("Autochanger OnlineClamp Panel");
        frame.setContentPane(d);
        frame.pack();
        frame.setVisible(true);
        frame.setDefaultCloseOperation(JFrame.EXIT_ON_CLOSE);

    }

}
