package org.lsst.ccs.subsystems.fcs.autochanger.ui;

import java.io.Serializable;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;
import javax.swing.SwingUtilities;
import org.lsst.ccs.bus.data.ConfigurationInfo;
import org.lsst.ccs.bus.data.KeyValueData;
import org.lsst.ccs.bus.data.KeyValueDataList;
import static org.lsst.ccs.subsystems.fcs.FCSCst.AC_STRAIN_GAUGE_NAME;
import static org.lsst.ccs.subsystems.fcs.FCSCst.AC_TRUCKS_NAME;
import static org.lsst.ccs.subsystems.fcs.FCSCst.CHANGER_TCPPROXY_NAME;
import org.lsst.ccs.subsystems.fcs.StatusDataPublishedByAutoChangerTrucks;
import org.lsst.ccs.subsystems.fcs.StatusDataPublishedByAutochangerLatch;
import org.lsst.ccs.subsystems.fcs.StatusDataPublishedByAutochangerOnlineClamp;
import org.lsst.ccs.subsystems.fcs.StatusDataPublishedByAutochangerThreeClamps;
import org.lsst.ccs.subsystems.fcs.StatusDataPublishedByAutochangerTruck;
import org.lsst.ccs.subsystems.fcs.StatusDataPublishedByAutochangerTwoLatches;
import org.lsst.ccs.subsystems.fcs.StatusDataPublishedByEPOSController;
import org.lsst.ccs.subsystems.fcs.StatusDataPublishedByHardware;
import org.lsst.ccs.subsystems.fcs.StatusDataPublishedByPlutoGateway;
import org.lsst.ccs.subsystems.fcs.ui.commons.CanOpenDevicePanel;
import org.lsst.ccs.subsystems.fcs.ui.commons.EPOSControllerPanel;
import org.lsst.ccs.subsystems.fcs.ui.commons.GatherPanel;
import org.lsst.ccs.subsystems.fcs.ui.commons.PlutoGatewayPanel;
import static org.lsst.ccs.subsystems.fcs.FCSCst.AC_PLUTOGATEWAY_NAME;
import static org.lsst.ccs.subsystems.fcs.ui.commons.Tools.getShortComponentName;

/**
 * This panel gathers all the panels needed for the autochanger GUI.
 * Except the subsystemPanel which is handled by the GeneralPanel.
 * It takes care of the initialization of the panels and the update of the panels.
 *
 * @author virieux
 */
public class AutochangerGatherPanel extends GatherPanel {
    private static final long serialVersionUID = -975949078768889254L;

    protected InterfaceAutochangerGUI subs;

    //CANopen devices panels
    protected EPOSControllerPanel acTruckXminusControllerPanel;
    protected EPOSControllerPanel acTruckXplusControllerPanel;
    protected EPOSControllerPanel latchXminusControllerPanel;
    protected EPOSControllerPanel latchXplusControllerPanel;
    protected EPOSControllerPanel onlineClampXminusControllerPanel;
    protected EPOSControllerPanel onlineClampXplusControllerPanel;
    protected EPOSControllerPanel onlineClampYminusControllerPanel;
    protected PlutoGatewayPanel plutoGatewayPanel;
    protected CanOpenDevicePanel onlineStrainGaugePanel;

    private EPOSControllerPanel[] controllerPanelsList;

    //Exchanger hardware panels
    protected AutochangerThreeOnlineClampsPanel onlineClampsPanel;


    private AutochangerTwoTrucksPanel acTrucksPanel;
    private AutochangerTwoLatchesPanel acLatchesPanel = new AutochangerTwoLatchesPanel();
    private AutochangerThreeOnlineClampsPanel acClampsPanel = new AutochangerThreeOnlineClampsPanel();
    private final AutochangerGeneralViewPanel acGeneralViewPanel = new AutochangerGeneralViewPanel();

    private PLCAutochangerPanel plcAutochangerPanel;

    public AutochangerGatherPanel(String tcpProxyName) {
        super(tcpProxyName);
        initPanels();
        setName("AutochangerGatherPanel");
    }

    public AutochangerGatherPanel(InterfaceAutochangerGUI subs, String tcpProxyName) {
        super(tcpProxyName);
        initPanels();
        this.subs = subs;
        this.acGeneralViewPanel.setSubsystem(subs);
        this.acClampsPanel.setSubsystem(subs);
        this.acLatchesPanel.setSubsystem(subs);
        this.acTrucksPanel.setSubsystem(subs);
        this.acTruckXminusControllerPanel.setSubsystem(subs);
        this.acTruckXplusControllerPanel.setSubsystem(subs);
        this.latchXminusControllerPanel.setSubsystem(subs);
        this.latchXplusControllerPanel.setSubsystem(subs);
        this.onlineClampXminusControllerPanel.setSubsystem(subs);
        this.onlineClampXplusControllerPanel.setSubsystem(subs);
        this.onlineClampYminusControllerPanel.setSubsystem(subs);
        setName("AutochangerGatherPanel");
    }



    public AutochangerTwoTrucksPanel getAcTrucksPanel() {
        return acTrucksPanel;
    }

    public AutochangerTwoLatchesPanel getAcLatchesPanel() {
        return acLatchesPanel;
    }

    public AutochangerThreeOnlineClampsPanel getAcClampsPanel() {
        return acClampsPanel;
    }

    public AutochangerGeneralViewPanel getAcGeneralViewPanel() {
        return acGeneralViewPanel;
    }

    public EPOSControllerPanel getAcTruckXminusControllerPanel() {
        return acTruckXminusControllerPanel;
    }

    public EPOSControllerPanel getAcTruckXplusControllerPanel() {
        return acTruckXplusControllerPanel;
    }

    public EPOSControllerPanel getLatchXminusControllerPanel() {
        return latchXminusControllerPanel;
    }

    public EPOSControllerPanel getLatchXplusControllerPanel() {
        return latchXplusControllerPanel;
    }

    public EPOSControllerPanel getOnlineClampXminusControllerPanel() {
        return onlineClampXminusControllerPanel;
    }

    public EPOSControllerPanel getOnlineClampXplusControllerPanel() {
        return onlineClampXplusControllerPanel;
    }

    public EPOSControllerPanel getOnlineClampYminusControllerPanel() {
        return onlineClampYminusControllerPanel;
    }

    public PlutoGatewayPanel getPlutoGatewayPanel() {
        return plutoGatewayPanel;
    }

    public CanOpenDevicePanel getOnlineStrainGaugePanel() {
        return onlineStrainGaugePanel;
    }

    public PLCAutochangerPanel getPLCAutochangerPanel() {
        return plcAutochangerPanel;
    }

    @Override
    public void initializeGUIWithConfigInfo(ConfigurationInfo configInfo,
            List<String> hardwareNames) {
        super.initializeGUIWithConfigInfo(configInfo, hardwareNames);
        SwingUtilities.invokeLater(new GuiInitialization(configInfo));
    }

    @Override
    public void initializeGuiWithAutochangerSensorsList(ConfigurationInfo configInfo, List<String> sensorsList) {
        plutoGatewayPanel.initializeGuiWithSensorsList(configInfo, sensorsList);
    }

    /**
     * Update GUI when a StatusConfigurationInfo message is received on the status bus.
     * @param configInfo
     */
    @Override
    public void updateFromConfigurationInfo(ConfigurationInfo configInfo) {
        SwingUtilities.invokeLater(new GuiInitialization(configInfo));
    }

    @Override
    public void updateFromStatusData(KeyValueData data) {

        if (CHANGER_TCPPROXY_NAME.equals(data.getKey())) {
            updateCanOpenHardwarePanel((int)data.getValue());

        } else if (data.getValue() instanceof StatusDataPublishedByEPOSController) {
            updateCanOpenHardwarePanel(data.getKey(), (StatusDataPublishedByHardware) data.getValue());
            updateController(getShortComponentName(data.getKey()), (StatusDataPublishedByEPOSController) data.getValue());

        } else if (data.getValue() instanceof StatusDataPublishedByPlutoGateway) {
            updateCanOpenHardwarePanel(data.getKey(), (StatusDataPublishedByHardware) data.getValue());
            this.plutoGatewayPanel.updatePlutoGateway(getShortComponentName(data.getKey()), (StatusDataPublishedByPlutoGateway) data.getValue());

        } else if (data.getValue() instanceof StatusDataPublishedByHardware) {
            updateCanOpenHardwarePanel(data.getKey(), (StatusDataPublishedByHardware) data.getValue());
            this.onlineStrainGaugePanel.updateCanOpenDevice(getShortComponentName(data.getKey()), (StatusDataPublishedByHardware) data.getValue());

        } else if (data.getValue() instanceof StatusDataPublishedByAutochangerTruck) {
            updateTruck(data.getKey(), (StatusDataPublishedByAutochangerTruck) data.getValue());

        } else if (data.getValue() instanceof StatusDataPublishedByAutoChangerTrucks) {
            updateTrucks((StatusDataPublishedByAutoChangerTrucks) data.getValue());

        } else if (data.getValue() instanceof StatusDataPublishedByAutochangerLatch) {
            updateLatch(data.getKey(), (StatusDataPublishedByAutochangerLatch) data.getValue());

        } else if (data.getValue() instanceof StatusDataPublishedByAutochangerTwoLatches) {
            updateLatches((StatusDataPublishedByAutochangerTwoLatches) data.getValue());

        } else if (data.getValue() instanceof StatusDataPublishedByAutochangerOnlineClamp) {
            updateClamp(data.getKey(), (StatusDataPublishedByAutochangerOnlineClamp) data.getValue());

        } else if (data.getValue() instanceof StatusDataPublishedByAutochangerThreeClamps) {
            updateClamps((StatusDataPublishedByAutochangerThreeClamps) data.getValue());

        } else if (data.getValue() instanceof KeyValueDataList
                && AC_PLUTOGATEWAY_NAME.equals(getShortComponentName(data.getKey()))) {
            plutoGatewayPanel.updateSensor((KeyValueDataList) data.getValue());
            Map<String, Serializable> sensorMap
                    = ((KeyValueDataList) data.getValue()).getListOfKeyValueData().stream().collect(Collectors.toMap(KeyValueData::getKey, KeyValueData::getValue));
            plcAutochangerPanel.updateFromSensorList(sensorMap);
            acGeneralViewPanel.updateFromSensorList(sensorMap);
        }
    }

    public void updateController(String name, StatusDataPublishedByEPOSController controllerStatus) {
        for (EPOSControllerPanel ctlPanel: controllerPanelsList) {
            ctlPanel.updateController(name, controllerStatus);
        }
        acTrucksPanel.updateController(name, controllerStatus);
        acClampsPanel.updateController(name, controllerStatus);
        acLatchesPanel.updateController(name, controllerStatus);
    }

    /**
     * Initialization of the GUI for data stored in configuration and
     * controllers names map *
     */
    private class GuiInitialization implements Runnable {

        private final ConfigurationInfo configInfo;

        public GuiInitialization(ConfigurationInfo configInfo) {
            this.configInfo = configInfo;
        }

        @Override
        public void run() {
            //** Autochanger General View Panel **/
            acGeneralViewPanel.initializeGui(configInfo);

            /**
             * Autochanger Clamps Panel *
             */
            acClampsPanel.initializeGui(configInfo);

            /**
             * Autochanger Latches Panel *
             */
            acLatchesPanel.initializeGui(configInfo);

            /**
             * Autochanger Trucks Panel *
             */
            acTrucksPanel.initializeGui(configInfo);

            /**
             * Linear Rail Master Controller Panel *
             */
            acTruckXminusControllerPanel.initializeGui(configInfo);

            /**
             * Linear Rail Follower Controller Panel*
             */
            acTruckXplusControllerPanel.initializeGui(configInfo);

            /**
             * Latch Xminus Controller Panel*
             */
            latchXminusControllerPanel.initializeGui(configInfo);

            /**
             * Latch Xplus Controller Panel*
             */
            latchXplusControllerPanel.initializeGui(configInfo);

            /**
             * Online Clamp Xminus Controller Panel*
             */
            onlineClampXminusControllerPanel.initializeGui(configInfo);

            /**
             * Online Clamp Xplus Controller Panel*
             */
            onlineClampXplusControllerPanel.initializeGui(configInfo);

            /**
             * Online Clamp Z Controller Panel*
             */
            onlineClampYminusControllerPanel.initializeGui(configInfo);

            /**
             * PlutoGateway Panel*
             */
            plutoGatewayPanel.initializeGui(configInfo);

            /**
             * OnlineStrainGauge Panel*
             */
            onlineStrainGaugePanel.initializeGui(configInfo);

        }
    }

    protected void updateClamp(String clampLongName, StatusDataPublishedByAutochangerOnlineClamp s) {
        this.acClampsPanel.updateClamp(getShortComponentName(clampLongName), s);
    }

    protected void updateClamps(StatusDataPublishedByAutochangerThreeClamps s) {
        this.acClampsPanel.updateClamps(s);
        this.acGeneralViewPanel.updateClamps(s);
    }

    protected void updateLatch(String latchLongName, StatusDataPublishedByAutochangerLatch s) {
        this.acLatchesPanel.updateLatch(getShortComponentName(latchLongName), s);
    }

    protected void updateLatches(StatusDataPublishedByAutochangerTwoLatches s) {
        this.acLatchesPanel.updateLatches(s);
        this.acGeneralViewPanel.updateLatches(s);
    }

    protected void updateTruck(String truckLongName, StatusDataPublishedByAutochangerTruck s) {
        this.acTrucksPanel.updateTruck(getShortComponentName(truckLongName), s);
    }

    protected void updateTrucks(StatusDataPublishedByAutoChangerTrucks s) {
        this.acTrucksPanel.updateTrucks(s);
        this.acGeneralViewPanel.updateTrucks(s);
    }

    private void initPanels() {
        //controllers Panels
        String fullPathPrefix = getCanOpenHardwareListPanel().getTcpProxyName() + "/";
        acTruckXminusControllerPanel = new EPOSControllerPanel(fullPathPrefix + "acTruckXminusController");
        acTruckXplusControllerPanel = new EPOSControllerPanel(fullPathPrefix + "acTruckXplusController");
        latchXminusControllerPanel = new EPOSControllerPanel(fullPathPrefix + "latchXminusController");
        latchXplusControllerPanel = new EPOSControllerPanel(fullPathPrefix + "latchXplusController");
        onlineClampXminusControllerPanel = new EPOSControllerPanel(fullPathPrefix + "onlineClampXminusController");
        onlineClampXplusControllerPanel = new EPOSControllerPanel(fullPathPrefix + "onlineClampXplusController");
        onlineClampYminusControllerPanel = new EPOSControllerPanel(fullPathPrefix + "onlineClampYminusController");
        controllerPanelsList = new EPOSControllerPanel[] {acTruckXminusControllerPanel, acTruckXplusControllerPanel,
            latchXminusControllerPanel, latchXplusControllerPanel,
            onlineClampXminusControllerPanel, onlineClampXplusControllerPanel, onlineClampYminusControllerPanel};
        //plutoGatewayPanel
        plutoGatewayPanel = new PlutoGatewayPanel(fullPathPrefix + AC_PLUTOGATEWAY_NAME);
        onlineStrainGaugePanel = new CanOpenDevicePanel(fullPathPrefix + AC_STRAIN_GAUGE_NAME);

        acClampsPanel = new AutochangerThreeOnlineClampsPanel();
        acLatchesPanel = new AutochangerTwoLatchesPanel();
        acTrucksPanel = new AutochangerTwoTrucksPanel(AC_TRUCKS_NAME);
        //PLC autochanger panel
        plcAutochangerPanel = new PLCAutochangerPanel();
    }

    @Override
    public void resetPanel() {
        FCSLOG.info(getName() + " resetting panels");
        super.resetPanel();
        acTruckXminusControllerPanel.resetPanel();
        acTruckXplusControllerPanel.resetPanel();
        latchXminusControllerPanel.resetPanel();
        latchXplusControllerPanel.resetPanel();
        onlineClampXminusControllerPanel.resetPanel();
        onlineClampXplusControllerPanel.resetPanel();
        onlineClampYminusControllerPanel.resetPanel();
        plutoGatewayPanel.resetPanel();
        onlineStrainGaugePanel.resetPanel();
        acGeneralViewPanel.resetPanel();
        acLatchesPanel.resetPanel();
        acTrucksPanel.resetPanel();
        acClampsPanel.resetPanel();
    }

}
