
package org.lsst.ccs.subsystems.fcs.ui;

import static org.lsst.ccs.subsystems.fcs.ui.commons.Tools.insets_std;

import java.awt.Color;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.io.Serializable;
import java.util.Map;
import java.util.stream.Collectors;

import javax.swing.BorderFactory;
import javax.swing.JFrame;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.SwingUtilities;
import org.lsst.ccs.bus.data.KeyValueData;
import org.lsst.ccs.bus.data.KeyValueDataList;
import static org.lsst.ccs.subsystems.fcs.loader.ui.PLCLoaderPanel.DOCKED;
import static org.lsst.ccs.subsystems.fcs.loader.ui.PLCLoaderPanel.DOCKEDB;
import org.lsst.ccs.subsystems.fcs.ui.commons.DigitalSwitch;

/**
 * A panel to display loader status LPS and LRH, and on Camera sensor
 *
 * @author virieux
 */
public class LoaderStatusPanel extends JPanel {

    private static final long serialVersionUID = 1L;

    private static final String title = "loader status";
    public static final String LPS = "LPS_0";
    public static final String LPSB = "LPS_1";
    public static final String LRH = "LRH_0";
    public static final String LRHB = "LRH_1";    
    private final DigitalSwitch loaderPresenceSwitch = new DigitalSwitch(DOCKED, DOCKEDB);
    private final DigitalSwitch loader_LPS_Switch = new DigitalSwitch(LPS, LPSB);
    private final DigitalSwitch loader_LRH_Switch = new DigitalSwitch(LRH, LRHB);

    public LoaderStatusPanel() {
        initComponents();
    }

    
    /**
     * A Runnable to initialize this Panel from status data published on the
     * STATUS bus by the loader
     */
    private class UpdateFromSensorList implements Runnable {

        private final KeyValueDataList kvdl;

        public UpdateFromSensorList(KeyValueDataList kvdl) {
            this.kvdl = kvdl;
        }

        @Override
        public void run() {
            /* sMap is the map obtained from kvdl : key= (String) sensorName, value = (int) sensorValue */
            Map<String, Serializable> sMap = 
                    kvdl.getListOfKeyValueData().stream().collect(Collectors.toMap(KeyValueData::getKey, KeyValueData::getValue));

            /* Loader_LPS and Loader_LRH are complementary sensors*/
            /*Loader_LPS (=1 if loader is at Storage, loader connected) */
            loader_LPS_Switch.updateColorComplementary(sMap);

            /*Loader_LRH (=1 if loader is holding filter at HANDOFF) */
            loader_LRH_Switch.updateColorComplementary(sMap);

            /* loaderOnCameraSensors are redondant sensors */
            /* loader presence status */
            loaderPresenceSwitch.updateColorRedondantSensor(sMap);
        }
    }

    void updateFromSensorList(KeyValueDataList kvdl) {
        SwingUtilities.invokeLater(new UpdateFromSensorList(kvdl));
    }

    // <editor-fold defaultstate="collapsed" desc="Form Description">
    private void initComponents() {
        setBorder(BorderFactory.createTitledBorder(BorderFactory.createLineBorder(
                Color.MAGENTA, WIDTH + 5), "Read on loaderPlutoGateway"));
        setLayout(new GridBagLayout());
        GridBagConstraints gbc = new GridBagConstraints();
        gbc.gridx = 0;
        gbc.gridy = 0;
        gbc.anchor = GridBagConstraints.WEST;
        gbc.insets = insets_std;
        add(new JLabel("Loader docked"), gbc);
        gbc.gridy++;
        add(new JLabel("Loader_LPS (loader at Storage)"), gbc);
        gbc.gridy++;
        add(new JLabel("Loader_LRH (loader holds filter)"), gbc);
        gbc.gridy = 0;
        gbc.gridx = 1;
        gbc.anchor = GridBagConstraints.EAST;
        add(this.loaderPresenceSwitch, gbc);
        gbc.gridy++;
        add(loader_LPS_Switch, gbc);
        gbc.gridy++;
        add(loader_LRH_Switch, gbc);
        setDefaultValues();
    }

    /**
     * set default values for components color and labels.
     */
    public void setDefaultValues() {
        loader_LPS_Switch.setColor(Color.GRAY);
        loader_LRH_Switch.setColor(Color.GRAY);
        loaderPresenceSwitch.setColor(Color.GRAY);
    }

    /**
     * Reset Panels to default values.
     */
    public void resetPanel() {
        setDefaultValues();
    }

    @Override
    public String toString() {
        return title;
    }

    /**
     * This main launches an GUI for the whole subsystem "fcs". It's used only
     * for tests.
     *
     * @param args the command line arguments
     */
    public static void main(String[] args) {

        LoaderStatusPanel d = new LoaderStatusPanel();
        JFrame frame = new JFrame("LoaderStatus Panel");
        frame.setContentPane(d);
        frame.pack();
        frame.setVisible(true);
        frame.setDefaultCloseOperation(JFrame.EXIT_ON_CLOSE);
    }

}
