package org.lsst.ccs.subsystems.fcs.loader.ui;

import java.awt.Color;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.event.ActionEvent;
import java.util.Map;
import javax.swing.BorderFactory;
import javax.swing.JButton;
import javax.swing.JFrame;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JProgressBar;
import javax.swing.SwingUtilities;
import org.lsst.ccs.bus.data.ConfigurationInfo;
import static org.lsst.ccs.subsystems.fcs.FCSCst.UNKNOWN;
import org.lsst.ccs.subsystems.fcs.StatusDataPublishedByLoader;
import org.lsst.ccs.subsystems.fcs.StatusDataPublishedByLoaderCarrier;
import org.lsst.ccs.subsystems.fcs.StatusDataPublishedByLoaderClamp;
import org.lsst.ccs.subsystems.fcs.ui.commons.DigitalSwitch;
import org.lsst.ccs.subsystems.fcs.ui.commons.Tools;
import static org.lsst.ccs.subsystems.fcs.ui.commons.Tools.ZERO_VALUE;
import static org.lsst.ccs.subsystems.fcs.ui.commons.Tools.greenColor;
import static org.lsst.ccs.subsystems.fcs.ui.commons.Tools.insets_big;
import static org.lsst.ccs.subsystems.fcs.ui.commons.Tools.insets_std;

/**
 * A view to display a global view of the loader.
 *
 * @author virieux
 */
public class LoaderGeneralViewPanel extends JPanel {

    private static final long serialVersionUID = 8834267363875256916L;
    private InterfaceLoaderGUI subs;

    private final JLabel loaderDockedLabel = new JLabel("LOADER DOCKED");
    private final JPanel fakeSensorsPanel = new JPanel();
    private final JPanel controlPanel = new JPanel();
    private final String carrierName = "loader/carrier";

    /* camera protection panel */
    private final JPanel cpsPanel = new JPanel();
    private final JPanel keysPanel = new JPanel();
    private final DigitalSwitch onCameraDigitalSwitch = new DigitalSwitch();
    private final DigitalSwitch ap2Switch = new DigitalSwitch();
    private final DigitalSwitch af0Switch = new DigitalSwitch();
    private final DigitalSwitch af1Switch = new DigitalSwitch();
    private final DigitalSwitch af3Switch = new DigitalSwitch();
    private final DigitalSwitch lockoutSwitch = new DigitalSwitch();
    private final DigitalSwitch engineeringSwitch = new DigitalSwitch();
    private final DigitalSwitch carrierRelaySwitch = new DigitalSwitch();
    private final DigitalSwitch hooksRelaySwitch = new DigitalSwitch();
    private final DigitalSwitch chainPresenceSwitch = new DigitalSwitch();
    private final DigitalSwitch defaultStatusSwitch = new DigitalSwitch();


    /* Clamp Panel*/
    private final JPanel clampPanel = new JPanel();
    private final DigitalSwitch filterInLoaderSwitch = new DigitalSwitch();
    private final JLabel clampHomingLabel = new JLabel();
    private final JLabel clampStateLabel = new JLabel();
    private final JLabel clampPositionLabel = new JLabel();
    private final JLabel forceStatusLabel = new JLabel();
    private final DigitalSwitch clampControllerSwitch = new DigitalSwitch();

    /* Carrier*/
    private final JPanel carrierPanel = new JPanel();
    private final JLabel carrierPositionLabel = new JLabel();
    private final JLabel carrierStatusLabel = new JLabel();
    private final JLabel carrierSpeedLabel = new JLabel();
    private final JProgressBar carrierPositionProgressBar = new JProgressBar(1, 0, 100);
    private final DigitalSwitch carrierControllerSwitch = new DigitalSwitch();

    //control panel
    private final JButton loadFilterButton = new JButton("loadFilterIntoStorageBox");
    private final JButton unloadFilterButton = new JButton("unloadFilterFromStorageBox");
    private final JButton moveFilterStoHButton = new JButton("moveFilterToHandoff");
    private final JButton moveFilterHtoSButton = new JButton("moveFilterToStorage");
    private final JButton openClampAndMoveEmptyToSButton = new JButton("openClampAndMoveEmptyToStorage");
    private final JButton moveEmptyToHandCloseButton = new JButton("moveEmptyToHandoffAndClose");
    // End of variables declaration

    /**
     * A Runnable To updateState the general items for loader.
     */
    class UpdateLoaderGeneral implements Runnable {

        private final StatusDataPublishedByLoader s;

        UpdateLoaderGeneral(StatusDataPublishedByLoader s) {
            this.s = s;
        }

        @Override
        public void run() {

            onCameraDigitalSwitch.updateColor(s.isLoaderOnCameraSensorsInError(), s.isLoaderOnCameraSensorValue());
            af0Switch.updateColor(s.isAf0InError(), s.isAf0());
            af1Switch.updateColor(s.isAf1InError(), s.isAf1());
            af3Switch.updateColor(s.isAf3InError(), s.isAf3());
            ap2Switch.updateColor(s.isAp2InError(), s.isAp2());

            engineeringSwitch.updateColor(s.isKeyEngInError(), s.isKeyEng());
            lockoutSwitch.updateColor(s.isKeyLockInError(), s.isKeyLock());

            /* loader status */
            carrierRelaySwitch.setColor(s.isCarrierRelayStatus() ? greenColor : Color.gray);
            hooksRelaySwitch.setColor(s.isHooksRelayStatus() ? greenColor : Color.gray);
            chainPresenceSwitch.setColor(s.isSafetyBeltPresence() ? greenColor : Color.gray);
            defaultStatusSwitch.setColor(s.isLoaderDefaultStatus() ? greenColor : Color.gray);
            filterInLoaderSwitch.setColor(s.getFilterPresenceSensorValue() ? greenColor : Color.GRAY);
        }
    }

    /**
     * A Runnable to update this panel from data published on the status bus
     * from the Loader Clamp.
     */
    private class UpdateClamp implements Runnable {

        private final StatusDataPublishedByLoaderClamp s;

        public UpdateClamp(StatusDataPublishedByLoaderClamp status) {
            this.s = status;
        }

        @Override
        public void run() {

            if (s.isHomingDone()) {
                clampHomingLabel.setText("DONE");
                clampHomingLabel.setForeground(greenColor);
            } else {
                clampHomingLabel.setText("NOT DONE");
                clampHomingLabel.setForeground(Color.red);
            }
            Color clampStateColor = s.getClampState().getColor();
            clampStateLabel.setText(s.getClampState().getText());
            clampStateLabel.setForeground(clampStateColor);
            forceStatusLabel.setText(s.getForceStatus().toString());
            clampPositionLabel.setText(Integer.toString(s.getPosition()));
            //controller status
            clampControllerSwitch.setColor(s.isControllerInError() ? Color.RED : greenColor);

        }
    }

    /**
     * A Runnable to update this panel from carrier data
     */
    private class UpdateCarrier implements Runnable {

        private final StatusDataPublishedByLoaderCarrier s;

        public UpdateCarrier(StatusDataPublishedByLoaderCarrier s) {
            this.s = s;
        }

        @Override
        public void run() {
            carrierPositionLabel.setText(Integer.toString(s.getPosition()));
            carrierSpeedLabel.setText(Integer.toString(s.getSpeed()));
            /* - : to give the illusion of the real motion of carrier*/
            carrierPositionProgressBar.setValue(-s.getPosition());
            carrierPositionProgressBar.setForeground(Color.PINK);

            if (s.isAtStorage()) {
                carrierStatusLabel.setText("STORAGE");

            } else if (s.isAtEngaged()) {
                carrierStatusLabel.setText("ENGAGED");

            } else if (s.isAtHandoff()) {
                carrierStatusLabel.setText("HANDOFF");

            } else {
                carrierStatusLabel.setText("IN_TRAVEL");
            }
            //controller status
            carrierControllerSwitch.setColor(s.isControllerInError() ? Color.RED : greenColor);

        }

    }

    /**
     * A constructor without any argument for tests.
     */
    public LoaderGeneralViewPanel() {
        initComponents();
    }

    /**
     * A constructor with a LoaderGUI.
     *
     * @param subs
     */
    public LoaderGeneralViewPanel(InterfaceLoaderGUI subs) {
        this.subs = subs;
        initComponents();
    }

    public void setSubsystem(InterfaceLoaderGUI subs) {
        this.subs = subs;
    }

    /**
     * Initialize this Panel from Configuration Info data. TODO : initialize
     * timeouts for commands loUnloadFilter and loLoadFilter ?
     *
     * @param configInfo
     */
    public void initializeGui(ConfigurationInfo configInfo) {
        SwingUtilities.invokeLater(new GuiInitialization(configInfo));
    }

    /**
     * To update the Panel when the loader publishes on the bus its global
     * status.
     *
     * @param status
     */
    public void updateLoaderGeneral(StatusDataPublishedByLoader status) {
        SwingUtilities.invokeLater(new UpdateLoaderGeneral(status));
    }

    /**
     * To update this Panel when the loader clamp publishes on the bus.
     *
     * @param status
     */
    public void updateClamp(StatusDataPublishedByLoaderClamp status) {
        SwingUtilities.invokeLater(new UpdateClamp(status));
    }

    /**
     * to update this panel from carrier data
     *
     * @param status
     */
    public void updateCarrier(StatusDataPublishedByLoaderCarrier status) {
        SwingUtilities.invokeLater(new UpdateCarrier(status));
    }

    /**
     * A Runnable to initialize this Panel from Configuration data.
     *
     */
    private class GuiInitialization implements Runnable {

        private final ConfigurationInfo configInfo;

        public GuiInitialization(ConfigurationInfo configInfo) {
            this.configInfo = configInfo;
        }

        @Override
        public void run() {
            //Retrieve the configuration data for my name.
            Map<String, String> configForCarrier = configInfo.getCurrentValuesFor(carrierName);
            int handoffPosition = Tools.toInt(configForCarrier, "handoffPosition");
            int storagePosition = Tools.toInt(configForCarrier, "storagePosition");
            carrierPositionProgressBar.setMinimum(-handoffPosition);
            carrierPositionProgressBar.setMaximum(-storagePosition);
            carrierPositionProgressBar.setValue(0);
        }
    }


    /**
     * To initialize all the components of this Panel.
     */
    // <editor-fold defaultstate="collapsed" desc="Form Description">
    private void initComponents() {
        setBorder(BorderFactory.createTitledBorder(BorderFactory.createLineBorder(new Color(255, 255, 0)), "LOADER"));
        setForeground(new Color(204, 204, 255));
        setLayout(new GridBagLayout());

        /**
         * ** loader clamp Panel **
         */
        JPanel clampControllerPanel = new JPanel();
        clampControllerPanel.setBorder(BorderFactory.createEtchedBorder());
        clampControllerPanel.setLayout(new GridBagLayout());
        GridBagConstraints gbc = new GridBagConstraints();
        gbc.insets = insets_std;
        gbc.gridx = 0;
        gbc.gridy = 0;
        gbc.anchor = GridBagConstraints.CENTER;
        clampControllerPanel.add(new JLabel("Controller Status"), gbc);
        gbc.gridy++;
        clampControllerPanel.add(clampControllerSwitch, gbc);
        clampControllerSwitch.setSize(40);

        JPanel filterPresencePanel = new JPanel();
        filterPresencePanel.setBorder(BorderFactory.createEtchedBorder());
        filterPresencePanel.setLayout(new GridBagLayout());
        gbc = new GridBagConstraints();
        gbc.insets = insets_std;
        gbc.gridx = 0;
        gbc.gridy = 0;
        gbc.anchor = GridBagConstraints.CENTER;
        filterPresencePanel.add(new JLabel("Filter Presence"), gbc);
        gbc.gridy++;
        filterPresencePanel.add(filterInLoaderSwitch, gbc);
        filterInLoaderSwitch.setSize(40);

        clampPanel.setBorder(BorderFactory.createTitledBorder("Loader Clamp"));
        clampPanel.setLayout(new GridBagLayout());
        gbc = new GridBagConstraints();
        gbc.insets = insets_std;
        gbc.gridx = 0;
        gbc.gridy = 0;
        gbc.anchor = GridBagConstraints.LINE_START;
        clampPanel.add(new JLabel("Homing:"), gbc);
        gbc.gridx++;
        gbc.anchor = GridBagConstraints.LINE_END;
        clampPanel.add(clampHomingLabel, gbc);
        gbc.gridx--;

        gbc.gridy++;
        gbc.anchor = GridBagConstraints.LINE_START;
        clampPanel.add(clampControllerPanel, gbc);
        gbc.gridx++;
        clampPanel.add(filterPresencePanel, gbc);
        gbc.gridx--;
        gbc.gridy++;
        gbc.anchor = GridBagConstraints.LINE_START;
        int y = gbc.gridy;
        clampPanel.add(new JLabel("Hooks Status"), gbc);
        gbc.gridy++;
        clampPanel.add(new JLabel("Hooks Position"), gbc);
        gbc.gridy++;
        clampPanel.add(new JLabel("Clamped Force Status"), gbc);
        gbc.gridy = y;
        gbc.gridx++;
        gbc.anchor = GridBagConstraints.CENTER;
        clampPanel.add(clampStateLabel, gbc);
        gbc.gridy++;
        gbc.anchor = GridBagConstraints.LINE_END;
        clampPanel.add(clampPositionLabel, gbc);
        gbc.gridy++;
        gbc.anchor = GridBagConstraints.CENTER;
        clampPanel.add(forceStatusLabel, gbc);


        /**
         * ** end of loader clamp Panel **
         */

        /**
         * ** loader carrier Panel **
         */
        JPanel controllerPanel = new JPanel();
        controllerPanel.setBorder(BorderFactory.createEtchedBorder());
        controllerPanel.setLayout(new GridBagLayout());

        gbc = new GridBagConstraints();
        gbc.insets = insets_std;
        gbc.gridx = 0;
        gbc.gridy = 0;
        gbc.anchor = GridBagConstraints.CENTER;
        controllerPanel.add(new JLabel("Controller Status"), gbc);
        gbc.gridy++;
        controllerPanel.add(carrierControllerSwitch, gbc);
        carrierControllerSwitch.setSize(50);

        carrierPanel.setBorder(BorderFactory.createTitledBorder("Loader Carrier"));
        carrierPanel.setLayout(new GridBagLayout());
        gbc = new GridBagConstraints();
        gbc.insets = insets_std;
        gbc.gridx = 0;
        gbc.gridy = 0;
        gbc.anchor = GridBagConstraints.LINE_START;
        carrierPanel.add(new JLabel("Carrier Position"), gbc);
        gbc.gridy++;
        carrierPanel.add(new JLabel("Carrier Status"), gbc);
        gbc.gridy++;
        carrierPanel.add(new JLabel("Carrier Speed"), gbc);
        gbc.gridy++;
        int y2 = gbc.gridy;
        carrierPanel.add(controllerPanel, gbc);

        gbc.gridy = 0;
        gbc.gridx++;
        gbc.anchor = GridBagConstraints.LINE_END;
        carrierPanel.add(carrierPositionLabel, gbc);
        gbc.gridy++;
        carrierPanel.add(carrierStatusLabel, gbc);
        gbc.gridy++;
        carrierPanel.add(carrierSpeedLabel, gbc);

        gbc.gridy = y2;
        gbc.gridx = 1;
        gbc.gridheight = 2;
        gbc.anchor = GridBagConstraints.CENTER;
        carrierPositionProgressBar.setBorderPainted(true);
        carrierPositionProgressBar.setBackground(greenColor);
        carrierPositionProgressBar.setStringPainted(true);
        carrierPanel.add(carrierPositionProgressBar, gbc);

        /**
         * ** end of loader carrier Panel **
         */

        /**
         ******************************************************************************
         ** Control Panel
         * *****************************************************************************
         */
        loadFilterButton.addActionListener((ActionEvent evt) -> {
            loadFilterActionPerformed(evt);
        });

        unloadFilterButton.addActionListener((ActionEvent evt) -> {
            unloadFilterActionPerformed(evt);
        });

        moveFilterStoHButton.addActionListener((ActionEvent evt) -> {
            moveFilterStoHActionPerformed(evt);
        });

        moveFilterHtoSButton.addActionListener((ActionEvent evt) -> {
            moveFilterHtoSActionPerformed(evt);
        });

        openClampAndMoveEmptyToSButton.addActionListener((ActionEvent evt) -> {
            openClampAndMoveEmptyToStoSActionPerformed(evt);
        });

        moveEmptyToHandCloseButton.addActionListener((ActionEvent evt) -> {
            moveEmptyToHandCloseActionPerformed(evt);
        });

        controlPanel.setLayout(new GridBagLayout());
        GridBagConstraints gbc0 = new GridBagConstraints();
        gbc0.insets = insets_std;
        gbc0.gridx = 0;
        gbc0.gridy = 0;
        gbc0.fill = GridBagConstraints.HORIZONTAL;
        //commands loadFilterIntoStorageBox and unloadFilterFromStorageBox are not yet implemented see Loader.java
//        controlPanel.add(loadFilterButton, gbc0);
//        gbc0.gridy++;
//        controlPanel.add(unloadFilterButton, gbc0);
//        gbc0.gridy++;
        controlPanel.add(moveFilterStoHButton, gbc0);
        gbc0.gridy++;
        controlPanel.add(moveFilterHtoSButton, gbc0);
        gbc0.gridy++;
        controlPanel.add(openClampAndMoveEmptyToSButton, gbc0);
        gbc0.gridy++;
        controlPanel.add(moveEmptyToHandCloseButton, gbc0);
        //end of Control Panel

        //Fake Sensors Panel
        fakeSensorsPanel.setBorder(BorderFactory.createLineBorder(Color.BLUE, WIDTH + 5));
        fakeSensorsPanel.setLayout(new GridBagLayout());
        GridBagConstraints gbc1 = new GridBagConstraints();
        gbc1.insets = insets_std;
        gbc1.gridx = 0;
        gbc1.gridy = 0;
        gbc1.fill = GridBagConstraints.HORIZONTAL;
        fakeSensorsPanel.add(loaderDockedLabel, gbc1);

        gbc1.gridx++;
        fakeSensorsPanel.add(onCameraDigitalSwitch, gbc1);
        //end Fake Sensors Panel

        //CameraProtectionPanel with status coming from autochanger
        cpsPanel.setBorder(BorderFactory.createLineBorder(Color.MAGENTA, WIDTH + 5));
        cpsPanel.setLayout(new GridBagLayout());
        gbc0 = new GridBagConstraints();
        gbc0.gridx = 0;
        gbc0.gridy = 0;
        gbc0.anchor = GridBagConstraints.WEST;
        gbc0.insets = insets_std;
        cpsPanel.add(new JLabel("AP2 (AC at handoff)"), gbc0);
        gbc0.gridy++;
        cpsPanel.add(new JLabel("AF0 (AC latches opened)"), gbc0);
        gbc0.gridy++;
        cpsPanel.add(new JLabel("AF1 (filter detected in AC)"), gbc0);
        gbc0.gridy++;
        cpsPanel.add(new JLabel("AF3 (filter locked in AC)"), gbc0);

        gbc0.gridx = 1;
        gbc0.gridy = 0;
        gbc0.anchor = GridBagConstraints.EAST;
        cpsPanel.add(ap2Switch, gbc0);
        gbc0.gridy++;
        cpsPanel.add(af0Switch, gbc0);
        gbc0.gridy++;
        cpsPanel.add(af1Switch, gbc0);
        gbc0.gridy++;
        cpsPanel.add(af3Switch, gbc0);

        //keys Panel : key lock/permit, engineering key ...
        keysPanel.setBorder(BorderFactory.createLineBorder(Color.MAGENTA, WIDTH + 5));
        keysPanel.setLayout(new GridBagLayout());
        gbc0 = new GridBagConstraints();
        gbc0.gridx = 0;
        gbc0.gridy = 0;
        gbc0.anchor = GridBagConstraints.WEST;
        gbc0.insets = insets_std;
        keysPanel.add(new JLabel("Lock Out Bay M"), gbc0);
        gbc0.gridy++;
        keysPanel.add(new JLabel("Engineering key"), gbc0);
        gbc0.gridy++;
        keysPanel.add(new JLabel("Carrier relay"), gbc0);
        gbc0.gridy++;
        keysPanel.add(new JLabel("Hooks relay"), gbc0);
        gbc0.gridy++;
        keysPanel.add(new JLabel("Safety belt presence"), gbc0);
        gbc0.gridy++;
        keysPanel.add(new JLabel("Default Status"), gbc0);

        gbc0.gridx = 1;
        gbc0.gridy++;
        gbc0.gridy = 0;
        gbc0.anchor = GridBagConstraints.EAST;
        keysPanel.add(lockoutSwitch, gbc0);
        gbc0.gridy++;
        keysPanel.add(engineeringSwitch, gbc0);
        gbc0.gridy++;
        keysPanel.add(carrierRelaySwitch, gbc0);
        gbc0.gridy++;
        keysPanel.add(hooksRelaySwitch, gbc0);
        gbc0.gridy++;
        keysPanel.add(chainPresenceSwitch, gbc0);
        gbc0.gridy++;
        keysPanel.add(defaultStatusSwitch, gbc0);

        //Whole Panel
        gbc = new GridBagConstraints();
        setBorder(BorderFactory.createTitledBorder(BorderFactory.createLineBorder(new Color(255, 255, 0)),
                "LOADER General View"));
        setForeground(new Color(204, 204, 255));
        setLayout(new GridBagLayout());
        gbc.anchor = GridBagConstraints.BASELINE_LEADING;
        gbc.insets = insets_big;
        gbc.gridx = 0;
        gbc.gridy = 0;
        add(fakeSensorsPanel, gbc);
        gbc.gridy++;
        add(cpsPanel, gbc);
        gbc.gridx++;
        add(keysPanel, gbc);

        gbc.gridx--;
        gbc.gridy++;
        add(clampPanel, gbc);
        gbc.gridx++;
        add(carrierPanel, gbc);
        gbc.gridx--;
        gbc.gridy++;
        add(controlPanel, gbc);


        setDefaultValues();
    }

    private void setDefaultValues() {
        onCameraDigitalSwitch.setColor(Color.gray);
        ap2Switch.setColor(Color.gray);
        af0Switch.setColor(Color.gray);
        af1Switch.setColor(Color.gray);
        af3Switch.setColor(Color.gray);
        lockoutSwitch.setColor(Color.gray);
        engineeringSwitch.setColor(Color.gray);
        carrierRelaySwitch.setColor(Color.gray);
        hooksRelaySwitch.setColor(Color.gray);
        chainPresenceSwitch.setColor(Color.gray);
        defaultStatusSwitch.setColor(Color.gray);
        clampHomingLabel.setText("TO BE DONE");
        clampHomingLabel.setForeground(Color.red);
        filterInLoaderSwitch.setColor(Color.gray);
        clampControllerSwitch.setColor(Color.gray);
        clampStateLabel.setText(UNKNOWN);
        clampPositionLabel.setText(ZERO_VALUE);
        forceStatusLabel.setText(UNKNOWN);
        carrierPositionLabel.setText(ZERO_VALUE);
        carrierStatusLabel.setText(UNKNOWN);
        carrierControllerSwitch.setColor(Color.gray);
    }

    /**
     * Reset components to default values.
     */
    public void resetPanel() {
        setDefaultValues();
    }

    private void loadFilterActionPerformed(ActionEvent evt) {
        subs.loLoadFilterIntoStorageBox();
    }

    private void unloadFilterActionPerformed(ActionEvent evt) {
        subs.loUnloadFilterFromStorageBox();
    }

    private void moveFilterStoHActionPerformed(ActionEvent evt) {
        subs.loMoveFilterStoH();
    }

    private void moveFilterHtoSActionPerformed(ActionEvent evt) {
        subs.loMoveFilterHtoS();
    }

    private void moveEmptyToHandCloseActionPerformed(ActionEvent evt) {
        subs.loMoveEmptyToHandClose();
    }

    private void openClampAndMoveEmptyToStoSActionPerformed(ActionEvent evt) {
        subs.loOpenClampAndMoveEmptyToS();
    }


    /**
     * To be displayed on the GUI.
     *
     * @return
     */
    @Override
    public String toString() {
        return "Loader General View Panel";
    }

    /**
     * Just for tests.
     *
     * @param args
     */
    public static void main(String[] args) {
        LoaderGeneralViewPanel d = new LoaderGeneralViewPanel();
        JFrame frame = new JFrame("LoaderGeneralViewPanel");
        frame.setContentPane(d);
        frame.pack();
        frame.setVisible(true);
        frame.setDefaultCloseOperation(JFrame.EXIT_ON_CLOSE);
    }
}
