package org.lsst.ccs.subsystems.fcs.ui.commons;

import java.awt.HeadlessException;
import java.io.Serializable;
import java.lang.reflect.InvocationTargetException;
import java.time.Duration;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ExecutionException;
import java.util.function.Predicate;
import javax.swing.JComponent;
import javax.swing.JOptionPane;
import javax.swing.JScrollPane;
import javax.swing.SwingWorker;
import org.freehep.graphicsbase.swing.ErrorDialog;
import org.lsst.ccs.bus.data.AgentInfo;
import org.lsst.ccs.bus.data.ConfigurationInfo;
import org.lsst.ccs.bus.messages.BusMessage;
import org.lsst.ccs.bus.messages.CommandAck;
import org.lsst.ccs.bus.messages.CommandNack;

import org.lsst.ccs.bus.messages.CommandRequest;
import org.lsst.ccs.bus.messages.CommandResult;
import org.lsst.ccs.bus.messages.StatusConfigurationInfo;

import org.lsst.ccs.bus.messages.StatusMessage;
import org.lsst.ccs.bus.messages.StatusStateChangeNotification;
import org.lsst.ccs.bus.messages.StatusSubsystemData;

import org.lsst.ccs.messaging.AgentMessagingLayer;
import org.lsst.ccs.messaging.AgentPresenceListener;
import org.lsst.ccs.messaging.BusMessageFilterFactory;
import org.lsst.ccs.messaging.CommandOriginator;
import org.lsst.ccs.messaging.ConcurrentMessagingUtils;
import org.lsst.ccs.messaging.StatusMessageListener;
import static org.lsst.ccs.subsystems.fcs.FCSCst.FCSLOG;

/**
 * This class gathers the common stuff for all FCS GUIs.
 *
 * @author virieux
 */
public abstract class GeneralGUI implements
        StatusMessageListener,
        AgentPresenceListener,
        Serializable {

    private static final long serialVersionUID = -3641559220265844557L;

    protected String destination;
    protected final String role;
    protected GeneralPanel generalPanel;
    protected boolean panelStateInitialized = false;
    public JScrollPane pane;
    public long abortTimeout;
    public long updateStateWithSensorsTimeout;

    private String name = "GeneralGUI";
    protected transient AgentMessagingLayer aml;
    protected final transient ConcurrentMessagingUtils cmu;
    private transient Predicate<BusMessage<? extends Serializable, ?>> originFilter;
    private final transient Predicate<BusMessage<? extends Serializable, ?>> dataStatusFilter;
    private final transient Predicate<BusMessage<? extends Serializable, ?>> configInfoStatusFilter;

    public long initializationTimeout;

    /**
     * To send command which takes time from a JButton.
     */
    public class CommandSwingWorker extends SwingWorker<Object, Object> {

        private String cmdDestination;
        private String cmdName;
        private long cmdTimeout;

        public CommandSwingWorker(String cmdN, long timeout) {
            super();
            if (cmdN == null) {
                throw new IllegalArgumentException("Command should not be null in CommandSwingWorker");
            }
            if (timeout == 0) {
                throw new IllegalArgumentException("timeout should not be equal to 0 in CommandSwingWorker");
            }
            this.cmdDestination = destination;
            this.cmdName = cmdN;
            this.cmdTimeout = timeout;
        }

        public CommandSwingWorker(String cmdN, long timeout, String moduleName) {
            super();
            if (cmdN == null) {
                throw new IllegalArgumentException("Command should not be null in CommandSwingWorker");
            }
            if (timeout == 0) {
                throw new IllegalArgumentException("timeout should not be equal to 0 in CommandSwingWorker");
            }
            if (moduleName == null) {
                throw new IllegalArgumentException("moduleName should not be null in this constructor of CommandSwingWorker");
            }
            this.cmdDestination = destination + "/" + moduleName;
            this.cmdName = cmdN;
            this.cmdTimeout = timeout;
        }

        @Override
        public Object doInBackground() {
            FCSLOG.info("Executing " + cmdName + " on " + cmdDestination);
            FCSLOG.info("/timeout=" + cmdTimeout);
            return sendCommand(new CommandRequest(cmdDestination, cmdName), cmdTimeout);
        }

        @Override
        protected void done() {
            try {
                Object response = get();
                String strRes;
                if (response instanceof Throwable) {
                    FCSLOG.error(String.valueOf(response));
                    FCSLOG.error(" command returned error", (Throwable) response);
                    ErrorDialog.showErrorDialog(null, response.toString(), (Throwable) response);

                } else if (response == null) {
                    strRes = "ok : DONE";
                    JOptionPane.showMessageDialog(null, strRes, cmdName, JOptionPane.INFORMATION_MESSAGE);

                } else {
                    strRes = String.valueOf(response);
                    JOptionPane.showMessageDialog(null, strRes, cmdName, JOptionPane.INFORMATION_MESSAGE);
                }
            } catch (InterruptedException | ExecutionException | HeadlessException ex) {
                FCSLOG.error(ex.toString());
                ErrorDialog.showErrorDialog(null, ex.toString(), (Throwable) ex);
            }
        }


        /**
         * Send a command to my subsystem and return the response of the
         * command. The response can be null.
         *
         * @param cmd
         * @param timeout
         * @return
         */
        public Object sendCommand(CommandRequest cmd, long timeout) {
            try {
                Object response = cmu.sendSynchronousCommand(cmd, Duration.ofMillis(timeout));
                String strRes = String.valueOf(response);
                FCSLOG.debug("strRes=" + strRes);
                if (response == null) {
                    return "ok : DONE";
                } else {
                    return strRes;
                }
            } catch (Exception ex) {
                return ex;
            }
        }
    }

    /**
     * Contructor of a GeneralGUI with a name, a subsystem name (destination,
     * and an AgentMessagingLayer.
     *
     * @param name
     * @param role
     * @param aml
     */
    public GeneralGUI(String name,
            String role,
            AgentMessagingLayer aml) {
        this.aml = aml;
        this.cmu = new ConcurrentMessagingUtils(aml);
        this.name = name;
        this.role = role;
        this.abortTimeout = 1000;
        this.updateStateWithSensorsTimeout = 2000;
        this.initializationTimeout = 2000;

        // ---- Filters
        /* originFilter is set when destination is set, and destination is set in connecting method */
        // filters messages that are subsystem data status
        dataStatusFilter = BusMessageFilterFactory.messageClass(StatusSubsystemData.class);
        /* filters messages that are StatusConfigurationInfo (when configuration data changes)*/
        configInfoStatusFilter = BusMessageFilterFactory.messageClass(StatusConfigurationInfo.class);
        // -----
    }

    /**
     * send a command to my subsystem with a timeout
     *
     * @param commandName
     * @param timeout
     */
    public void sendCommand(String commandName, long timeout) {
        new CommandSwingWorker(commandName, timeout).execute();
    }

    /**
     * send a command to my subsystem with a timeout and the name of a component
     * of my subsystem
     *
     * @param commandName
     * @param timeout
     * @param componentName
     */
    public void sendCommand(String commandName, long timeout, String componentName) {
        new CommandSwingWorker(commandName, timeout, componentName).execute();
    }

    public String getName() {
        return name;
    }

    /**
     * Initialize GUI
     */
    public void initGui() {
        FCSLOG.info(name + ":initGUI");
        // This GUI listens to agent connection/disconnection events
        aml.getAgentPresenceManager().addAgentPresenceListener(this);
        /* done when connecting : aml.addStatusMessageListener(this, originFilter) */
    }

    public JComponent getGuiLayout() {
        return generalPanel;
    }

    public void shutdownMyDestinationSubsystem() {
        System.out.println("Shutting down " + destination);
        new CommandSwingWorker("shutdown", abortTimeout).execute();
    }

    public void abortAction() {
        System.out.println("Aborting " + destination);
        new CommandSwingWorker("abort", abortTimeout).execute();
    }

    public void stopMyDestinationSubsystem() {
        System.out.println("Stopping " + destination);
        cmu.sendAsynchronousCommand(new CommandRequest(destination, "stop", 1000L));
    }

    public void updateStateWithSensors() {
        FCSLOG.info("Reading sensors");
        FCSLOG.info("/timeout=" + updateStateWithSensorsTimeout);
        new CommandSwingWorker("updateStateWithSensors", updateStateWithSensorsTimeout).execute();
    }

    public void sendCommandSwingWorker(String commandName, long timeout, String moduleName) {
        FCSLOG.info(getName() + ": sending command= " + commandName + " to Module=" + moduleName);
        new CommandSwingWorker(commandName, timeout, moduleName).execute();
    }

    public void sendCommandSwingWorker(String commandName, long timeout) {
        FCSLOG.info(getName() + ": sending command= " + commandName);
        new CommandSwingWorker(commandName, timeout).execute();
    }

    /**
     * To update the Subsystem state panel only when the subsytem state has
     * changed.
     *
     * @param s
     */
    @Override
    public void onStatusMessage(StatusMessage s) {
        if (!panelStateInitialized) {
            this.generalPanel.getSubsystemPanel().updateState(s.getState());
            panelStateInitialized = true;

        } else {
            if (!(s instanceof StatusStateChangeNotification)) {
                return;
            }
            this.generalPanel.getSubsystemPanel().updateState(s.getState());
        }
    }

    @Override
    public void connected(AgentInfo... agents) {
        for (AgentInfo agent: agents) {
            /* a susystem is connecting on the bus*/
            if (!agent.hasAgentProperty(role)) {
                continue;
            }
            //my subsystem is connecting on the bus
            destination = agent.getName();
            // filters only the messages coming from my subsystem.
            originFilter = BusMessageFilterFactory.messageOrigin(destination);
            aml.addStatusMessageListener(this, originFilter);

            FCSLOG.warning(destination + " is connecting NOW :");

            String runMode = agent.getAgentProperty("runMode");
            generalPanel.setRunMode(runMode);

            FCSLOG.debug("runMode" + runMode);

            List<String> hardwareList = getHardwareNamesList();
            FCSLOG.debug("retrieved the list of connected hardwares" + hardwareList);

            List<String> loaderHardwareList = getLoaderHardwareNamesList();
            FCSLOG.debug("retrieved the list of LOADER connected hardwares" + loaderHardwareList);

            List<String> filterNamesList = getFilterNames();
            FCSLOG.debug("retrieved the list of filter names:" + filterNamesList);

            Map<String, String> controllersMap = getMobilNameControllerNameMap();

            /* Send asynchronous command to populate panels with configuration data.*/
            CommandRequest cmd = new CommandRequest(destination, "getConfigurationInfo");
            aml.sendCommandRequest(cmd, new CommandOriginator() {

                @Override
                public void processAck(CommandAck ack) {
                    FCSLOG.debug(destination + " has received a getConfigurationInfo command:"
                            + ack.toString());
                }

                @Override
                public void processResult(CommandResult result) {
                    try {
                        FCSLOG.debug("--- Got a reply to getConfigurationInfo from subs " + destination);
                        ConfigurationInfo configInfo = (ConfigurationInfo) result.getReply();
                        // Gui Initialization with configuration information
                        generalPanel.initializeGUIWithConfigInfo(configInfo, hardwareList);

                        if (!loaderHardwareList.isEmpty()) {
                            generalPanel.initializeGUIWithLoaderConfigInfo(configInfo, loaderHardwareList);
                        }

                        FCSLOG.debug("--- Initializing FilterListPanel from filter list " + filterNamesList);
                        generalPanel.initializeGUIWithFiltersInfo(configInfo, filterNamesList);

                        FCSLOG.debug("--- Initializing GUI from controllersMap " + controllersMap.toString());
                        generalPanel.initializeGUIWithControllersMap(controllersMap);

                        // Filters only specific data status from the worker subsystem
                        aml.addStatusMessageListener(generalPanel, originFilter.and(dataStatusFilter.or(configInfoStatusFilter)));

                        /* Send command to populate panels with dynamic data.*/
                        //TODO find another solution in the case of hardware cf tests in January 2016
                        //the code below doesn't work because the subsystem can receive the command "updateStateWithSensors"
                        //when some hardware is not already booted so an Alert is raised too early.
    //                    FCSLOG.debug("sending updateStateWithSensors");
    //                    cmu.sendAsynchronousCommand(new CommandRequest(destination, "updateStateWithSensors"));
                        cmu.sendAsynchronousCommand(new CommandRequest(destination, "publishData"));
                    } catch (InterruptedException | InvocationTargetException ex) {
                        FCSLOG.error(name + ":Error occured when initializing GUI with config info: " + ex);
                    }
                }

                @Override
                public void processNack(CommandNack nack) {
                }
            });
        }
    }

    /**
     * Sends a command to retrieve the list of hardware connected to the given
     * bridge. It sends a command to the worker subsystem
     *
     * @param bridge the bridge to list the hardware connected to.
     * @param tcpProxy the tcpProxy associated to the bridge.
     * @return a list of hardware names
     */
    private List<String> getHardwareNamesList() {
        try {
            @SuppressWarnings("unchecked")
            List<String> commandList = (List<String>) cmu.sendSynchronousCommand(
                new CommandRequest(destination, "listHardwareNames"),
                Duration.ofSeconds(10)
            );
            return commandList;
        } catch (Exception ex) {
            FCSLOG.error(" Unable to retrieve the list of hardware names.", ex);
            return Collections.emptyList();
        }
    }

    /**
     * Sends a command to retrieve the list of hardware connected to the given
     * bridge. It sends a command to the worker subsystem
     *
     * @param bridge the bridge to list the hardware connected to.
     * @param tcpProxy the tcpProxy associated to the bridge.
     * @return a list of hardware names
     */
    private List<String> getLoaderHardwareNamesList() {
        try {
            @SuppressWarnings("unchecked")
            List<String> hardwareList = (List<String>) cmu.sendSynchronousCommand(
                new CommandRequest(destination, "listLoaderHardwareNames"),
                Duration.ofSeconds(10)
            );
            return hardwareList;
        } catch (Exception ex) {
            FCSLOG.error(" Unable to retrieve the list of hardware names.", ex);
            return Collections.emptyList();
        }
    }

    /**
     * Sends a command to the worker subsystem to retrieve the list of filter
     * names. This list is then used to initialize the filterListPanels. When
     * there is no filter names list, it returns an empty list.
     *
     * @return
     */
    private List<String> getFilterNames() {
        try {
            @SuppressWarnings("unchecked")
            List<String> filterNames = (List<String>) cmu.sendSynchronousCommand(
                new CommandRequest(destination, "getFilterNames"),
                Duration.ofSeconds(10)
            );
            return filterNames;
        } catch (Exception ex) {
            FCSLOG.error(" Unable to retrieve the list of hardware names.", ex);
            return Collections.emptyList();
        }
    }

    private Map<String, String> getMobilNameControllerNameMap() {
        Map<String, String> ctlNames = new HashMap();
        try {
            ctlNames = (Map<String, String>) cmu.sendSynchronousCommand(
                    new CommandRequest(destination, "getMobilNameControllerNameMap"),
                    Duration.ofSeconds(10)
            );

        } catch (Exception ex) {
            FCSLOG.error(" Unable to retrieve the list of controllers names.", ex);
        }
        return ctlNames;
    }


    @Override
    public void disconnected(AgentInfo... agents) {
        for (AgentInfo agent : agents) {
            /* a subsystem is disconnecting from the bus*/
            if (!agent.getName().equals(this.destination)) {
                continue;
            }
            FCSLOG.warning(destination + " is disconnecting NOW");
            /*my subsystem is disconnecting from the bus.*/
            /*we have to reset all the panels*/
            generalPanel.resetPanel();
        }
    }

}
