
package org.lsst.ccs.subsystems.fcs.autochanger.ui;

import java.awt.Color;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import javax.swing.BorderFactory;
import javax.swing.JFrame;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JProgressBar;
import javax.swing.SwingUtilities;
import org.lsst.ccs.subsystems.fcs.StatusDataPublishedByAutochangerTruck;
import org.lsst.ccs.subsystems.fcs.StatusDataPublishedByEPOSController;
import org.lsst.ccs.subsystems.fcs.ui.commons.ControllerStatePanel;
import org.lsst.ccs.subsystems.fcs.ui.commons.DigitalSwitch;
import static org.lsst.ccs.subsystems.fcs.ui.commons.Tools.BIG_FONT;
import static org.lsst.ccs.subsystems.fcs.ui.commons.Tools.greenColor;
import static org.lsst.ccs.subsystems.fcs.ui.commons.Tools.insets_std;

/**
 * A panel for an AC truck.
 * There are 2 AC trucks.
 * @author virieux
 */
public class AutochangerTruckPanel extends JPanel {
    private static final long serialVersionUID = 6421842448844475453L;
    private final String truckName;

    //private final
    private final DigitalSwitch standbyDigitalSwitch = new DigitalSwitch();
    private final DigitalSwitch inTravelDigitalSwitch = new DigitalSwitch();
    private final DigitalSwitch handoffDigitalSwitch = new DigitalSwitch();
    private final DigitalSwitch onlineDigitalSwitch = new DigitalSwitch();
    private final DigitalSwitch inErrorDigitalSwitch = new DigitalSwitch();
    private final JLabel cajLabel = new JLabel("Truck position");
    private final JLabel standbyjLabel = new JLabel("STANDBY");
    private final JLabel inTraveljLabel = new JLabel("In Travel");
    private final JLabel handoffjLabel = new JLabel("HAND-OFF");
    private final JLabel onlinejLabel = new JLabel("ONLINE");
    private final JLabel inErrorjLabel = new JLabel();
    private final JLabel handoffSensorsErrorjLabel = new JLabel();
    private final JLabel onlineSensorsErrorjLabel = new JLabel();
    private final JLabel standbySensorsErrorjLabel = new JLabel();
    private static final String ERROR_IN_SENSORS = "SENSORS ERROR";
    private static final String NO_ERROR_IN_SENSORS = "NO ERROR";

    
    private final JProgressBar positionProgressBar = new JProgressBar(1, 0, 500);
    
    //parameters Panel
    private final JLabel ctlNameJLabel = new JLabel();
    private final JLabel actualPositionjLabel = new JLabel("0");
    private final JLabel currentjLabel = new JLabel("0");
    
    //Controller State Panel
    private final ControllerStatePanel controllerPanel = new ControllerStatePanel();
    
    // End of variables declaration 
  
    /**
     * A Runnable to update this Panel from data published on the STATUS bus.
     */
    class UpdateTruck implements Runnable {

        private final StatusDataPublishedByAutochangerTruck s;

        /**
         * Creates a class UpdateTruck from a Status published by autochanger truck.
         * @param status 
         */
        public UpdateTruck(StatusDataPublishedByAutochangerTruck status) {
            this.s = status;
        }
        
        /**
         * update Error Digital Switches and Label 
         */
        private void updateErrorComponents() {
            /*ERROR Digital Switch is red as soon as there is an error 
            on the carrier position sensors.*/
            if ((s.isHandoffInError() || s.isOnlineInError() || s.isStandbyInError())) {
                inErrorDigitalSwitch.setColor(Color.RED);
                inErrorjLabel.setText(ERROR_IN_SENSORS);
            } else {
                inErrorDigitalSwitch.setColor(greenColor);
                inErrorjLabel.setText(NO_ERROR_IN_SENSORS);
            }
            
            handoffSensorsErrorjLabel.setText(s.isHandoffInError() ? "<HTML>Handoff sensors<BR>in error</HTML>" : null);
            onlineSensorsErrorjLabel.setText(s.isOnlineInError() ? "<HTML>Online sensors<BR>in error</HTML>" : null);
            standbySensorsErrorjLabel.setText(s.isStandbyInError() ? "<HTML>Standby sensors<BR>in error</HTML>" : null);
        }

        @Override
        public void run() {
            handoffDigitalSwitch.updateColor(s.isHandoffInError(), s.getHandoffSensorValue());
            
            onlineDigitalSwitch.updateColor(s.isOnlineInError(), s.getOnlineSensorValue());

            standbyDigitalSwitch.updateColor(s.isStandbyInError(), s.getStandbySensorValue());

            inTravelDigitalSwitch.setColor(
                    (!s.getHandoffSensorValue()
                    && !s.getOnlineSensorValue()
                    && !s.getStandbySensorValue()) ? Color.YELLOW : Color.gray);
            
            updateErrorComponents();

            positionProgressBar.setValue(s.getPosition());
            positionProgressBar.setForeground(Color.PINK);

            //TODO move that to updateController
            actualPositionjLabel.setText(Integer.toString(s.getPosition()));
            currentjLabel.setText(Integer.toString(s.getCurrent()));
        }
    }

    class GuiInitialization implements Runnable {

        private final String ctlName;

        public GuiInitialization(String controllerName) {
            this.ctlName = controllerName;
        }

        @Override
        public void run() {
            ctlNameJLabel.setText(ctlName);
        }
    }

    class GuiInitializationFromConfigInfo implements Runnable {

        private final int standbyPosition;
        private final int onlinePosition;

        public GuiInitializationFromConfigInfo(int standbyPosition, int onlinePosition) {
            this.standbyPosition = standbyPosition;
            this.onlinePosition = onlinePosition;
        }

        @Override
        public void run() {
            /*initialize position progress bar*/
            positionProgressBar.setMinimum(standbyPosition);
            positionProgressBar.setMaximum(onlinePosition);
        }
    }
    
    /**
     * Creates new form AutochangerTrucksPanel
     * @param truckName
     */
    public AutochangerTruckPanel(String truckName) {
        this.truckName = truckName;
        initComponents();
    }

    /**
     * Initialize this Panel from configuration data. 
     * @param standbyPosition
     * @param onlinePosition
     */
    public void initializeGui(int standbyPosition, int onlinePosition) {
        SwingUtilities.invokeLater(new AutochangerTruckPanel.GuiInitializationFromConfigInfo(standbyPosition, onlinePosition));
    }

    public void initializeControllerName(String ctlName) {
        SwingUtilities.invokeLater(new GuiInitialization(ctlName));
    }

    public void updateTruck(StatusDataPublishedByAutochangerTruck status) {
        SwingUtilities.invokeLater(new AutochangerTruckPanel.UpdateTruck(status));
    }

    public void updateController(String controllerName, StatusDataPublishedByEPOSController s) {
        controllerPanel.update(controllerName, s);
    }

    // <editor-fold defaultstate="collapsed" desc="Form Description">                          
    private void initComponents() {
        setDefaultValues();

        /*
         **  Add graphical components to the main panel
         */
        setBorder(BorderFactory.createTitledBorder(
                BorderFactory.createLineBorder(Color.YELLOW,2),truckName));
        setLayout(new GridBagLayout());
        GridBagConstraints gbc = new GridBagConstraints();
        gbc.anchor = GridBagConstraints.NORTHWEST;
        gbc.insets = insets_std;
        gbc.gridx = 2;
        gbc.gridy = 0;
        cajLabel.setFont(BIG_FONT);
        cajLabel.setText("<HTML><CENTER>"+truckName + "<BR>position</CENTER></HTML>");
        add(cajLabel, gbc);

        
        //sensors in the 2 first columns 
        gbc.anchor = GridBagConstraints.NORTHEAST;
        gbc.gridx = 0;
        gbc.gridy = 1;
        add(standbyjLabel, gbc);
        gbc.gridx++;
        add(standbyDigitalSwitch, gbc);

        gbc.anchor = GridBagConstraints.EAST;
        gbc.gridx = 0;
        gbc.gridy++;
        add(inTraveljLabel, gbc);
        gbc.gridx++;
        add(inTravelDigitalSwitch, gbc);

        gbc.gridx = 0;
        gbc.gridy++;
        add(handoffjLabel, gbc);
        gbc.gridx++;
        add(handoffDigitalSwitch, gbc);

        gbc.gridx = 0;
        gbc.gridy++;
        add(onlinejLabel, gbc);
        gbc.gridx++;
        add(onlineDigitalSwitch, gbc);

        //carrier position progress bar in the column 2
        gbc.anchor = GridBagConstraints.CENTER;
        gbc.gridwidth = 1;
        gbc.gridheight = 4;
        gbc.fill = GridBagConstraints.VERTICAL;
        gbc.gridx = 2;
        gbc.gridy = 1;
        positionProgressBar.setBorderPainted(true);
        positionProgressBar.setValue(0);
        positionProgressBar.setStringPainted(true);
        add(positionProgressBar, gbc);
        gbc.gridheight = 1;        
        
        //sensor in Error Panel
        //creation Error Panel
        JPanel errorjPanel = new JPanel();
        errorjPanel.setLayout(new GridBagLayout());
        GridBagConstraints gbc1 = new GridBagConstraints();
        gbc1.insets = insets_std;
        inErrorDigitalSwitch.setSize(30);
        gbc1.gridx = 0;
        gbc1.gridy = 0;
        errorjPanel.add(inErrorDigitalSwitch, gbc1);
        gbc1.gridy++;
        errorjPanel.add(inErrorjLabel, gbc1);
        gbc1.gridy++;
        errorjPanel.add(standbySensorsErrorjLabel, gbc1);
        gbc1.gridy++;
        errorjPanel.add(handoffSensorsErrorjLabel, gbc1);
        //error panel on the grid
        gbc.gridx = 3;
        gbc.gridy = 4;
        gbc.anchor = GridBagConstraints.CENTER;
        gbc.gridheight = 1;
        add(errorjPanel, gbc);
        /**
         ******************************************************************************
         ** Parameters Panel
         * *****************************************************************************
         */
        JPanel paramPanel = new JPanel();
        //paramPanel.setPreferredSize(new Dimension(200,50));
        paramPanel.setBorder(BorderFactory.createLineBorder(Color.BLACK));
        paramPanel.setLayout(new GridBagLayout());
        GridBagConstraints gbc2 = new GridBagConstraints();
        gbc2.insets = insets_std;
        //first colomn
        gbc2.gridx = 0;
        gbc2.gridy = 0;
        gbc2.anchor = GridBagConstraints.CENTER;
        gbc2.gridwidth = 2;
        ctlNameJLabel.setFont(BIG_FONT);
        paramPanel.add(ctlNameJLabel, gbc2);
        gbc2.gridy++;
        
        gbc2.anchor = GridBagConstraints.LINE_START;
        gbc2.gridwidth = 1;
        paramPanel.add(new JLabel("<HTML>Position :<BR>(in um)</HTML>"), gbc2);
        gbc2.gridy++;
        paramPanel.add(new JLabel("<HTML>Current : <BR>(in mA) </HTML>"), gbc2);
        //second colomn
        gbc2.gridx++;
        gbc2.gridy = 1;
        gbc2.anchor = GridBagConstraints.LINE_END;
        paramPanel.add(actualPositionjLabel, gbc2);
        gbc2.gridy++;
        paramPanel.add(currentjLabel, gbc2);

        //parameters panel on the grid
        gbc.gridx = 3;
        gbc.gridy = 2;
        add(paramPanel, gbc);
        /**
         ******************************************************************************
         ** End of parameters Panel
         * *****************************************************************************
         */
        
        //Master Controller Fault panel on the grid
        gbc.gridx = 3;
        gbc.gridy = 0;
        gbc.gridheight = 2;
        add(controllerPanel, gbc);
    }
    
    private void setDefaultValues() {
        standbyDigitalSwitch.setColor(Color.gray);
        inTravelDigitalSwitch.setColor(Color.gray);
        handoffDigitalSwitch.setColor(Color.gray);
        onlineDigitalSwitch.setColor(Color.gray);
        inErrorDigitalSwitch.setColor(Color.GRAY);
        actualPositionjLabel.setText("0");
        currentjLabel.setText("0");
        inErrorjLabel.setText(NO_ERROR_IN_SENSORS);
        this.ctlNameJLabel.setText("UNKNOWN CTL NAME");
    }
    
    /**
     * Reset components to default values.
     */
    public void resetPanel() {
        setDefaultValues();
    }
    
    /**
     * For tests.
     * @param argv 
     */
    public static void main(String[] argv) {

        AutochangerTruckPanel d = new AutochangerTruckPanel("acTruckXminus");
        JFrame frame = new JFrame("Autochanger Truck Main Panel");
        frame.setContentPane(d);
        frame.pack();
        frame.setVisible(true);
        frame.setDefaultCloseOperation(JFrame.EXIT_ON_CLOSE);

    }    
    
}
