
package org.lsst.ccs.subsystems.fcs.carousel.ui;

import java.awt.Color;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.event.ActionEvent;
import java.text.DecimalFormat;
import javax.swing.BorderFactory;
import javax.swing.JButton;
import javax.swing.JFrame;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JTextField;
import javax.swing.SwingUtilities;
import org.lsst.ccs.bus.data.ConfigurationInfo;
import static org.lsst.ccs.subsystems.fcs.FCSCst.FCSLOG;
import static org.lsst.ccs.subsystems.fcs.FCSCst.NO_FILTER;
import static org.lsst.ccs.subsystems.fcs.FCSCst.AT_STANDBY;
import static org.lsst.ccs.subsystems.fcs.FCSCst.NOT_AT_STANDBY;
import static org.lsst.ccs.subsystems.fcs.FCSCst.UNKNOWN;
import org.lsst.ccs.subsystems.fcs.StatusDataPublishedByCarousel;
import org.lsst.ccs.subsystems.fcs.StatusDataPublishedByEPOSController;
import org.lsst.ccs.subsystems.fcs.ui.commons.DigitalSwitch;
import static org.lsst.ccs.subsystems.fcs.ui.commons.Tools.BIG_FONT;
import static org.lsst.ccs.subsystems.fcs.ui.commons.Tools.UNKNOWN_STATE;
import static org.lsst.ccs.subsystems.fcs.ui.commons.Tools.ZERO_VALUE;
import static org.lsst.ccs.subsystems.fcs.ui.commons.Tools.greenColor;
import static org.lsst.ccs.subsystems.fcs.ui.commons.Tools.insets_std;

/**
 * This view represents the carousel in a general design.
 * @author virieux
 */
public class CarouselGeneralViewPanel extends JPanel {
    private static final long serialVersionUID = 3579024322081964467L;

    private static final String NONE = "None";

    private InterfaceCarouselGUI subs;

    private final DecimalFormat formatter = new DecimalFormat("######.000");

    /*General Informations Panel (Configuration parameters)*/
    private final JPanel generalInfoPanel = new JPanel();
    private final JLabel profileVelocityLabel = new JLabel();
    private final JLabel profileAccelerationLabel = new JLabel();
    private final JLabel profileDecelerationLabel = new JLabel();
    private final JLabel rotationTimeoutLabel = new JLabel();

    /*Actual parameters (dynamic parameters */
    private final JPanel actualParametersPanel = new JPanel();
    private final JLabel homingLabel = new JLabel();
    private final JLabel actualPositionLabel = new JLabel();
    private final JLabel positionSensorTypeLabel = new JLabel();
    private final JLabel actualSpeedLabel = new JLabel();
    private final JLabel actualCurrentLabel = new JLabel();
    private final JLabel standbyLabel = new JLabel();
    private final DigitalSwitch standbySwitch = new DigitalSwitch();
    private final JLabel socketAtStandbyLabel = new JLabel();
    private final JLabel filterAtStandbyLabel = new JLabel();

    /*Carousel Controller Panel */
    private final JPanel controllerPanel = new JPanel();
    private final DigitalSwitch controllerSwitch = new DigitalSwitch();
    private final JLabel enableStatusLabel = new JLabel();
    private final JLabel eposStateLabel = new JLabel();

    /*Carousel Disk*/
    private final CarouselDisk carouselDisk = new CarouselDisk();

    /*Control Panel*/
    private final JPanel controlPanel = new JPanel();
    private final JButton updatePositionButton = new JButton("Update Position");
    private final JButton rotateButton = new JButton("RotateSocketToStandby");
    private final JButton slowModeButton = new JButton("SetSlowMode");
    private final JButton fastModeButton = new JButton("SetFastMode");
    private final JButton homingButton = new JButton("Homing");
    private final JTextField socketNBField = new JTextField(1);
    private final JLabel socketLabel = new JLabel("enter socket nb:");
    /* socket to rotate at Standby */
    private int socketID;

    /* Temperatures panel*/
    private final JPanel temperaturesPanel = new JPanel();
    private final JLabel temp1Label = new JLabel();
    private final JLabel temp2Label = new JLabel();
    private final JLabel temp3Label = new JLabel();
    private final JLabel temp4Label = new JLabel();

    /* Sensors panel*/
    private final JPanel sensorsPanel = new JPanel();
    private final JLabel s1Label = new JLabel();
    private final JLabel s2Label = new JLabel();
    private final JLabel s3Label = new JLabel();
    private final JLabel s4Label = new JLabel();
    private final JLabel s5Label = new JLabel();
    private final JLabel s6Label = new JLabel();
    private final JLabel s7Label = new JLabel();
    private final JLabel s8Label = new JLabel();



    public CarouselGeneralViewPanel() {
        initComponents();
    }

    protected void setSubsystem(InterfaceCarouselGUI subs) {
        this.subs = subs;
    }

    private void setDefaultValues() {
        profileVelocityLabel.setText(ZERO_VALUE);
        profileAccelerationLabel.setText(ZERO_VALUE);
        profileDecelerationLabel.setText(ZERO_VALUE);
        rotationTimeoutLabel.setText(ZERO_VALUE);
        homingLabel.setText("NOT DONE");
        homingLabel.setForeground(Color.RED);
        actualPositionLabel.setText(ZERO_VALUE);
        actualSpeedLabel.setText(ZERO_VALUE);
        actualCurrentLabel.setText(ZERO_VALUE);
        standbyLabel.setText(NOT_AT_STANDBY);
        standbySwitch.setColor(Color.gray);
        socketAtStandbyLabel.setText(NONE);
        filterAtStandbyLabel.setText(NO_FILTER);
        temp1Label.setText(ZERO_VALUE);
        temp2Label.setText(ZERO_VALUE);
        temp3Label.setText(ZERO_VALUE);
        temp4Label.setText(ZERO_VALUE);
        s1Label.setText(ZERO_VALUE);
        s2Label.setText(ZERO_VALUE);
        s3Label.setText(ZERO_VALUE);
        s4Label.setText(ZERO_VALUE);
        s5Label.setText(ZERO_VALUE);
        s6Label.setText(ZERO_VALUE);
        s7Label.setText(ZERO_VALUE);
        s8Label.setText(ZERO_VALUE);
        controllerSwitch.setColor(Color.gray);
        enableStatusLabel.setText(UNKNOWN);
        eposStateLabel.setText(UNKNOWN_STATE);
    }

    protected void initializeGui(ConfigurationInfo configInfo) {
        //here initialize this Panel from configInfo
    }

    protected void update(StatusDataPublishedByCarousel s) {
        FCSLOG.debug(getName() + " updating from carousel data:");
        SwingUtilities.invokeLater(new Update(s));
    }

    protected void update(String filterName, String socketName) {
        FCSLOG.debug(getName() + " updating from filter data:");
        SwingUtilities.invokeLater(new UpdateFilter(filterName, socketName));
    }

    protected void update(StatusDataPublishedByEPOSController s) {
        FCSLOG.debug(getName() + " updating from carousel controller data:");
        SwingUtilities.invokeLater(new UpdateController(s));
    }

    /**
     * A Runnable class to update this panel from an object StatusDataPublishedByCarousel.
     */
    public class Update implements Runnable {
        private final StatusDataPublishedByCarousel s;

        public Update(StatusDataPublishedByCarousel s) {
            this.s = s;
        }

        @Override
        public void run() {
            /*Update of actual parameters Panel*/
            if (s.isHomingDone()) {
                homingLabel.setText("DONE");
                homingLabel.setForeground(greenColor);
            } else {
                homingLabel.setText("TO_BE_DONE");
                homingLabel.setForeground(Color.RED);
            }
            actualPositionLabel.setText(Integer.toString(s.getPosition()));
            String sensorType;
            switch (s.getPositionSensorType()) {
                case 4:
                    sensorType = "Ssi";
                    break;
                case 8:
                    sensorType = "Inc";
                    break;
                default:
                    sensorType = Integer.toString(s.getPositionSensorType());
                    break;
            }
            positionSensorTypeLabel.setText(sensorType);
            actualSpeedLabel.setText(Double.toString(s.getVelocity()));
            actualCurrentLabel.setText(Double.toString(s.getCurrent()));
            profileVelocityLabel.setText(Long.toString(s.getProfileVelocity()));
            profileAccelerationLabel.setText(Long.toString(s.getProfileAcceleration()));
            profileDecelerationLabel.setText(Long.toString(s.getProfileDeceleration()));
            rotationTimeoutLabel.setText(Long.toString(s.getRotationTimeout()));
            if (s.isAtStandby()) {
                standbyLabel.setText(AT_STANDBY);
                standbySwitch.setColor(greenColor);
                socketAtStandbyLabel.setText(s.getSocketAtStandbyName());
                socketAtStandbyLabel.setFont(BIG_FONT);
                filterAtStandbyLabel.setText(s.isEmptyAtStandby() ? NO_FILTER : s.getFilterAtStandbyName());
                filterAtStandbyLabel.setFont(BIG_FONT);

            } else {
                standbyLabel.setText(NOT_AT_STANDBY);
                standbySwitch.setColor(Color.orange);
                socketAtStandbyLabel.setText(s.getSocketAtStandbyName());
                filterAtStandbyLabel.setText(NONE);
            }

            carouselDisk.update(s);

            /* Temperatures read on pt100 */
            temp1Label.setText(formatter.format((double) s.getTemperature1() / 1000));
            temp2Label.setText(formatter.format((double) s.getTemperature2() / 1000));
            temp3Label.setText(formatter.format((double) s.getTemperature3() / 1000));
            temp4Label.setText(formatter.format((double) s.getTemperature4() / 1000));

            /* Sensors read on ai814 */
            s1Label.setText(Long.toString(s.getSensor1()));
            s2Label.setText(Long.toString(s.getSensor2()));
            s3Label.setText(Long.toString(s.getSensor3()));
            s4Label.setText(Long.toString(s.getSensor4()));
            s5Label.setText(Long.toString(s.getSensor5()));
            s6Label.setText(Long.toString(s.getSensor6()));
            s7Label.setText(Long.toString(s.getSensor7()));
            s8Label.setText(Long.toString(s.getSensor8()));
        }

    }

    /**
     * A Runnable to update carouselDisk with data published by a filter.
     */
    public class UpdateFilter implements Runnable {

        private final String filterName;
        private final String socketName;

        public UpdateFilter(String filterName, String socketName) {
            this.filterName = filterName;
            this.socketName = socketName;
        }

        @Override
        public void run() {
                carouselDisk.update(filterName, socketName);
        }
    }

    /**
     * A Runnable to update controller panel with data published by carousel
     * controller.
     */
    public class UpdateController implements Runnable {

        private final StatusDataPublishedByEPOSController s;

        private UpdateController(StatusDataPublishedByEPOSController s) {
            this.s = s;
        }

        @Override
        public void run() {
            if (!s.isBooted()) {
                controllerSwitch.setColor(Color.RED);
                enableStatusLabel.setText("not booted");
                eposStateLabel.setText("");
            } else if (s.isEnabled()) {
                controllerSwitch.setColor(greenColor);
                enableStatusLabel.setText("enable");
            } else {
                controllerSwitch.setColor(Color.ORANGE);
                enableStatusLabel.setText("disable");
            }
            if (s.isBooted() && s.getState() != null) {
                eposStateLabel.setText(s.getState().name());
            }
        }
    }



    // <editor-fold defaultstate="collapsed" desc="Form Description">
    private void initComponents() {
        setName("carouselGeneralViewPanel");

        /**
         ******************************************************************************
         ** Buttons initialization
         * *****************************************************************************
         */
        updatePositionButton.addActionListener((ActionEvent evt) -> {
            updatePositionButtonActionPerformed(evt);
        });

        slowModeButton.addActionListener((ActionEvent evt) -> {
            slowModeButtonActionPerformed(evt);
        });

        fastModeButton.addActionListener((ActionEvent evt) -> {
            fastModeButtonActionPerformed(evt);
        });

        homingButton.addActionListener((ActionEvent evt) -> {
            homingButtonActionPerformed(evt);
        });


        rotateButton.addActionListener((ActionEvent evt) -> {
            rotateButtonActionPerformed(evt);
        });

        /**
         ******************************************************************************
         ** end of Buttons initialization
         * *****************************************************************************
         */


        setDefaultValues();
        /**
         ******************************************************************************
         ** General Informations Panel
         * *****************************************************************************
         */
        generalInfoPanel.setBorder(BorderFactory.createTitledBorder("General Informations"));
        generalInfoPanel.setLayout(new GridBagLayout());
        GridBagConstraints gbc = new GridBagConstraints();
        gbc.insets = insets_std;
        //first colomn
        gbc.gridx = 0;
        gbc.gridy = 0;
        gbc.anchor = GridBagConstraints.LINE_START;

        generalInfoPanel.add(new JLabel("ProfileVelocity:"), gbc);
        gbc.gridy++;
        generalInfoPanel.add(new JLabel("ProfileAcceleration:"), gbc);
        gbc.gridy++;
        generalInfoPanel.add(new JLabel("ProfileDeceleration:"), gbc);
        gbc.gridy++;
        generalInfoPanel.add(new JLabel("RotationTimeout:"), gbc);

        //second colomn
        gbc.gridx++;
        gbc.gridy = 0;
        gbc.anchor = GridBagConstraints.LINE_END;
        generalInfoPanel.add(profileVelocityLabel, gbc);
        gbc.gridy++;
        generalInfoPanel.add(profileAccelerationLabel, gbc);
        gbc.gridy++;
        generalInfoPanel.add(profileDecelerationLabel, gbc);
        gbc.gridy++;
        generalInfoPanel.add(rotationTimeoutLabel, gbc);

        /**
         ******************************************************************************
         ** end of General Informations Panel
         * *****************************************************************************
         */

        /**
         ******************************************************************************
         ** Actual Position Panel
         * *****************************************************************************
         */
        actualParametersPanel.setBorder(BorderFactory.createLineBorder(Color.yellow));
        actualParametersPanel.setLayout(new GridBagLayout());
        GridBagConstraints gbc1 = new GridBagConstraints();
        gbc1.insets = insets_std;
        //first colomn
        gbc1.gridx = 0;
        gbc1.gridy = 0;
        gbc1.anchor = GridBagConstraints.LINE_START;
        actualParametersPanel.add(new JLabel("Homing: "),gbc1);
        gbc1.gridy++;
        actualParametersPanel.add(new JLabel("Position: "),gbc1);
        gbc1.gridy++;
        actualParametersPanel.add(new JLabel("P. Sensor Type: "), gbc1);
        gbc1.gridy++;
        actualParametersPanel.add(new JLabel("Speed: "),gbc1);
        gbc1.gridy++;
        actualParametersPanel.add(new JLabel("Current: "),gbc1);
        gbc1.gridy++;
        actualParametersPanel.add(standbyLabel,gbc1);
        gbc1.gridy++;
        actualParametersPanel.add(new JLabel("Socket at STANDBY: "),gbc1);
        gbc1.gridy++;
        actualParametersPanel.add(new JLabel("Filter at STANDBY: "),gbc1);
        //second colomn
        gbc1.gridx++;
        gbc1.gridy = 0;
        gbc1.anchor = GridBagConstraints.LINE_END;
        actualParametersPanel.add(homingLabel, gbc1);
        gbc1.gridy++;
        actualParametersPanel.add(actualPositionLabel, gbc1);
        gbc1.gridy++;
        actualParametersPanel.add(positionSensorTypeLabel, gbc1);
        gbc1.gridy++;
        actualParametersPanel.add(actualSpeedLabel, gbc1);
        gbc1.gridy++;
        actualParametersPanel.add(actualCurrentLabel, gbc1);
        gbc1.gridy++;
        standbySwitch.setColor(Color.gray);
        actualParametersPanel.add(standbySwitch,gbc1);
        gbc1.gridy++;
        actualParametersPanel.add(socketAtStandbyLabel,gbc1);
        gbc1.gridy++;
        actualParametersPanel.add(filterAtStandbyLabel,gbc1);

        /**
         ******************************************************************************
         ** end of Actual Position Panel
         * *****************************************************************************
         */
        /**
         ******************************************************************************
         ** Carousel Controller Panel
         * *****************************************************************************
         */
        controllerPanel.setBorder(BorderFactory.createLineBorder(Color.MAGENTA, WIDTH + 5));
        controllerPanel.setLayout(new GridBagLayout());
        gbc = new GridBagConstraints();
        gbc.insets = insets_std;
        gbc.anchor = GridBagConstraints.LINE_START;
        gbc.gridx = 0;
        gbc.gridy = 0;
        controllerPanel.add(new JLabel("Controller status :"), gbc);
        gbc.gridy++;
        gbc.anchor = GridBagConstraints.CENTER;
        controllerSwitch.setSize(40);
        gbc.gridheight = 2;
        controllerPanel.add(controllerSwitch, gbc);
        gbc.gridheight = 1;
        gbc.gridx++;
        controllerPanel.add(enableStatusLabel, gbc);
        gbc.gridy++;
        controllerPanel.add(eposStateLabel, gbc);
        /**
         ******************************************************************************
         ** end of Carousel Controller Panel
         * *****************************************************************************
         */

        /**
         ******************************************************************************
         ** Temperatures Panel
         * *****************************************************************************
         */
        temperaturesPanel.setBorder(BorderFactory.createTitledBorder("Temperatures (pt100)"));
        temperaturesPanel.setLayout(new GridBagLayout());
        gbc = new GridBagConstraints();
        gbc.insets = insets_std;
        //first colomn
        gbc.gridx = 0;
        gbc.gridy = 0;
        gbc.anchor = GridBagConstraints.LINE_START;

        temperaturesPanel.add(new JLabel("Temperature1:"), gbc);
        gbc.gridy++;
        temperaturesPanel.add(new JLabel("Temperature2:"), gbc);
        gbc.gridy++;
        temperaturesPanel.add(new JLabel("Temperature3:"), gbc);
        gbc.gridy++;
        temperaturesPanel.add(new JLabel("Temperature4:"), gbc);

        //second colomn
        gbc.gridx++;
        gbc.gridy = 0;
        gbc.anchor = GridBagConstraints.LINE_END;
        temperaturesPanel.add(temp1Label, gbc);
        gbc.gridy++;
        temperaturesPanel.add(temp2Label, gbc);
        gbc.gridy++;
        temperaturesPanel.add(temp3Label, gbc);
        gbc.gridy++;
        temperaturesPanel.add(temp4Label, gbc);

        /**
         ******************************************************************************
         ** end of Temperatures Panel
         * *****************************************************************************
         */
        /**
         ******************************************************************************
         ** Sensors Panel
         * *****************************************************************************
         */
        sensorsPanel.setBorder(BorderFactory.createTitledBorder("Sensors (ai814)"));
        sensorsPanel.setLayout(new GridBagLayout());
        gbc = new GridBagConstraints();
        gbc.insets = insets_std;
        //first colomn
        gbc.gridx = 0;
        gbc.gridy = 0;
        gbc.anchor = GridBagConstraints.LINE_START;

        sensorsPanel.add(new JLabel("Sensor1:"), gbc);
        gbc.gridy++;
        sensorsPanel.add(new JLabel("Sensor2:"), gbc);
        gbc.gridy++;
        sensorsPanel.add(new JLabel("Sensor3:"), gbc);
        gbc.gridy++;
        sensorsPanel.add(new JLabel("Sensor4:"), gbc);
        gbc.gridy++;
        sensorsPanel.add(new JLabel("Sensor5:"), gbc);
        gbc.gridy++;
        sensorsPanel.add(new JLabel("Sensor6:"), gbc);
        gbc.gridy++;
        sensorsPanel.add(new JLabel("Sensor7:"), gbc);
        gbc.gridy++;
        sensorsPanel.add(new JLabel("Sensor8:"), gbc);

        //second colomn
        gbc.gridx++;
        gbc.gridy = 0;
        gbc.anchor = GridBagConstraints.LINE_END;
        sensorsPanel.add(s1Label, gbc);
        gbc.gridy++;
        sensorsPanel.add(s2Label, gbc);
        gbc.gridy++;
        sensorsPanel.add(s3Label, gbc);
        gbc.gridy++;
        sensorsPanel.add(s4Label, gbc);
        gbc.gridy++;
        sensorsPanel.add(s5Label, gbc);
        gbc.gridy++;
        sensorsPanel.add(s6Label, gbc);
        gbc.gridy++;
        sensorsPanel.add(s7Label, gbc);
        gbc.gridy++;
        sensorsPanel.add(s8Label, gbc);

        /**
         ******************************************************************************
         ** end of Sensors Panel
         * *****************************************************************************
         */
        /**
         ******************************************************************************
         ** Control Panel
         * *****************************************************************************
         */
        controlPanel.setLayout(new GridBagLayout());
        GridBagConstraints gbc4 = new GridBagConstraints();
        gbc4.insets = insets_std;
        gbc4.anchor = GridBagConstraints.FIRST_LINE_START;
        gbc4.fill = GridBagConstraints.HORIZONTAL;
        gbc4.gridx = 0;
        gbc4.gridy = 0;
        gbc4.gridwidth = 2;
        controlPanel.add(homingButton, gbc4);
        gbc4.gridy++;
        controlPanel.add(updatePositionButton, gbc4);
        gbc4.gridy++;
        controlPanel.add(slowModeButton, gbc4);
        gbc4.gridy++;
        controlPanel.add(fastModeButton, gbc4);
        gbc4.gridy++;
        gbc4.anchor = GridBagConstraints.EAST;
        gbc4.gridwidth = 1;
        controlPanel.add(socketLabel, gbc4);
        gbc4.gridx++;
        gbc4.anchor = GridBagConstraints.WEST;
        gbc4.fill = GridBagConstraints.NONE;
        controlPanel.add(socketNBField, gbc4);
        gbc4.gridy++;
        gbc4.gridx = 0;
        gbc4.gridwidth = 2;
        gbc4.fill = GridBagConstraints.HORIZONTAL;
        gbc4.anchor = GridBagConstraints.FIRST_LINE_START;
        controlPanel.add(rotateButton, gbc4);
        /**
         ******************************************************************************
         ** end of Control Panel
         * *****************************************************************************
         */

        /**
         ******************************************************************************
         ** Whole Panel
         * *****************************************************************************
         */
        /*General Information Panel*/
        setBorder(BorderFactory.createTitledBorder("CAROUSEL GENERAL VIEW"));
        setLayout(new java.awt.GridBagLayout());
        gbc = new GridBagConstraints();
        gbc.insets = insets_std;
        gbc.anchor = GridBagConstraints.CENTER;

        gbc.gridx = 0;
        gbc.gridy = 0;
        gbc.gridheight = 3;
        gbc.gridwidth = 2;
        add(carouselDisk, gbc);
        gbc.gridheight = 1;
        gbc.gridwidth = 1;

        gbc.gridx = 2;
        add(generalInfoPanel, gbc);

        gbc.gridy++;
        add(actualParametersPanel, gbc);

        gbc.gridy++;
        add(controllerPanel, gbc);

        gbc.gridy++;
        add(controlPanel, gbc);

        gbc.gridx = 0;
        gbc.gridy = 3;
        add(temperaturesPanel, gbc);

        gbc.gridx++;
        add(sensorsPanel, gbc);

        /**
         ******************************************************************************
         ** end of Whole Panel
         * *****************************************************************************
         */
    }

    private void updatePositionButtonActionPerformed(ActionEvent evt) {
        subs.updatePosition();
    }

    private void slowModeButtonActionPerformed(ActionEvent evt) {
        subs.setSlowMode();
    }

    private void fastModeButtonActionPerformed(ActionEvent evt) {
        subs.setFastMode();
    }

    private void homingButtonActionPerformed(ActionEvent evt) {
        subs.homing();
    }

    private void rotateButtonActionPerformed(ActionEvent evt) {
        this.socketID = Integer.parseInt(socketNBField.getText());
        subs.rotate(socketID);
    }


    /**
     * Reset Panel to default values when subsystem is shutdown.
     */
    public void resetPanel() {
        setDefaultValues();
        carouselDisk.resetPanel();
    }

    @Override
    public String toString() {
        return "Carousel General View";
    }

    /**
     * Just to test the Panel.
     * @param argv
     */
    public static void main(String[] argv) {

        CarouselGeneralViewPanel d = new CarouselGeneralViewPanel();
        JFrame frame = new JFrame("Test");
        frame.setContentPane(d);
        frame.pack();
        frame.setVisible(true);
        frame.setDefaultCloseOperation(JFrame.EXIT_ON_CLOSE);

    }

}
