package org.lsst.ccs.subsystems.fcs.carousel.ui;

import org.lsst.ccs.subsystems.fcs.ui.commons.HYTTC580Panel;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.TreeMap;
import javax.swing.SwingUtilities;
import org.lsst.ccs.bus.data.ConfigurationInfo;
import org.lsst.ccs.bus.data.KeyValueData;
import static org.lsst.ccs.subsystems.fcs.FCSCst.CHANGER_TCPPROXY_NAME;
import static org.lsst.ccs.subsystems.fcs.FCSCst.SOCKET_NAME;
import org.lsst.ccs.subsystems.fcs.StatusDataPublishedByCarousel;
import org.lsst.ccs.subsystems.fcs.StatusDataPublishedByCarouselClamp;
import org.lsst.ccs.subsystems.fcs.StatusDataPublishedByCarouselSocket;
import org.lsst.ccs.subsystems.fcs.StatusDataPublishedByEPOSController;
import org.lsst.ccs.subsystems.fcs.StatusDataPublishedByHYTTC580;
import org.lsst.ccs.subsystems.fcs.StatusDataPublishedByHardware;
import org.lsst.ccs.subsystems.fcs.ui.commons.CanOpenDevicePanel;
import org.lsst.ccs.subsystems.fcs.ui.commons.EPOSControllerPanel;
import org.lsst.ccs.subsystems.fcs.ui.commons.GatherPanel;

/**
 * This panel gathers all the panels needed for the carousel GUI. Except the
 * subsystemPanel which is handled by the GeneralPanel.
 *
 * It takes care of initialization and update of the panels.
 *
 * @author virieux
 */
public class CarouselGatherPanel extends GatherPanel {

    private static final long serialVersionUID = -975949078768889254L;

    private final String[] socketNames = {"socket1", "socket2", "socket3", "socket4", "socket5"};
    private final CarouselGeneralViewPanel caGeneralViewPanel = new CarouselGeneralViewPanel();
    protected transient final Map<String, CarouselSocketPanel> socketPanels = new TreeMap<>();

    protected InterfaceCarouselGUI subs;

    //CANopen devices panels
    private EPOSControllerPanel clampXminusControllerPanel;
    private EPOSControllerPanel clampXplusControllerPanel;
    private EPOSControllerPanel carouselControllerPanel;
    private HYTTC580Panel hyTTCPanel;
    private CanOpenDevicePanel cancbxai814Panel;
    private CanOpenDevicePanel cancbxpt100Panel;

    //Filters panels
    private FilterListPanel filterListPanel;



    /**
     * A Runnable class to initialize the GUI with data coming from the filter
     * names list.
     */
    private class GuiInitializationFilterListPanel implements Runnable {

        private final ConfigurationInfo configInfo;
        private final List<String> filterNamesList;

        public GuiInitializationFilterListPanel(ConfigurationInfo configInfo, List<String> filterNamesList) {
            this.configInfo = configInfo;
            /*initialize this.filterNamesList from a copy of the List<String> argument.*/
            this.filterNamesList = new ArrayList<>();
            this.filterNamesList.addAll(filterNamesList);
        }

        @Override
        public void run() {
            /**
             * Filter List Panel
             */
            filterListPanel.initializeGui(configInfo, filterNamesList);
        }

    }

    /**
     * Initialization of the GUI for data stored in tcpConfiguration. *
     */
    private class GuiInitialization implements Runnable {

        private final ConfigurationInfo configInfo;

        /**
         * Create a new GuiInitialization from a CarouselConfig and a
         * ConfigurationInfo. The CarouselConfig should be suppressed and
         * initialization of the GUI done from ConfigurationInfo only.
         *
         * @param configInfo
         */
        public GuiInitialization(ConfigurationInfo configInfo) {
            this.configInfo = configInfo;
        }

        @Override
        public void run() {
            /**
             * ClampXminus Controller Panel *
             */
            clampXminusControllerPanel.initializeGui(configInfo);

            /**
             * ClampXplus Controller Panel *
             */
            clampXplusControllerPanel.initializeGui(configInfo);

            /**
             * Carousel Controller Panel *
             */
            carouselControllerPanel.initializeGui(configInfo);

            /**
             * CANOpen HY-TTC Panel*
             */
            hyTTCPanel.initializeGui(configInfo);

            /**
             * CANOpen CAN-CBX-AI814 Panel*
             */
            cancbxai814Panel.initializeGui(configInfo);

            /**
             * CANOpen CAN-CBX-PT100 Panel*
             */
            cancbxpt100Panel.initializeGui(configInfo);

            //** Carousel General View Panel **/
            caGeneralViewPanel.initializeGui(configInfo);

            /**
             * Carousel Sockets Panel *
             */
            socketPanels.entrySet().stream().forEach(entry -> {
                entry.getValue().initializeGui(configInfo);
            });

        }
    }

    /**
     * Build a new CarouselGatherPanel with a tcpProxy name and initialize
     * components.
     *
     * @param tcpProxyName
     */
    public CarouselGatherPanel(String tcpProxyName) {
        super(tcpProxyName);
        initPanels();
    }

    /**
     * Build a new CarouselGatherPanel with a tcpProxy name and a CarouselGUI.
     * It initializes components.
     *
     * @param subs
     * @param tcpProxyName
     */
    public CarouselGatherPanel(InterfaceCarouselGUI subs, String tcpProxyName) {
        super(tcpProxyName);
        initPanels();
        this.subs = subs;
        this.clampXminusControllerPanel.setSubsystem(subs);
        this.clampXplusControllerPanel.setSubsystem(subs);
        this.carouselControllerPanel.setSubsystem(subs);
        this.caGeneralViewPanel.setSubsystem(subs);
        socketPanels.entrySet().stream().forEach((entry) -> {
            entry.getValue().setSubsystem(subs);
        });
        this.filterListPanel.setSubsystem(subs);
    }

    public EPOSControllerPanel getClampXminusControllerPanel() {
        return clampXminusControllerPanel;
    }

    public EPOSControllerPanel getClampXplusControllerPanel() {
        return clampXplusControllerPanel;
    }

    public EPOSControllerPanel getCarouselControllerPanel() {
        return carouselControllerPanel;
    }

    public HYTTC580Panel getHyTTCPanel() {
        return hyTTCPanel;
    }

    public CanOpenDevicePanel getCancbxAI814Panel() {
        return cancbxai814Panel;
    }

    public CanOpenDevicePanel getCancbxPT100Panel() {
        return cancbxpt100Panel;
    }

    public CarouselGeneralViewPanel getCaGeneralViewPanel() {
        return caGeneralViewPanel;
    }

    public FilterListPanel getFilterListPanel() {
        return filterListPanel;
    }

    public Map<String, CarouselSocketPanel> getSocketPanels() {
        return socketPanels;
    }



    @Override
    public void  updateFromStatusData(KeyValueData data) {
        
        if (CHANGER_TCPPROXY_NAME.equals(data.getKey())) {
            updateCanOpenHardwarePanel((int)data.getValue());

        } else if (data.getValue() instanceof StatusDataPublishedByEPOSController) {
            updateControllerPanel(data.getKey(), (StatusDataPublishedByEPOSController) data.getValue());
            updateCanOpenHardwarePanel(data.getKey(),(StatusDataPublishedByHardware) data.getValue());
            
        } else if (data.getValue() instanceof StatusDataPublishedByHYTTC580) {
            updateHYTTC580Panel(data.getKey(), (StatusDataPublishedByHYTTC580) data.getValue());
            updateCanOpenHardwarePanel(data.getKey(),(StatusDataPublishedByHardware) data.getValue());            
            
        } else if (data.getValue() instanceof StatusDataPublishedByHardware) { 
            updateCanOpenHardwarePanel(data.getKey(),(StatusDataPublishedByHardware) data.getValue());

        } else if (data.getValue() instanceof StatusDataPublishedByCarouselClamp) {
            updateSocketPanels(data.getKey(), (StatusDataPublishedByCarouselClamp) data.getValue());

        } else if (data.getValue() instanceof StatusDataPublishedByCarouselSocket) {
            updateSocketPanels(data.getKey(), (StatusDataPublishedByCarouselSocket) data.getValue());
            String filterName = ((StatusDataPublishedByCarouselSocket) data.getValue()).getFilterName();
            String socketName = SOCKET_NAME + ((StatusDataPublishedByCarouselSocket) data.getValue()).getSocketID();
            updateFilterListPanel(filterName, socketName);
            caGeneralViewPanel.update(filterName, socketName);

        } else if (data.getValue() instanceof StatusDataPublishedByCarousel) {
            FCSLOG.debug("STATUS data coming from carousel.");
            caGeneralViewPanel.update((StatusDataPublishedByCarousel) data.getValue());
            updateSocketPanels((StatusDataPublishedByCarousel) data.getValue());
        } 
    }
    
    @Override
    public void updateFromConfigurationInfo(ConfigurationInfo configInfo) {
        SwingUtilities.invokeLater(new GuiInitialization(configInfo));
    }

    @Override
    public void initializeGUIWithConfigInfo(ConfigurationInfo configInfo, 
            List<String> hardwareNames) {
        super.initializeGUIWithConfigInfo(configInfo, hardwareNames);
        SwingUtilities.invokeLater(new GuiInitialization(configInfo));
    }

    /**
     *
     * @param configInfo
     * @param filterNamesList
     */
    @Override
    public void initializeGUIWithFiltersInfo(ConfigurationInfo configInfo, List<String> filterNamesList) {
        SwingUtilities.invokeLater(new GuiInitializationFilterListPanel(configInfo, filterNamesList));
    }
    
    /**
     * updates hardware information (fields booted, initialized) from StatusDataPublishedByHardware object.
     * @param deviceToUpdate
     * @param status 
     */
    @Override
    public void updateCanOpenHardwarePanel(String deviceToUpdate, StatusDataPublishedByHardware status) {
        super.updateCanOpenHardwarePanel(deviceToUpdate,status);
        cancbxai814Panel.updateCanOpenDevice(deviceToUpdate, (StatusDataPublishedByHardware) status);
        cancbxpt100Panel.updateCanOpenDevice(deviceToUpdate, (StatusDataPublishedByHardware) status);
    }   
    
    public void updateHYTTC580Panel(String deviceToUpdate, StatusDataPublishedByHYTTC580 status) {
        hyTTCPanel.updateDevice(deviceToUpdate, status);
    }

    /**
     * Update the controller Panels from data coming from STATUS bus.
     *
     * @param controllerName
     * @param s
     */
    protected void updateControllerPanel(String controllerName, StatusDataPublishedByEPOSController s) {
        switch (controllerName) {
            case "clampXminusController":
                clampXminusControllerPanel.updateController("clampXminusController",s);
                break;

            case "clampXplusController":
                clampXplusControllerPanel.updateController("clampXplusController",s);
                break;

            case "carouselController":
                carouselControllerPanel.updateController("carouselController", s);
                break;

            default:
                //TODO get rid of this log because it can be a controller for autochanger or for loader.
                FCSLOG.finest(this.getName() + " :received an unknown controller name on STATUS bus:"
                        + controllerName);
                break;
        }
    }

    /**
     * Update the filterListPanel from a StatusDataPublishedByFilter. Update the
     * panel where all the filters are listed from the data published on the
     * Status bus by a particular filter.
     *
     * @param s
     */
    private void updateFilterListPanel(String filterName, String socketName) {
        filterListPanel.updateFilterListPanel(filterName, socketName);
    }

    /**
     * Update the Carousel Socket Panels from the data published on the Status
     * bus by a clamp.
     *
     * @param s
     */
    private void updateSocketPanels(String clampName, StatusDataPublishedByCarouselClamp s) {
        /**
         * Carousel Sockets Panel *
         */
        /*To find the name of the socket needed to find the socket panel to be updated.*/
        char num = clampName.charAt(clampName.length() - 1);
        String socketName = "socket" + num;

        socketPanels.get(socketName).update(clampName, s);
    }
    
    /**
     * Update the Carousel Socket Panels from the data published on the Status
     * bus by carousel.
     *
     * @param s
     */
    private void updateSocketPanels(StatusDataPublishedByCarousel s) {
        /**
         * Carousel Sockets Panel *
         */
        socketPanels.entrySet().stream().forEach((entry) -> {
            entry.getValue().updateSocketFromCarousel(s);
        });
    }    

    /**
     * Update the Carousel Socket Panels from the data published on the Status
     * bus by a Carousel Socket.
     *
     * @param s
     */
    private void updateSocketPanels(String socketName, StatusDataPublishedByCarouselSocket s) {
        socketPanels.get(socketName).updateSocket(s);
    }

    private void initPanels() {
        setName("CarouselGatherPanel");

        //controllers Panels
        clampXminusControllerPanel = new EPOSControllerPanel("clampXminusController");
        clampXplusControllerPanel = new EPOSControllerPanel("clampXplusController");
        carouselControllerPanel = new EPOSControllerPanel("carouselController");

        //Clamps Sensors Reader Device
        hyTTCPanel = new HYTTC580Panel("hyttc580");

        //CANOpen devices to read brakes status and temperatures
        cancbxai814Panel = new CanOpenDevicePanel("ai814");
        cancbxpt100Panel = new CanOpenDevicePanel("pt100");

        //Sockets Panels        
        for (String socketName : socketNames) {
            socketPanels.put(socketName, new CarouselSocketPanel(socketName));
        }

        /**
         * Filter List Panel
         */
        filterListPanel = new FilterListPanel();
    }
    
    
    /**
     * Reset Panel with initial text, color, reset hardware table and filter
     * table.
     */
    @Override
    public void resetPanel() {
        super.resetPanel();
        filterListPanel.resetPanel();
        socketPanels.entrySet().stream().forEach((entry) -> {
            entry.getValue().resetPanel();
        });
        caGeneralViewPanel.resetPanel();
        clampXminusControllerPanel.resetPanel();
        clampXplusControllerPanel.resetPanel();
        carouselControllerPanel.resetPanel();
        hyTTCPanel.resetPanel();
    }

}
