package org.lsst.ccs.subsystems.fcs.autochanger.ui;

import javax.swing.JFrame;
import javax.swing.JScrollPane;
import org.lsst.ccs.Agent;
import org.lsst.ccs.bus.data.AgentInfo;
import org.lsst.ccs.messaging.AgentMessagingLayer;
import static org.lsst.ccs.subsystems.fcs.FCSCst.AC_CLAMPSNAME;
import static org.lsst.ccs.subsystems.fcs.FCSCst.AC_LATCHES_NAME;
import static org.lsst.ccs.subsystems.fcs.FCSCst.AC_TRUCKS_NAME;
import static org.lsst.ccs.subsystems.fcs.FCSCst.CHANGER_TCPPROXY_NAME;
import static org.lsst.ccs.subsystems.fcs.FCSCst.FCSLOG;

import org.lsst.ccs.subsystems.fcs.ui.commons.GeneralGUI;
import org.lsst.ccs.subsystems.fcs.ui.commons.GeneralPanel;

/**
 *
 * @author virieux
 */
public class AutochangerGUI extends GeneralGUI implements InterfaceAutochangerGUI {
    private static final long serialVersionUID = 1528028505886594591L;

    private final long updateTrucksPositionTimeout;
    private long timeoutForTrucksMotion;

    private final long timeoutForClosingClamps;
    private final long timeoutForOpeningClamps;
    private long timeoutForOnlineClampMotion;

    private final long timeoutForClosingLatches;

    public AutochangerGUI(String name, String role, AgentMessagingLayer aml) {
        super(name, role, aml);
        AutochangerGatherPanel autochangerGatherPanel;
        autochangerGatherPanel = new AutochangerGatherPanel(this, CHANGER_TCPPROXY_NAME);
        AutochangerBottomPanel autochangerBottomPanel = new AutochangerBottomPanel(autochangerGatherPanel,
                "AUTOCHANGER");
        this.generalPanel = new GeneralPanel(this, autochangerBottomPanel);
        pane = new JScrollPane(generalPanel);

        // TODO to be initialized from the configuration system
        this.updateTrucksPositionTimeout = 1000;
        this.timeoutForTrucksMotion = 60000;
        this.timeoutForClosingClamps = 10000;
        this.timeoutForOpeningClamps = 10000;
        this.timeoutForClosingLatches = 10000;
    }

    @Override
    public void setTimeoutForTrucksMotion(long timeoutForTrucksMotion) {
        this.timeoutForTrucksMotion = timeoutForTrucksMotion;
    }

    @Override
    public void setTimeoutForOnlineClampMotion(long timeoutForOnlineClampMotion) {
        this.timeoutForOnlineClampMotion = timeoutForOnlineClampMotion;
    }

    /**
     ******************************************************************************
     ** Trucks Actions
     * *****************************************************************************
     */
    @Override
    public void acUpdateTrucksPosition() {
        FCSLOG.info("Autochanger updating Trucks position");
        FCSLOG.info("/timeout=" + updateTrucksPositionTimeout);
        new CommandSwingWorker("updatePosition", this.updateTrucksPositionTimeout, AC_TRUCKS_NAME).execute();
    }

    @Override
    public void acGoToHandOff() {
        FCSLOG.info("Autochanger Trucks going to Handoff Position");
        FCSLOG.info("/timeout=" + timeoutForTrucksMotion);
        new CommandSwingWorker("goToHandOff", this.timeoutForTrucksMotion, AC_TRUCKS_NAME).execute();
    }

    @Override
    public void acGoToStandby() {
        FCSLOG.info("Autochanger Trucks going to Standby Position");
        FCSLOG.info("/timeout=" + timeoutForTrucksMotion);
        new CommandSwingWorker("goToStandby", this.timeoutForTrucksMotion, AC_TRUCKS_NAME).execute();
    }

    @Override
    public void acGoToOnline() {
        FCSLOG.info("Autochanger Trucks going to Online Position");
        FCSLOG.info("/timeout=" + timeoutForTrucksMotion);
        new CommandSwingWorker("goToOnline", this.timeoutForTrucksMotion, AC_TRUCKS_NAME).execute();
    }

    @Override
    public void acMoveAndClampFilterOnline() {
        FCSLOG.info("Autochanger Trucks moving to Online Position, and clamping");
        FCSLOG.info("/timeout=" + timeoutForTrucksMotion);
        new CommandSwingWorker("moveAndClampFilterOnline", this.timeoutForTrucksMotion, AC_TRUCKS_NAME).execute();
    }

    /**
     ******************************************************************************
     ** end of Trucks Actions
     * *****************************************************************************
     */

    /**
     ******************************************************************************
     ** Clamps Actions
     * *****************************************************************************
     */
    @Override
    public void unlockOnlineClamps() {
        FCSLOG.info("Autochanger unlocking online clamps");
        new CommandSwingWorker("unlockClamps", timeoutForOnlineClampMotion, AC_CLAMPSNAME).execute();
    }

    @Override
    public void lockOnlineClamps() {
        FCSLOG.info("Autochanger locking online clamps");
        new CommandSwingWorker("lockClamps", timeoutForOnlineClampMotion, AC_CLAMPSNAME).execute();
    }

    @Override
    public void closeOnlineClamps() {
        FCSLOG.info("Autochanger closing online clamps");
        new CommandSwingWorker("closeClamps", timeoutForOnlineClampMotion, AC_CLAMPSNAME).execute();
    }

    @Override
    public void openOnlineClamps() {
        FCSLOG.info("Autochanger opening online clamps");
        new CommandSwingWorker("openClamps", timeoutForOnlineClampMotion, AC_CLAMPSNAME).execute();
    }

    @Override
    public void unlockAndOpenOnlineClamps() {
        FCSLOG.info("Autochanger unlocking and opening online clamps");
        new CommandSwingWorker("unlockAndOpenClamps", timeoutForOnlineClampMotion, AC_CLAMPSNAME).execute();
    }

    @Override
    public void closeAndLockOnlineClamps() {
        FCSLOG.info("Autochanger closing and locking online clamps");
        new CommandSwingWorker("closeAndLockClamps", timeoutForOnlineClampMotion, AC_CLAMPSNAME).execute();
    }

    @Override
    public void homingOnlineClamp(String clampName) {
        FCSLOG.info("Autochanger homing online clamp: " + clampName);
        new CommandSwingWorker("openInCurrentModeAndHoming", timeoutForOnlineClampMotion, clampName).execute();
    }

    @Override
    public void openOnlineClamp(String clampName) {
        FCSLOG.info("Autochanger opening clamp: " + clampName);
        new CommandSwingWorker("open", this.timeoutForOpeningClamps, clampName).execute();
    }

    @Override
    public void closeOnlineClamp(String clampName) {
        FCSLOG.info("Autochanger closing clamp: " + clampName);
        new CommandSwingWorker("close", this.timeoutForClosingClamps, clampName).execute();
    }

    @Override
    public void lockOnlineClamp(String clampName) {
        FCSLOG.info("Autochanger clamping clamp: " + clampName);
        new CommandSwingWorker("lock", this.timeoutForClosingClamps, clampName).execute();
    }

    @Override
    public void unlockOnlineClamp(String clampName) {
        FCSLOG.info("Autochanger unclamping clamp: " + clampName);
        new CommandSwingWorker("unlock", this.timeoutForClosingClamps, clampName).execute();
    }

    /**
     ******************************************************************************
     ** end of Clamps Actions
     * *****************************************************************************
     */
    /**
     ******************************************************************************
     ** Latches Actions
     * *****************************************************************************
     *
     * @param latchName
     */
    @Override
    public void closeLatch(String latchName) {
        FCSLOG.info("Autochanger locking latch: " + latchName);
        FCSLOG.info("/timeout=" + timeoutForClosingLatches);
        new CommandSwingWorker("close", this.timeoutForClosingLatches, latchName).execute();
    }

    @Override
    public void openLatch(String latchName) {
        FCSLOG.info("Autochanger unlocking latch: " + latchName);
        FCSLOG.info("/timeout=" + timeoutForClosingLatches);
        new CommandSwingWorker("open", this.timeoutForClosingLatches, latchName).execute();
    }

    @Override
    public void closeLatches() {
        FCSLOG.info("Autochanger locking latches.");
        FCSLOG.info("/timeout=" + timeoutForClosingLatches);
        new CommandSwingWorker("close", this.timeoutForClosingLatches, AC_LATCHES_NAME).execute();
    }

    @Override
    public void openLatches() {
        FCSLOG.info("Autochanger unlocking latches.");
        FCSLOG.info("/timeout=" + timeoutForClosingLatches);
        new CommandSwingWorker("open", this.timeoutForClosingLatches, AC_LATCHES_NAME).execute();
    }

    /**
     ******************************************************************************
     ** end of Latches Actions
     * *****************************************************************************
     */
    /**
     * @param args the command line arguments
     */
    public static void main(String[] args) {

        Agent agent = new Agent("AutochangerGUI", AgentInfo.AgentType.CONSOLE);
        AutochangerGUI t = new AutochangerGUI(agent.getName(), "Autochanger", agent.getMessagingAccess());
        agent.startAgent();

        JFrame frame = new JFrame("Autochanger View GUI");
        frame.setContentPane(t.getGuiLayout());
        frame.pack();
        frame.setVisible(true);
        t.initGui();
        frame.setDefaultCloseOperation(JFrame.EXIT_ON_CLOSE);
    }

}
