
package org.lsst.ccs.subsystems.fcs.ui;

import java.awt.Color;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import javax.swing.BorderFactory;
import javax.swing.JFrame;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.SwingUtilities;
import org.lsst.ccs.bus.data.KeyValueData;
import static org.lsst.ccs.subsystems.fcs.FCSCst.AT_STANDBY;
import static org.lsst.ccs.subsystems.fcs.FCSCst.FCSLOG;
import static org.lsst.ccs.subsystems.fcs.FCSCst.NOT_AT_STANDBY;
import static org.lsst.ccs.subsystems.fcs.FCSCst.NO_FILTER;
import org.lsst.ccs.subsystems.fcs.FcsEnumerations;
import org.lsst.ccs.subsystems.fcs.StatusDataPublishedByAutoChanger;
import org.lsst.ccs.subsystems.fcs.StatusDataPublishedByAutoChangerTrucks;
import org.lsst.ccs.subsystems.fcs.StatusDataPublishedByAutochangerThreeClamps;
import org.lsst.ccs.subsystems.fcs.StatusDataPublishedByAutochangerTwoLatches;
import org.lsst.ccs.subsystems.fcs.StatusDataPublishedByCarousel;
import org.lsst.ccs.subsystems.fcs.StatusDataPublishedByCarouselSocket;
import org.lsst.ccs.subsystems.fcs.ui.commons.DigitalSwitch;
import static org.lsst.ccs.subsystems.fcs.ui.commons.Tools.BIG_FONT;
import static org.lsst.ccs.subsystems.fcs.ui.commons.Tools.UNKNOWN_STATE;
import static org.lsst.ccs.subsystems.fcs.ui.commons.Tools.ZERO_VALUE;
import static org.lsst.ccs.subsystems.fcs.ui.commons.Tools.greenColor;
import static org.lsst.ccs.subsystems.fcs.ui.commons.Tools.insets_std;

/**
 *
 * @author virieux
 */
public class FcsGeneralViewPanel extends JPanel {
    private static final long serialVersionUID = 1L;
    private static final String NONE = "NONE";
    
    private final String title = "FCS OVERVIEW";

    
    private JPanel autochangerPanel;
    private JPanel carouselPanel;
    private JPanel loaderPanel;
    
    /*autochanger information*/
    private final JLabel filterONLINELabel = new JLabel(NONE);
    private final JLabel filterOnTrucksLabel = new JLabel(NONE);
    private final JLabel onlineClampsStateLabel = new JLabel("UNKNOWN");
    private final JLabel latchesStateLabel = new JLabel("UNKNOWN");
    private final JLabel trucksPositionLabel = new JLabel("UNKNOWN");
    private final JLabel trucksPositionNameLabel = new JLabel("UNKNOWN");

    /*carousel information*/
    private final JLabel homingLabel = new JLabel();
    private final JLabel actualPositionLabel = new JLabel();
    private final JLabel actualSpeedLabel = new JLabel();
    private final JLabel actualCurrentLabel = new JLabel();
    private final JLabel standbyLabel = new JLabel();
    private final DigitalSwitch standbySwitch = new DigitalSwitch();
    private final JLabel socketAtStandbyLabel = new JLabel();
    private final JLabel slaveModuleStatusLabel = new JLabel();
    private final JLabel filterAtStandbyLabel = new JLabel();
    private final JLabel caClampsStateLabel = new JLabel();

    /* status information */
    private final FcsCameraProtectionPanel cameraStatusPanel = new FcsCameraProtectionPanel();

    public FcsGeneralViewPanel() {
        initComponents();
    }

    void updateFromStatusData(KeyValueData data) {
        if (data.getValue() instanceof StatusDataPublishedByAutochangerThreeClamps) {
            updateClamps((StatusDataPublishedByAutochangerThreeClamps) data.getValue());
        } else if (data.getValue() instanceof StatusDataPublishedByAutochangerTwoLatches) {
            updateLatches((StatusDataPublishedByAutochangerTwoLatches) data.getValue());
        } else if (data.getValue() instanceof StatusDataPublishedByAutoChangerTrucks) {
            updateTrucks((StatusDataPublishedByAutoChangerTrucks) data.getValue());
        } else if (data.getValue() instanceof StatusDataPublishedByAutoChanger) {
            cameraStatusPanel.updateAutochanger((StatusDataPublishedByAutoChanger) data.getValue());
        } else if (data.getValue() instanceof StatusDataPublishedByCarousel) {
            updateCarousel((StatusDataPublishedByCarousel) data.getValue());
        } else if (data.getValue() instanceof StatusDataPublishedByCarouselSocket) {
            updateCarouselSocket((StatusDataPublishedByCarouselSocket) data.getValue());
        }
    }

    void updateCarouselSocket(StatusDataPublishedByCarouselSocket s) {
        SwingUtilities.invokeLater(new FcsGeneralViewPanel.UpdateCarouselSocket(s));
    }

    void updateCarousel(StatusDataPublishedByCarousel s) {
        FCSLOG.debug(getName() + " updating from carousel data:");
        SwingUtilities.invokeLater(new FcsGeneralViewPanel.UpdateCarousel(s));
        cameraStatusPanel.updateCarousel(s);
    }

    void updateTrucks(StatusDataPublishedByAutoChangerTrucks s) {
        SwingUtilities.invokeLater(new FcsGeneralViewPanel.UpdateTrucks(s));
    }

    void updateLatches(StatusDataPublishedByAutochangerTwoLatches s) {
        SwingUtilities.invokeLater(new FcsGeneralViewPanel.UpdateLatches(s));
    }

    void updateClamps(StatusDataPublishedByAutochangerThreeClamps s) {
        SwingUtilities.invokeLater(new FcsGeneralViewPanel.UpdateOnlineClamps(s));
    }

    /**
     * A Runnable class to update carousel panel from an object
     * StatusDataPublishedByCarousel.
     */
    public class UpdateCarousel implements Runnable {

        private final StatusDataPublishedByCarousel s;

        public UpdateCarousel(StatusDataPublishedByCarousel s) {
            this.s = s;
        }

        @Override
        public void run() {
            /*Update of actual parameters Panel*/
            if (s.isHomingDone()) {
                homingLabel.setText("DONE");
                homingLabel.setForeground(greenColor);
            } else {
                homingLabel.setText("TO_BE_DONE");
                homingLabel.setForeground(Color.RED);
            }
            actualPositionLabel.setText(Integer.toString(s.getPosition()));
            actualSpeedLabel.setText(Double.toString(s.getVelocity()));
            actualCurrentLabel.setText(Double.toString(s.getCurrent()));
            if (s.isAtStandby()) {
                standbyLabel.setText(AT_STANDBY);
                standbySwitch.setColor(greenColor);
                socketAtStandbyLabel.setText(s.getSocketAtStandbyName());
                socketAtStandbyLabel.setFont(BIG_FONT);
                filterAtStandbyLabel.setText(s.getFilterAtStandbyName());
                filterAtStandbyLabel.setFont(BIG_FONT);

            } else {
                standbyLabel.setText(NOT_AT_STANDBY);
                standbySwitch.setColor(Color.orange);
                socketAtStandbyLabel.setText(s.getSocketAtStandbyName());
                filterAtStandbyLabel.setText(NONE);
            }
        }
    }

    /**
     * A Runnable to updateSocket this panel from a
     * StatusDataPublishedByCarouselSocket.
     */
    public class UpdateCarouselSocket implements Runnable {

        private StatusDataPublishedByCarouselSocket s;

        public UpdateCarouselSocket(StatusDataPublishedByCarouselSocket s) {
            this.s = s;
        }

        @Override
        public void run() {
            if (s.isAtStandby()) {
                caClampsStateLabel.setText(s.getClampState().toString());
                caClampsStateLabel.setForeground(s.getClampState().getColor());
                slaveModuleStatusLabel.setText(s.getSlaveStatus().toString());
                slaveModuleStatusLabel.setForeground(s.getSlaveStatus().getColor());
            }
        }
    }

    /**
     * A Runnable to initialize trucks data from status data published on the
     * STATUS bus by autochanger trucks.
     */
    private class UpdateTrucks implements Runnable {

        private StatusDataPublishedByAutoChangerTrucks s;

        public UpdateTrucks(StatusDataPublishedByAutoChangerTrucks status) {
            this.s = status;
        }

        @Override
        public void run() {
            trucksPositionLabel.setText(String.valueOf(s.getMasterPosition()));
            trucksPositionNameLabel.setForeground(greenColor);
            if (s.isAtHandoff()) {
                trucksPositionNameLabel.setText("HANDOFF");
            } else if (s.isAtOnline()) {
                trucksPositionNameLabel.setText("ONLINE");
            } else if (s.isAtStandby()) {
                trucksPositionNameLabel.setText("STANDBY");
            } else {
                trucksPositionNameLabel.setText("IN_TRAVEL");
                trucksPositionNameLabel.setForeground(Color.YELLOW);
            }

//            handoffDigitalSwitch.setColor(s.isAtHandoff() ? greenColor : Color.gray);
//
//            onlineDigitalSwitch.setColor(s.isAtOnline() ? greenColor : Color.gray);
//
//            standbyDigitalSwitch.setColor(s.isAtStandby() ? greenColor : Color.gray);
//
//            //ERROR Digital Switch is red as soon as there is an error
//            //on the carrier position sensors.
//            inErrorDigitalSwitch.setColor(s.isInError() ? Color.RED : greenColor);
//            inErrorjLabel.setText(s.isInError() ? "IN ERROR" : "NO ERROR");
//
//            inTravelDigitalSwitch.setColor((!s.isAtHandoff()
//                    && !s.isAtOnline() && !s.isAtStandby()) ? Color.YELLOW : Color.gray);
        }

    }

    /**
     * A Runnable to update latches data from status data published on the
     * STATUS bus by autochanger latches.
     */
    class UpdateLatches implements Runnable {

        private StatusDataPublishedByAutochangerTwoLatches s;

        public UpdateLatches(StatusDataPublishedByAutochangerTwoLatches status) {
            this.s = status;
        }

        @Override
        public void run() {
            latchesStateLabel.setText(s.getLockStatus().toString());
            latchesStateLabel.setForeground(s.getLockStatus().getColor());
            if (s.getFilterPresenceStatus().equals(FcsEnumerations.FilterPresenceStatus.ENGAGED)) {
                filterOnTrucksLabel.setText(s.getFilterName());
            }
        }
    }

    /**
     * A Runnable to update ONLINE clamps data from status data published on the
     * STATUS bus by autochanger ONLINE clamps.
     */
    class UpdateOnlineClamps implements Runnable {

        private StatusDataPublishedByAutochangerThreeClamps s;

        public UpdateOnlineClamps(StatusDataPublishedByAutochangerThreeClamps status) {
            this.s = status;
        }

        @Override
        public void run() {
            onlineClampsStateLabel.setText(s.getLockStatus().toString());
            onlineClampsStateLabel.setForeground(s.getLockStatus().getColor());
        }
    }
    
    

    // <editor-fold defaultstate="collapsed" desc="Form Description">                          
    private void initComponents() {
        
        /**
         ******************************************************************************
         ** Autochanger Panel
         * *****************************************************************************
         */
        autochangerPanel = new JPanel();
        autochangerPanel.setBorder(BorderFactory.createTitledBorder("autochanger"));
        autochangerPanel.setLayout(new GridBagLayout());
        GridBagConstraints gbc1 = new GridBagConstraints();
        gbc1.insets = insets_std;
        //first colomn
        gbc1.gridx = 0;
        gbc1.gridy = 0;
        gbc1.anchor = GridBagConstraints.LINE_START;
        autochangerPanel.add(new JLabel("Filter ONLINE: "),gbc1);
        gbc1.gridy++;
        autochangerPanel.add(new JLabel("Filter on trucks: "), gbc1);
        gbc1.gridy++;
        autochangerPanel.add(new JLabel("ONLINE clamps state: "),gbc1); 
        gbc1.gridy++;
        autochangerPanel.add(new JLabel("Latches state: "),gbc1);        
        gbc1.gridy++;
        autochangerPanel.add(new JLabel("Trucks Position: "),gbc1); 
        //second colomn
        gbc1.gridx++;
        gbc1.gridy = 0;
        gbc1.anchor = GridBagConstraints.LINE_END;
        autochangerPanel.add(filterONLINELabel, gbc1);
        gbc1.gridy++;
        autochangerPanel.add(filterOnTrucksLabel, gbc1);
        gbc1.gridy++;
        autochangerPanel.add(onlineClampsStateLabel, gbc1);
        gbc1.gridy++;
        autochangerPanel.add(latchesStateLabel, gbc1);        
        gbc1.gridy++;
        autochangerPanel.add(trucksPositionNameLabel, gbc1);
        gbc1.gridx++;
        autochangerPanel.add(trucksPositionLabel, gbc1);
         /**
         ******************************************************************************
         ** Carousel Panel
         * *****************************************************************************
         */
        carouselPanel = new JPanel();
        carouselPanel.setBorder(BorderFactory.createLineBorder(Color.yellow));
        carouselPanel.setBorder(BorderFactory.createTitledBorder("carousel"));
        carouselPanel.setLayout(new GridBagLayout());
        gbc1 = new GridBagConstraints();
        gbc1.insets = insets_std;
        //first colomn
        gbc1.gridx = 0;
        gbc1.gridy = 0;
        gbc1.anchor = GridBagConstraints.LINE_START;
        carouselPanel.add(new JLabel("Homing: "), gbc1);
        gbc1.gridy++;
        carouselPanel.add(new JLabel("Position: "), gbc1);
        gbc1.gridy++;
        carouselPanel.add(new JLabel("Speed: "), gbc1);
        gbc1.gridy++;
        carouselPanel.add(new JLabel("Current: "), gbc1);
        gbc1.gridy++;
        carouselPanel.add(standbyLabel, gbc1);
        gbc1.gridy++;
        carouselPanel.add(new JLabel("Socket at STANDBY: "), gbc1);
        gbc1.gridy++;
        carouselPanel.add(new JLabel("Filter at STANDBY: "), gbc1);
        gbc1.gridy++;
        carouselPanel.add(new JLabel("Slave Status: "), gbc1);
        gbc1.gridy++;
        carouselPanel.add(new JLabel("Clamps Status: "), gbc1);
        //second colomn
        gbc1.gridx++;
        gbc1.gridy = 0;
        gbc1.anchor = GridBagConstraints.LINE_END;
        carouselPanel.add(homingLabel, gbc1);
        gbc1.gridy++;
        carouselPanel.add(actualPositionLabel, gbc1);
        gbc1.gridy++;
        carouselPanel.add(actualSpeedLabel, gbc1);
        gbc1.gridy++;
        carouselPanel.add(actualCurrentLabel, gbc1);
        gbc1.gridy++;
        standbySwitch.setColor(Color.gray);
        carouselPanel.add(standbySwitch, gbc1);
        gbc1.gridy++;
        carouselPanel.add(socketAtStandbyLabel, gbc1);
        gbc1.gridy++;
        carouselPanel.add(filterAtStandbyLabel, gbc1);
        gbc1.gridy++;
        carouselPanel.add(slaveModuleStatusLabel, gbc1);
        gbc1.gridy++;
        carouselPanel.add(caClampsStateLabel, gbc1);
         /**
         ******************************************************************************
         ** Loader Panel
         * *****************************************************************************
         */
        loaderPanel = new JPanel();
        loaderPanel.setBorder(BorderFactory.createTitledBorder("loader"));         
        
        /**
         ******************************************************************************
         ** Whole Panel
         * *****************************************************************************
         */
        /*General Information Panel*/
        setBorder(BorderFactory.createTitledBorder(title));
        setLayout(new java.awt.GridBagLayout());
        GridBagConstraints gbc = new GridBagConstraints();
        gbc.insets = insets_std;
        gbc.anchor = GridBagConstraints.BASELINE_LEADING;

        gbc.gridx = 0;
        gbc.gridy = 0;
        gbc.gridwidth = 3;
        add(cameraStatusPanel, gbc);

        gbc.gridy++;
        gbc.gridwidth = 1;
        add(carouselPanel, gbc);
        
        gbc.gridx++;
        add(autochangerPanel, gbc);
        
        gbc.gridx++;
        add(loaderPanel, gbc);
        /**
         ******************************************************************************
         ** end of Whole Panel
         * *****************************************************************************
         */
        setDefaultValues();
    }

    /**
     * set default values for components color and labels.
     */
    private void setDefaultValues() {
        /*carousel*/
        homingLabel.setText("NOT DONE");
        homingLabel.setForeground(Color.RED);
        actualPositionLabel.setText(ZERO_VALUE);
        actualSpeedLabel.setText(ZERO_VALUE);
        actualCurrentLabel.setText(ZERO_VALUE);
        standbyLabel.setText(NOT_AT_STANDBY);
        standbySwitch.setColor(Color.gray);
        socketAtStandbyLabel.setText(NONE);
        filterAtStandbyLabel.setText(NO_FILTER);
        caClampsStateLabel.setText(UNKNOWN_STATE);
        slaveModuleStatusLabel.setText(UNKNOWN_STATE);
        /*autochanger*/
        trucksPositionLabel.setText(ZERO_VALUE);
        trucksPositionNameLabel.setText(UNKNOWN_STATE);
        onlineClampsStateLabel.setText(UNKNOWN_STATE);
        latchesStateLabel.setText(UNKNOWN_STATE);
    }

    /**
     * Reset Panels to default values.
     */
    public void resetPanel() {
        setDefaultValues();
    }
    
    @Override
    public String toString() {
        return title;
    }

    /**
     * This main launches an GUI for the whole subsystem "fcs". It's used only
     * for tests.
     *
     * @param args the command line arguments
     */
    public static void main(String[] args) {

        FcsGeneralViewPanel d = new FcsGeneralViewPanel();
        JFrame frame = new JFrame("FCS General View Panel");
        frame.setContentPane(d);
        frame.pack();
        frame.setVisible(true);
        frame.setDefaultCloseOperation(JFrame.EXIT_ON_CLOSE);
    }
    
}
