
package org.lsst.ccs.subsystems.fcs.carousel.ui;

import java.awt.Color;
import java.awt.Font;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.util.Map;
import javax.swing.BorderFactory;
import javax.swing.JFrame;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.SwingConstants;
import javax.swing.SwingUtilities;
import org.lsst.ccs.bus.data.ConfigurationInfo;
import org.lsst.ccs.subsystems.fcs.StatusDataPublishedByCarouselClamp;
import static org.lsst.ccs.subsystems.fcs.ui.commons.Tools.BIG_FONT;
import static org.lsst.ccs.subsystems.fcs.ui.commons.Tools.UNKNOWN_STATE;
import static org.lsst.ccs.subsystems.fcs.ui.commons.Tools.ZERO_VALUE;
import static org.lsst.ccs.subsystems.fcs.ui.commons.Tools.insets_std;
import static org.lsst.ccs.subsystems.fcs.ui.commons.Tools.nicePink;

/**
 * A Panel to display data coming from a carousel clamp.
 * @author virieux
 */
public class CarouselClampPanel extends JPanel {
    private static final long serialVersionUID = -1273081770223753869L;
    private String clampName;
    private String lockSensorName;
    private String filterPresenceSensorName;
    private JLabel clampStateLabel;
    
    private final JLabel explainText = new JLabel("<HTML>LOCKED: valueB < v < maxValue "
             + "<P> UNLOCKED: minValue < v < valueA </HTML>");
    
    /* lockSensorPanel*/
    private AnalogicSensorPanel lockSensorPanel;
    
    /* filterPresenceSensorPanel*/
    private AnalogicSensorPanel filterPresenceSensorPanel;
    
    /* general parameters*/
    private JPanel paramsPanel;
    private final JLabel currentToUnlockLabel = new JLabel("0");
    private final JLabel currentToMaintainUnlockLabel = new JLabel("0");


    public CarouselClampPanel(String clampName) {
        this.clampName = clampName;
        
        /* to find the name of the sensors of this clamp*/
        String[] words = clampName.split("X");
        lockSensorName = "lockSensor" + "X" + words[1];
        filterPresenceSensorName = "filterPresence" + "X" + words[1];
        initComponents();
    }
    
    /**
     * Initialize the GUI with the configuration data.
     * 
     * @param configInfo
     */
    public void initializeGui(ConfigurationInfo configInfo) {
        Map<String, String> config = configInfo.getCurrentValuesFor(clampName); 
        currentToUnlockLabel.setText(config.get("currentToUnlock"));
        currentToMaintainUnlockLabel.setText(config.get("currentToMaintainUnlocked"));
        
        lockSensorPanel.initializeGui(configInfo);
        filterPresenceSensorPanel.initializeGui(configInfo);
        
    }
    
    /**
     * Update from status data published by a Clamp on Status bus.
     * @param s 
     */
    protected void updateClamp(StatusDataPublishedByCarouselClamp s) {
        SwingUtilities.invokeLater(new UpdateClamp(s));
    }
    
    /**
     * A Runnable class to update Panel from a StatusDataPublishedByCarouselClamp.
     */
    public class UpdateClamp implements Runnable {
        private StatusDataPublishedByCarouselClamp s;

        public UpdateClamp(StatusDataPublishedByCarouselClamp s) {
            this.s = s;
        }
        
        
        @Override
        public void run() {
            clampStateLabel.setText(s.getClampState().toString());
            clampStateLabel.setForeground(s.getClampState().getColor());
            lockSensorPanel.update(s.getLockSensorValue(), s.getLockStatus());
            filterPresenceSensorPanel.update(s.getFilterPositionSensorValue(), s.getFilterPresenceStatus());
        }
    
    }

    
    // <editor-fold defaultstate="collapsed" desc="Form description">                          
    private void initComponents() {
        /*Clamp State*/
        clampStateLabel = new JLabel(UNKNOWN_STATE);
        clampStateLabel.setFont(BIG_FONT);

        /**
         ******************************************************************************
         ** Sensors Panels
         * *****************************************************************************
         */
        lockSensorPanel = new AnalogicSensorPanel(clampName,lockSensorName, SwingConstants.HORIZONTAL);
        filterPresenceSensorPanel = new AnalogicSensorPanel(clampName, filterPresenceSensorName, 
                SwingConstants.VERTICAL);
        
        /**
         ******************************************************************************
         ** end Sensors Panel
         * *****************************************************************************
         */  
        
        /**
         ******************************************************************************
         ** General Parameters Panels
         * *****************************************************************************
         */
        paramsPanel = new JPanel();
        paramsPanel.setBorder(BorderFactory.createLineBorder(Color.blue));
        paramsPanel.setLayout(new GridBagLayout());
        GridBagConstraints gbcp = new GridBagConstraints();
        gbcp.insets = insets_std;
        gbcp.gridx = 0;
        gbcp.gridy = 0;
        /*first columm*/
        gbcp.anchor = GridBagConstraints.LINE_START;
        paramsPanel.add(new JLabel("Current to lock:"),gbcp);
        gbcp.gridy++;
        paramsPanel.add(new JLabel("<HTML>Current to <P>maintain unlock:</HTML>"),gbcp);
        gbcp.gridx++;
        /*second columm*/
        gbcp.anchor = GridBagConstraints.LINE_END;
        gbcp.gridx = 1;
        gbcp.gridy = 0;
        paramsPanel.add(currentToUnlockLabel, gbcp);
        gbcp.gridy++;
        paramsPanel.add(currentToMaintainUnlockLabel, gbcp);
        /**
         ******************************************************************************
         ** end of General Parameters Panels
         * *****************************************************************************
         */
        

        /**
        /**
         * 
         ******************************************************************************
         ** Whole Panel
         * *****************************************************************************
         */
        setBorder(BorderFactory.createTitledBorder(BorderFactory.createLineBorder(nicePink),clampName));
        setLayout(new GridBagLayout());
        GridBagConstraints gbc = new GridBagConstraints();
        gbc.insets = insets_std;
        gbc.gridx = 0;
        gbc.gridy = 0;
        /*clamp state*/
        add(clampStateLabel,gbc);
        gbc.gridx++;
        /*parameters Panel*/
        add(paramsPanel, gbc);
        
        gbc.gridx = 0;
        gbc.gridy++;
        add(explainText,gbc);
        gbc.gridy++;
        add(lockSensorPanel, gbc);
        
        gbc.gridx++;
        add(filterPresenceSensorPanel, gbc);
        /**
         ******************************************************************************
         ** end Whole Panel
         * *****************************************************************************
         */
        
    }
    
    /**
     * Reset panel to default values.
     */
    public void resetPanel() {
        currentToUnlockLabel.setText(ZERO_VALUE);
        currentToMaintainUnlockLabel.setText(ZERO_VALUE);
        clampStateLabel.setText(UNKNOWN_STATE);
        lockSensorPanel.resetPanel();
        filterPresenceSensorPanel.resetPanel();
    }
    
    /**
     * Just to test the Panel.
     * @param argv 
     */
    public static void main(String[] argv) {

        CarouselClampPanel d = new CarouselClampPanel("clampXminus");
        JFrame frame = new JFrame("Test");
        frame.setContentPane(d);
        frame.pack();
        frame.setVisible(true);
        frame.setDefaultCloseOperation(JFrame.EXIT_ON_CLOSE);

    }
}
