package org.lsst.ccs.subsystems.fcs.ui.commons;

import java.awt.Color;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import javax.swing.BorderFactory;
import javax.swing.JFrame;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.JTable;
import javax.swing.SwingUtilities;
import javax.swing.table.DefaultTableModel;
import javax.swing.table.TableColumn;
import org.lsst.ccs.bus.data.ConfigurationInfo;
import org.lsst.ccs.subsystems.fcs.FCSCst;
import org.lsst.ccs.subsystems.fcs.StatusDataPublishedByHardware;
import static org.lsst.ccs.subsystems.fcs.ui.commons.Tools.ZERO_VALUE;
import static org.lsst.ccs.subsystems.fcs.ui.commons.Tools.insets_std;
import org.lsst.ccs.utilities.logging.Logger;

/**
 * This Panel displays in a table the list of CANopen devices the subsystem manages.
 * It displays also the portnumber of the tcp connection of the tcpProxy and the 
 * numbers of devices on the CAN bus.
 * @author virieux
 */
public class CanOpenHardwareListPanel extends JPanel {
    
    private final String tcpProxyName;
    private List<String> hardwareNames;

    // Variables declaration 
    protected static final Logger FCSLOG = FCSCst.FCSLOG;
    private static final long serialVersionUID = 4473630289328745341L;

    private JLabel tcpProxyLabel;
    private JLabel bootedNodesNBLabel;
    private JLabel expectedNodesNBLabel;
    private JLabel hardwareBootTimeoutLabel;
    private JPanel hardwarePanel;
    private JTable hardwareTable;
    private JLabel portNumberLabel;
    private JScrollPane jScrollPane;
    private boolean resetted;
    // End of variables declaration 
    
    /**
     * Initialization of the GUI for data stored in tcpConfiguration. *
     */
    private class GuiInitializationWithConfigInfo implements Runnable {
        private final ConfigurationInfo configInfo; 
        private final List<String> hardwareList;

        public GuiInitializationWithConfigInfo(ConfigurationInfo configInfo, List<String> hardwareList) {
            this.configInfo = configInfo;
            /* copy of the hardwareList*/
            this.hardwareList = new ArrayList<>();
            this.hardwareList.addAll(hardwareList);
        }
        
        @Override
        public void run() {
            int nodesNB = hardwareList.size();
            Map<String, String> config = configInfo.getCurrentValuesFor(tcpProxyName);
            portNumberLabel.setText(config.get("portNumber"));
            expectedNodesNBLabel.setText(Integer.toString(nodesNB));
            hardwareBootTimeoutLabel.setText(config.get("hardwareBootTimeout"));

            //configure HardwareTable           
            //TODO : update bootedNodesNBLabel from status data.
            DefaultTableModel hardwareTableModel = (DefaultTableModel) hardwareTable.getModel();
            for (String hardwareName: hardwareList) {
                Map<String, String> hardwareConfig = configInfo.getCurrentValuesFor(hardwareName);
                String nodeID = Integer.toHexString(Integer.parseInt(hardwareConfig.get("nodeID")));
                String serialNB = hardwareConfig.get("serialNB");
                Object[] tab = {hardwareName, nodeID, serialNB,
                    false, false};
                hardwareTableModel.addRow(tab);
            }
            hardwareTableModel.fireTableDataChanged();
        }
        
    }
    


    /**
     * A Runnable to update the Panel when CANopen hardware publishes on the Status 
     * bus its status.
     */
    private class UpdateCanOpenHardwarePanel implements Runnable {

        private final StatusDataPublishedByHardware s;

        public UpdateCanOpenHardwarePanel(StatusDataPublishedByHardware status) {
            this.s = status;
        }

        @Override
        public void run() {
            DefaultTableModel hardwareTableModel = (DefaultTableModel) hardwareTable.getModel();
            int rowNB = hardwareTableModel.getRowCount();
            for (int i = 0; i < rowNB; i++) {
                if ((hardwareTableModel.getValueAt(i, 0)).equals(s.getName())) {
                    hardwareTableModel.setValueAt(s.isBooted(), i, 3);
                    hardwareTableModel.setValueAt(s.isInitialized(), i, 4);
                }
            }
            hardwareTableModel.fireTableDataChanged();
        }
    }
    
    /**
     * A Runnable to update panel from tcp proxy data.
     */
    private class UpdateTcpProxy implements Runnable {
        
        private final int nbBootedNodes;
        
        public UpdateTcpProxy(int nbBootedNodes) {
            this.nbBootedNodes = nbBootedNodes;
        }
        
        @Override
        public void run() {
            bootedNodesNBLabel.setText(Integer.toString(nbBootedNodes));

        }
    }
    
    /**
     * Creates new form CanOpenHardware with a tcpProxy name.
     * @param tcpProxyName
     */
    public CanOpenHardwareListPanel(String tcpProxyName
    ) {
        this.tcpProxyName = tcpProxyName;
        initComponents();
    }

    public void setHardwareNames(List<String> hardwareNamesList) {
        this.hardwareNames = new ArrayList<>();
        this.hardwareNames.addAll(hardwareNamesList);
    }

    /**
     * return a copy of hardware names list.
     * @return 
     */
    public List<String> getHardwareNames() {
        List<String> list = new ArrayList<>();
        list.addAll(hardwareNames);
        return list;
    }
    
    public void initializeGui(ConfigurationInfo configInfo, List<String> hardwareList) {
        SwingUtilities.invokeLater(new GuiInitializationWithConfigInfo(configInfo,hardwareList));
    }    

    /**
     * Reset the Panel with the initial text values.
     * Delete hardware table rows.
     */
    public void resetPanel()  {
        if (!resetted) {
            portNumberLabel.setText(ZERO_VALUE);
            expectedNodesNBLabel.setText(ZERO_VALUE);
            hardwareBootTimeoutLabel.setText(ZERO_VALUE);
            bootedNodesNBLabel.setText("0");
            DefaultTableModel hardwareTableModel = (DefaultTableModel) hardwareTable.getModel();
            int rowNB = hardwareTableModel.getRowCount();
            while (hardwareTableModel.getRowCount() > 0) {
                hardwareTableModel.removeRow(hardwareTableModel.getRowCount()-1);
            }
            hardwareTableModel.fireTableRowsDeleted(0, rowNB-1);
            resetted = true;
        }
    }


   

    public void updateCanOpenHardwarePanel(StatusDataPublishedByHardware status) {
        SwingUtilities.invokeLater(new UpdateCanOpenHardwarePanel(status));
    }
    
    public void updateCanOpenHardwarePanel(int nbBootedNodes) {
        SwingUtilities.invokeLater(new UpdateTcpProxy(nbBootedNodes));
    }    

    // <editor-fold defaultstate="collapsed" desc="Form description">                          
    private void initComponents() {

        hardwarePanel = new JPanel();
        this.resetted = false;
        
        tcpProxyLabel = new JLabel(this.tcpProxyName);
        bootedNodesNBLabel = new JLabel("0");
        hardwareBootTimeoutLabel = new JLabel("0");
        expectedNodesNBLabel = new JLabel("0");
        portNumberLabel = new JLabel("0");

        /**
         ******************************************************************************
         ** CANopen Hardware Panel
         * *****************************************************************************
         */
        jScrollPane = new JScrollPane();
        hardwareTable = new JTable();
        hardwareTable.setModel(new DefaultTableModel(
                new Object[][]{},
                new String[]{
                    "Name", "Node ID", "Serial NB", "Booted", "Initialized"
                }
        ) {
            private static final long serialVersionUID = 1L;
            private final Class[] types = new Class[]{
                String.class, String.class, Object.class, Boolean.class, Boolean.class
            };
            private final boolean[] canEdit = new boolean[]{
                false, false, false, false, false
            };

            @Override
            public Class getColumnClass(int columnIndex) {
                return types[columnIndex];
            }

            @Override
            public boolean isCellEditable(int rowIndex, int columnIndex) {
                return canEdit[columnIndex];
            }
        });
        TableColumn column;
        for (int i = 0; i < 5; i++) {
            column = hardwareTable.getColumnModel().getColumn(i);
            if (i == 0) {
                column.setPreferredWidth(150); //first column is bigger
            }
        }
        jScrollPane.setViewportView(hardwareTable);

        /**
         * General info panel *
         */
        JPanel generalInfoPanel = new JPanel();
        generalInfoPanel.setBorder(BorderFactory.createTitledBorder("General Informations"));
        generalInfoPanel.setLayout(new GridBagLayout());
        GridBagConstraints gbc4 = new GridBagConstraints();
        gbc4.insets = insets_std;
        //first colomn
        gbc4.gridx = 0;
        gbc4.gridy = 0;
        gbc4.anchor = GridBagConstraints.LINE_START;
        generalInfoPanel.add(new JLabel("Tcp Proxy name:"), gbc4);
        gbc4.gridy++;
        generalInfoPanel.add(new JLabel("Port number:"), gbc4);
        gbc4.gridy++;
        generalInfoPanel.add(new JLabel("Hardware boot timeout:"), gbc4);
        gbc4.gridy++;
        generalInfoPanel.add(new JLabel("Number of expected CANopen Nodes:"), gbc4);
        gbc4.gridy++;
        generalInfoPanel.add(new JLabel("Number of booted CANopen Nodes:"), gbc4);
        //second colomn
        gbc4.gridx++;
        gbc4.gridy = 0;
        gbc4.anchor = GridBagConstraints.LINE_END;
        generalInfoPanel.add(tcpProxyLabel, gbc4);
        gbc4.gridy++;
        generalInfoPanel.add(portNumberLabel, gbc4);
        gbc4.gridy++;
        generalInfoPanel.add(hardwareBootTimeoutLabel, gbc4);
        gbc4.gridy++;
        generalInfoPanel.add(expectedNodesNBLabel, gbc4);
        gbc4.gridy++;
        generalInfoPanel.add(bootedNodesNBLabel, gbc4);
        /**
         * end of General info panel *
         */

        hardwarePanel.setBorder(BorderFactory.createLineBorder(new Color(255, 0, 255)));
        hardwarePanel.setLayout(new GridBagLayout());
        GridBagConstraints gbc0 = new GridBagConstraints();
        gbc0.anchor = GridBagConstraints.NORTHWEST;
        gbc0.insets = insets_std;
        gbc0.gridx = 0;
        gbc0.gridy = 0;
        hardwarePanel.add(generalInfoPanel, gbc0);
        gbc0.gridx = 0;
        gbc0.gridy++;
        hardwarePanel.add(jScrollPane, gbc0);
        /**
         ******************************************************************************
         ** end of CANopen Hardware Panel
         * *****************************************************************************
         */
        //Whole Panel
        GridBagConstraints gbc = new GridBagConstraints();

        gbc.anchor = java.awt.GridBagConstraints.BASELINE_LEADING;
        gbc.insets = insets_std;
        gbc.gridx = 0;
        gbc.gridy = 0;
        add(hardwarePanel, gbc);

    }
    


    
    @Override
    public String toString() {
        return "CANopen hardware list";
    }

    /**
     * Just for tests.
     * @param argv 
     */
    public static void main(String[] argv) {

        CanOpenHardwareListPanel p = new CanOpenHardwareListPanel("fakeTcpProxy");

        JFrame frame = new JFrame("CANopen Hardware");
        frame.setContentPane(p);
        frame.pack();
        frame.setVisible(true);
        frame.setDefaultCloseOperation(JFrame.EXIT_ON_CLOSE);

    }

}
