/*
 * To change this license header, choose License Headers in Project Properties.
 * To change this template file, choose Tools | Templates
 * and open the template in the editor.
 */
package org.lsst.ccs.subsystems.fcs.ui.commons;

import java.awt.Color;
import java.awt.Dimension;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.event.ActionListener;
import java.util.Map;
import javax.swing.BorderFactory;
import javax.swing.JButton;
import javax.swing.JFrame;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.SwingUtilities;
import org.lsst.ccs.bus.data.ConfigurationInfo;
import org.lsst.ccs.subsystems.fcs.EPOSEnumerations.EposMode;
import static org.lsst.ccs.subsystems.fcs.FCSCst.FCSLOG;
import org.lsst.ccs.subsystems.fcs.StatusDataPublishedByEPOSController;
import org.lsst.ccs.subsystems.fcs.StatusDataPublishedByHardware;
import static org.lsst.ccs.subsystems.fcs.ui.commons.Tools.BIG_FONT;
import static org.lsst.ccs.subsystems.fcs.ui.commons.Tools.greenColor;
import static org.lsst.ccs.subsystems.fcs.ui.commons.Tools.insets_std;
import org.lsst.ccs.utilities.conv.InputConversionEngine;

/**
 * A Panel to display data coming from an EPOS motor controller.
 *
 * @author virieux
 */
public class EPOSControllerPanel extends JPanel {

    private static final long serialVersionUID = 2482213337630248322L;
    private InterfaceGeneralGUI subs;
    private final String controllerName;

    /**
     * Variables declaration *
     */
    //general informations panel
    private final CanOpenDevicePanel generalInfoPanel;

    //Fault Panel
    private JPanel faultPanel;
    private final DigitalSwitch faultSwitch = new DigitalSwitch();
    private final JLabel errorRegisterLabel = new JLabel("99=XXXXXXXX error");
    private final JLabel errorHistoryNameLabel = new JLabel("Error History:");
    private final JLabel errorHistoryLabel = new JLabel();

    //mode panel
    private JPanel modePanel;
    private final JLabel modeLabel = new JLabel("UNKNOWN");

    //control Panel
    private JPanel controlPanel;
    private final JButton checkFaultButton = new JButton("CheckFault");
    private final JButton faultResetButton = new JButton("FaultReset");
    private final JButton refreshButton = new JButton("Refresh");

    //panel for CURRENT mode parameters
    private final EPOSParametersPanel currentParamsPanel = new EPOSParametersPanel();
    //panel for PROFILE_POSITION mode parameters
    private final EPOSParametersPanel profile_positionParamsPanel = new EPOSParametersPanel();
    //panel for HOMING mode parameters
    private final EPOSParametersPanel homingParamsPanel = new EPOSParametersPanel();

    /**
     * A Runnable to initialize this Panel from data coming from the Configuration system.
     */
    private class GuiInitialization implements Runnable {

        private final ConfigurationInfo configInfo;

        public GuiInitialization(ConfigurationInfo configInfo) {
            this.configInfo = configInfo;
        }

        @Override
        public void run() {
            //General Informations Panel
            generalInfoPanel.initializeGui(configInfo);

            //Fault Panel
            faultSwitch.setColor(Color.GRAY);
            errorRegisterLabel.setText("Unknown ERROR");

            //Retrieve this controller configuration data
            Map<String, String> configForName = configInfo.getCurrentValuesFor(controllerName);
           
            //panel for CURRENT mode parameters
            Map<String, String> currentParamsMap = (Map<String, String>)InputConversionEngine.convertArgToType(configForName.get("paramsForCurrent"), Map.class);
            currentParamsPanel.initializeGUI(EposMode.CURRENT.toString(), currentParamsMap);
            
            //panel for PROFILE_POSITION mode parameters
            Map<String, String> profileParamsMap = (Map<String, String>)InputConversionEngine.convertArgToType(configForName.get("paramsForProfilePosition"), Map.class);
            profile_positionParamsPanel.initializeGUI(EposMode.PROFILE_POSITION.toString(), profileParamsMap);
            
            //panel for HOMING mode parameters
            Map<String, String> homingParamsMap = (Map<String, String>)InputConversionEngine.convertArgToType(configForName.get("paramsForHoming"), Map.class);
            homingParamsPanel.initializeGUI(EposMode.HOMING.toString(), homingParamsMap);
        }
    }

    /**
     * A Runnable class to update this panel from data published on the STATUS bus
     * by an EPOS Controller.
     */
    private class UpdateController implements Runnable {

        private StatusDataPublishedByEPOSController s;

        public UpdateController(StatusDataPublishedByEPOSController status) {
            this.s = status;
        }

        @Override
        public void run() {
            FCSLOG.debug("UpdateController:" + s.getName());
            //General Informations Panel
            generalInfoPanel.updateCanOpenDevice(s);

            //Mode Panel
            modeLabel.setText(s.getMode().toString());
            modeLabel.setFont(BIG_FONT);
            //Fault Panel
            if (s.isInitialized()) {
                faultSwitch.setColor(s.isInError() ? Color.RED : greenColor);
                errorRegisterLabel.setText(s.getErrorRegister());
                //error History
                String[] historyList = s.getErrorHistory();
                if (historyList == null ||historyList.length == 0) {
                    errorHistoryLabel.setText("no error");
                } else {
                    StringBuilder sb = new StringBuilder("<html>");
                    for (int i = 0; i < s.getErrorHistory().length; i++) {
                        if (i > 0) {
                            sb.append("<br>");
                        }
                        sb.append(historyList[i]);

                    }
                    sb.append("</html>");
                    errorHistoryLabel.setText(sb.toString());
                }
            } else {
                faultSwitch.setColor(Color.ORANGE);
                errorRegisterLabel.setText("Not initialized");
            }

        }
    }
    
    /**
     * A Runnable class to update this panel from data published on the STATUS bus
     * by a CANopen device.
     */
    private class UpdateDevice implements Runnable {
        
        private final StatusDataPublishedByHardware status;
        
        public UpdateDevice(StatusDataPublishedByHardware s) {
            this.status = s;
        }

        @Override
        public void run() {
            if (status.getName().equals(controllerName)) {
                generalInfoPanel.updateCanOpenDevice(status);
            }
        }
        
    }

    public EPOSControllerPanel(String controllerName) {
        this.controllerName = controllerName;
        this.generalInfoPanel = new CanOpenDevicePanel(controllerName);
        initComponents();
    }


    /**
     * To initialize the subsystem for this EPOSControllerPanel.
     * @param newSubs 
     */
    public void setSubsystem(InterfaceGeneralGUI newSubs) {
        this.subs = newSubs;
    }
    
    

    
    public void initializeGui(ConfigurationInfo configInfo) {
        SwingUtilities.invokeLater(new GuiInitialization(configInfo));
    }

    public void updateController(StatusDataPublishedByEPOSController status) {
        SwingUtilities.invokeLater(new UpdateController(status));
    }
    
    public void updateDevice(StatusDataPublishedByHardware s) {
        SwingUtilities.invokeLater(new UpdateDevice(s));
    }



    @SuppressWarnings("unchecked")
    // <editor-fold defaultstate="collapsed" desc="Form description">                          
    private void initComponents() {
        //Buttons
        checkFaultButton.addActionListener(new ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                checkFaultActionPerformed(evt);
            }
        });

        faultResetButton.addActionListener(new ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                faultResetActionPerformed(evt);
            }
        });

        refreshButton.addActionListener(new ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                refreshActionPerformed(evt);
            }
        });

        /**
         ******************************************************************************
         ** Mode Panel
         * *****************************************************************************
         */
        modePanel = new JPanel();
        modePanel.setBorder(BorderFactory.createTitledBorder("EPOS mode"));
        modePanel.setPreferredSize(new Dimension(160, 50));
        modePanel.setLayout(new GridBagLayout());
        GridBagConstraints gbc2 = new GridBagConstraints();
        gbc2.insets = insets_std;
        //first colomn
        gbc2.gridx = 0;
        gbc2.gridy = 0;
        gbc2.anchor = GridBagConstraints.LINE_START;
        //modePanel.add(mLabel,gbc2);
        //second colomn
//        gbc2.gridx++;
//        gbc2.anchor = GridBagConstraints.LINE_END;
        modeLabel.setFont(BIG_FONT);
        modePanel.add(modeLabel, gbc2);
        /**
         ******************************************************************************
         ** end of Mode Panel
         * *****************************************************************************
         */
        /**
         ******************************************************************************
         ** Fault Panel
         * *****************************************************************************
         */
        faultPanel = new JPanel();
        faultPanel.setBorder(BorderFactory.createTitledBorder("Error"));
        faultPanel.setLayout(new GridBagLayout());
        GridBagConstraints gbc1 = new GridBagConstraints();
        gbc1.insets = insets_std;
        gbc1.gridx = 0;
        gbc1.gridy = 0;
        gbc1.anchor = GridBagConstraints.CENTER;
        faultPanel.add(faultSwitch, gbc1);
        gbc1.gridy++;
        errorRegisterLabel.setFont(BIG_FONT);
        faultPanel.add(errorRegisterLabel, gbc1);
        gbc1.gridy++;
        faultPanel.add(errorHistoryNameLabel, gbc1);
        gbc1.gridy++;
        faultPanel.add(errorHistoryLabel, gbc1);
        /**
         ******************************************************************************
         ** end of Fault Panel
         * *****************************************************************************
         */
        /**
         * ****************************************************************************
         ** Control Panel
         * *****************************************************************************
         */
        controlPanel = new JPanel();
        controlPanel.setLayout(new java.awt.GridBagLayout());
        GridBagConstraints gbc4 = new java.awt.GridBagConstraints();
        gbc4.anchor = GridBagConstraints.FIRST_LINE_START;
        gbc4.fill = GridBagConstraints.HORIZONTAL;
        gbc4.insets = insets_std;
        gbc4.gridx = 0;
        gbc4.gridy = 0;
        controlPanel.add(refreshButton, gbc4);
        gbc4.gridy++;
        controlPanel.add(checkFaultButton, gbc4);
        gbc4.gridx++;
        controlPanel.add(faultResetButton, gbc4);

        /**
         ******************************************************************************
         ** End of Control Panel
         * *****************************************************************************
         */
        /**
         ******************************************************************************
         ** Whole Panel
         * *****************************************************************************
         */
        setBorder(javax.swing.BorderFactory.createLineBorder(Color.GREEN));
        //setForeground(new java.awt.Color(204, 204, 255));
        setLayout(new java.awt.GridBagLayout());
        GridBagConstraints gbc = new GridBagConstraints();
        gbc.insets = insets_std;
        gbc.anchor = GridBagConstraints.NORTHWEST;

        gbc.gridx = 0;
        gbc.gridy = 0;
        add(generalInfoPanel, gbc);

        gbc.gridx++;
        add(faultPanel, gbc);

        gbc.gridx = 0;
        gbc.gridy++;
        add(modePanel, gbc);

        gbc.gridx++;
        add(controlPanel, gbc);

        gbc.gridx = 0;
        gbc.gridy++;
        gbc.gridwidth = 4;
        add(currentParamsPanel, gbc);
        gbc.gridy++;
        add(profile_positionParamsPanel, gbc);
        gbc.gridy++;
        add(homingParamsPanel, gbc);

        /**
         ******************************************************************************
         ** end of Whole Panel
         * *****************************************************************************
         */
    }
    
    /**
     * Reset to default values;
     */
    public void resetPanel() {
        faultSwitch.setColor(Color.GRAY);
        errorRegisterLabel.setText("Unknown ERROR");
        errorHistoryLabel.setText("Unknown ERROR");
        this.currentParamsPanel.resetPanel();
        this.homingParamsPanel.resetPanel();
        this.profile_positionParamsPanel.resetPanel();
    }

    private void checkFaultActionPerformed(java.awt.event.ActionEvent evt) {
        subs.sendCommandSwingWorker("checkFault", 1000, controllerName);
    }

    private void faultResetActionPerformed(java.awt.event.ActionEvent evt) {
        subs.sendCommandSwingWorker("faultReset", 1000, controllerName);
    }

    private void refreshActionPerformed(java.awt.event.ActionEvent evt) {
        subs.sendCommandSwingWorker("publishData", 1000, controllerName);
    }

    @Override
    public String toString() {
        if (controllerName == null) {
            return "Unamed Controller Panel";

        } else {
            return controllerName;
        }
    }

    public static void main(String[] argv) {

        EPOSControllerPanel p = new EPOSControllerPanel("myController");

        JFrame frame = new JFrame("EPOS Controller");
        frame.setContentPane(p);
        frame.pack();
        frame.setVisible(true);
        frame.setDefaultCloseOperation(JFrame.EXIT_ON_CLOSE);
    }
}
