
package org.lsst.ccs.subsystems.fcs.loader.ui;

import java.lang.reflect.InvocationTargetException;
import java.util.List;
import javax.swing.JFrame;
import javax.swing.SwingUtilities;
import org.lsst.ccs.bus.data.ConfigurationInfo;
import org.lsst.ccs.bus.data.KeyValueData;
import org.lsst.ccs.bus.messages.StatusMessage;
import org.lsst.ccs.bus.messages.StatusSubsystemData;
import static org.lsst.ccs.subsystems.fcs.FCSCst.LOADER_TCPPROXY_NAME;
import org.lsst.ccs.subsystems.fcs.StatusDataPublishedByEPOSController;
import org.lsst.ccs.subsystems.fcs.StatusDataPublishedByHardware;
import org.lsst.ccs.subsystems.fcs.StatusDataPublishedByLoader;
import org.lsst.ccs.subsystems.fcs.StatusDataPublishedByLoaderCarrier;
import org.lsst.ccs.subsystems.fcs.StatusDataPublishedByLoaderClamp;
import org.lsst.ccs.subsystems.fcs.ui.commons.EPOSControllerPanel;
import org.lsst.ccs.subsystems.fcs.ui.commons.GatherPanel;
import org.lsst.ccs.subsystems.fcs.ui.commons.PlutoGatewayPanel;

/**
 * This panel gathers all the panels needed for the loader GUI.
 * Except the subsystemPanel which is handled by the GeneralPanel.
 * 
 * It takes care of initialization and update of the panels.
 * 
 * @author virieux
 */
public class LoaderGatherPanel extends GatherPanel {
    private static final long serialVersionUID = 4741041603580940956L;

    private InterfaceLoaderGUI subs;
    
    
    //CANopen devices panels
    private EPOSControllerPanel hooksControllerPanel;
    private EPOSControllerPanel carrierControllerPanel;
    private PlutoGatewayPanel plutoGatewayPanel;
    
    /** Loader hardware view Panel*/
    private LoaderGeneralViewPanel loaderGeneralViewPanel;
    private LoaderClampPanel loaderClampPanel;
    private LoaderCarrierPanel loaderCarrierPanel;
    

    /**
     * A Runnable to update the carrier controller panel.
     */
    class UpdateCarrierController implements Runnable {

        private final StatusDataPublishedByEPOSController s;

        public UpdateCarrierController(StatusDataPublishedByEPOSController s) {
            this.s = s;
        }

        @Override
        public void run() {
            carrierControllerPanel.updateController(s);
        }
    }

    /**
     * A Runnable to update the clamp controller panel.
     */
    class UpdateClampController implements Runnable {

        private final StatusDataPublishedByEPOSController s;

        public UpdateClampController(StatusDataPublishedByEPOSController s) {
            this.s = s;
        }

        @Override
        public void run() {
            hooksControllerPanel.updateController(s);
        }
    }

    /**
     * Creates new form LoaderTopPanel
     * @param tcpProxyName
     */
    public LoaderGatherPanel(String tcpProxyName) {
        super(tcpProxyName);
        initPanels();
    }

    public LoaderGatherPanel(InterfaceLoaderGUI subs, String tcpProxyName) {
        super(tcpProxyName);
        this.subs = subs;

        initPanels();
        loaderGeneralViewPanel.setSubsystem(subs);
        loaderCarrierPanel.setSubsystem(subs);
        loaderClampPanel.setSubsystem(subs);
        this.hooksControllerPanel.setSubsystem(subs);
        this.carrierControllerPanel.setSubsystem(subs);
    }


    public EPOSControllerPanel getHooksControllerPanel() {
        return hooksControllerPanel;
    }

    public EPOSControllerPanel getCarrierControllerPanel() {
        return carrierControllerPanel;
    }

    public PlutoGatewayPanel getPlutoGatewayPanel() {
        return plutoGatewayPanel;
    }

    public LoaderGeneralViewPanel getLoaderGeneralViewPanel() {
        return loaderGeneralViewPanel;
    }
    
    

    public LoaderClampPanel getLoaderClampPanel() {
        return loaderClampPanel;
    }

    public LoaderCarrierPanel getLoaderCarrierPanel() {
        return loaderCarrierPanel;
    }
    
    
    

    @Override
    public void initializeGUIWithConfigInfo(ConfigurationInfo configInfo, List<String> hardwareNames) 
            throws InterruptedException, InvocationTargetException {
        
        /**
         * top panel initialization
         */
        super.initializeGUIWithConfigInfo(configInfo, hardwareNames);
        
        /**
         * Clamp Controller Panel *
         */
        hooksControllerPanel.initializeGui(configInfo);
        /**
         * Carrier Controller Panel *
         */
        carrierControllerPanel.initializeGui(configInfo);
        
        /**
         * PlutoGateway Panel *
         */
        plutoGatewayPanel.initializeGui(configInfo);
        
        /**
         * LoaderGeneralViewPanel initialization *
         */
        loaderGeneralViewPanel.initializeGui(configInfo);

        /**
         * Loader Clamp Panel *
         */
        loaderClampPanel.initializeGui(configInfo);
        
        /**
         * Loader Carrier  Panel *
         */
        loaderCarrierPanel.initializeGui(configInfo);
        

    }
    
    void updateLoaderGeneral(StatusDataPublishedByLoader statusDataPublishedByLoader) {
        this.loaderGeneralViewPanel.updateLoaderGeneral(statusDataPublishedByLoader);
        this.loaderClampPanel.updateClamp(statusDataPublishedByLoader);
    }
    
    void updateFakeAutochanger(int sensorValue) {
        this.loaderGeneralViewPanel.updateFakeAutochanger(sensorValue);
    }

    void updateClamp(StatusDataPublishedByLoaderClamp statusDataPublishedByLoaderClamp) {
        this.loaderClampPanel.updateClamp(statusDataPublishedByLoaderClamp);
        this.loaderGeneralViewPanel.updateClamp(statusDataPublishedByLoaderClamp);
    }

    void updateCarrier(StatusDataPublishedByLoaderCarrier statusDataPublishedByLoaderCarrier) {
        this.loaderCarrierPanel.updateCarrier(statusDataPublishedByLoaderCarrier);
    }


    public void updateCarrierController(StatusDataPublishedByEPOSController status) {
        SwingUtilities.invokeLater(new UpdateCarrierController(status));
    }

    public void updateClampController(StatusDataPublishedByEPOSController status) {
        SwingUtilities.invokeLater(new UpdateClampController(status));
    }
    
    @Override
    public void updateCanOpenHardwarePanel(StatusDataPublishedByHardware status) {
        super.updateCanOpenHardwarePanel(status);
        this.hooksControllerPanel.updateDevice(status);
        this.carrierControllerPanel.updateDevice(status);
        this.plutoGatewayPanel.updateDevice(status);
    }

    @Override
    public void onStatusMessage(StatusMessage sm) {
        // Safe cast : filtering ensures we only receive StatusSubsystemData messages
        StatusSubsystemData d = (StatusSubsystemData)sm;
        KeyValueData data = d.getObject();
        FCSLOG.debug(subs.getName() + " read data dataType:" + d.getDataKey());
        switch (d.getDataKey()) {
            case "loaderGeneral":
                updateLoaderGeneral((StatusDataPublishedByLoader) data.getValue());
                break;
                
            case "fakeAutochanger":
                updateFakeAutochanger((int) data.getValue());
                break;    
                
            case "loaderClamp":
                updateClamp((StatusDataPublishedByLoaderClamp) data.getValue());
                break;
                
            case "loaderCarrier":
                updateCarrier((StatusDataPublishedByLoaderCarrier) data.getValue());
                break;

            case "loaderTcpProxy":
                updateCanOpenHardwarePanel((StatusDataPublishedByHardware) data.getValue());
                break;

            case "carrierController":
                updateCarrierController(
                        (StatusDataPublishedByEPOSController) data.getValue());
                break;

            case "hooksController":
                updateClampController(
                        (StatusDataPublishedByEPOSController) data.getValue());
                break;

            default:
                FCSLOG.warning("LoaderGUIModule An unexpected dataType was read on the status bus : " 
                        + d.getDataKey());
                break;
        }
    }
    // <editor-fold defaultstate="collapsed" desc="Form description">                          
    private void initPanels() {
        //controllers Panels
        hooksControllerPanel = new EPOSControllerPanel("hooksController");
        carrierControllerPanel = new EPOSControllerPanel("carrierController");
        
        //plutoGateway panel        
        plutoGatewayPanel = new PlutoGatewayPanel("loaderPlutoGateway");
        
        //loader hardware panels
        loaderGeneralViewPanel = new LoaderGeneralViewPanel();
        loaderClampPanel = new LoaderClampPanel("clamp");
        loaderCarrierPanel = new LoaderCarrierPanel("carrier");
    }
    
    /**
     * To reset panel to default values.
     */
    @Override
    public void resetPanel() {
        super.resetPanel();
        hooksControllerPanel.resetPanel();
        carrierControllerPanel.resetPanel();
        loaderGeneralViewPanel.resetPanel();
        loaderClampPanel.resetPanel();
        loaderCarrierPanel.resetPanel();
    }

    public static void main(String[] argv) {

        LoaderGatherPanel d = new LoaderGatherPanel(LOADER_TCPPROXY_NAME);
        JFrame frame = new JFrame("Loader High Panel");
        frame.setContentPane(d);
        frame.pack();
        frame.setVisible(true);
        frame.setDefaultCloseOperation(JFrame.EXIT_ON_CLOSE);

    }

}
