package org.lsst.ccs.subsystems.fcs.ui.commons;

import java.awt.HeadlessException;
import java.io.PrintWriter;
import java.io.Serializable;
import java.lang.reflect.InvocationTargetException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Future;
import javax.swing.JComponent;
import javax.swing.JOptionPane;
import javax.swing.JScrollPane;
import javax.swing.SwingWorker;
import org.freehep.graphicsbase.swing.ErrorDialog;
import org.lsst.ccs.bus.data.AgentInfo;
import org.lsst.ccs.bus.data.ConfigurationInfo;
import org.lsst.ccs.bus.messages.CommandAck;
import org.lsst.ccs.bus.messages.CommandError;
import org.lsst.ccs.bus.messages.CommandNack;

import org.lsst.ccs.bus.messages.CommandRequest;
import org.lsst.ccs.bus.messages.CommandResult;

import org.lsst.ccs.bus.messages.StatusMessage;
import org.lsst.ccs.bus.messages.StatusStateChangeNotification;
import org.lsst.ccs.bus.messages.StatusSubsystemData;

import org.lsst.ccs.bus.utils.SynchronousCommandAgent;

import org.lsst.ccs.messaging.AgentMessagingLayer;
import org.lsst.ccs.messaging.AgentPresenceListener;
import org.lsst.ccs.messaging.BusMessageFilter;
import org.lsst.ccs.messaging.CommandOriginator;
import org.lsst.ccs.messaging.ConcurrentMessagingUtils;
import org.lsst.ccs.messaging.StatusMessageListener;
import static org.lsst.ccs.subsystems.fcs.FCSCst.FCSLOG;

import org.lsst.ccs.utilities.structs.TreeBranch;

/**
 * This class gathers the common stuff for all FCS GUIs.
 *
 * @author virieux
 */
public abstract class GeneralGUI implements
        StatusMessageListener,
        AgentPresenceListener,
        Serializable {

    private static final long serialVersionUID = -3641559220265844557L;
  

    protected final String destination;
    protected GeneralPanel generalPanel;
    protected boolean panelStateInitialized = false;
    public JScrollPane pane;
    public long abortTimeout;
    public long updateStateWithSensorsTimeout;

    private String name;
    protected transient AgentMessagingLayer aml;
    private final transient ConcurrentMessagingUtils cmu;
    private final transient BusMessageFilter originFilter;
    private final transient BusMessageFilter dataStatusFilter;

    public long initializationTimeout;
    
    /**
     * To send command which takes time from a JButton.
     */
    public class CommandSwingWorker extends SwingWorker<Object, Object> {

        String cmdDestination;
        String cmdName;
        long cmdTimeout;

        public CommandSwingWorker(String cmdN, long timeout) {
            super();
            if (cmdN == null) {
                throw new IllegalArgumentException("Command should not be null in CommandSwingWorker");
            }
            if (timeout == 0) {
                throw new IllegalArgumentException("timeout should not be equal to 0 in CommandSwingWorker");
            }
            this.cmdDestination = destination;
            this.cmdName = cmdN;
            this.cmdTimeout = timeout;
        }

        public CommandSwingWorker(String cmdN, long timeout, String moduleName) {
            super();
            if (cmdN == null) {
                throw new IllegalArgumentException("Command should not be null in CommandSwingWorker");
            }
            if (timeout == 0) {
                throw new IllegalArgumentException("timeout should not be equal to 0 in CommandSwingWorker");
            }
            if (moduleName == null) {
                throw new IllegalArgumentException("moduleName should not be null in this constructor of CommandSwingWorker");
            }
            this.cmdDestination = destination + "/" + moduleName;
            this.cmdName = cmdN;
            this.cmdTimeout = timeout;
        }

        @Override
        public Object doInBackground() {
            FCSLOG.info("Executing " + cmdName + " on " + cmdDestination);
            FCSLOG.info("/timeout=" + cmdTimeout);
            return sendCommand(new CommandRequest(cmdDestination, cmdName), cmdTimeout);
        }

        @Override
        protected void done() {
            try {
                Object response = get();
                String strRes;
                if (response instanceof Throwable) {
                    FCSLOG.error(String.valueOf(response));
//                    JOptionPane.showMessageDialog(null, response.toString(), "ALERT", JOptionPane.ERROR_MESSAGE);
                    FCSLOG.error(" command returned error", (Throwable) response);
                    //ErrorDialog.showErrorDialog(null,"Command returned error",(Throwable) response);
                    ErrorDialog.showErrorDialog(null, response.toString(), (Throwable) response);
                    // if (!(response instanceof BadCommandException)) raiseAlarm(response.toString());

                } else if (response == null) {
                    strRes = "ok : DONE";
                    JOptionPane.showMessageDialog(null, strRes, cmdName, JOptionPane.INFORMATION_MESSAGE);

                } else {
                    strRes = String.valueOf(response);
                    JOptionPane.showMessageDialog(null, strRes, cmdName, JOptionPane.INFORMATION_MESSAGE);
                }
            } catch (InterruptedException | ExecutionException | HeadlessException ex) {
                FCSLOG.error(ex.toString());
                //JOptionPane.showMessageDialog(null, ex.toString(), "ALERT", JOptionPane.ERROR_MESSAGE);
                ErrorDialog.showErrorDialog(null, ex.toString(), (Throwable) ex);
                // raiseAlarm(ex.toString());
            }
        }
    }

    public GeneralGUI(String name,
            String destination,
            AgentMessagingLayer aml) {
        this.aml = aml;
        this.cmu = new ConcurrentMessagingUtils(aml);
        this.name = name;
        this.destination = destination;
        this.abortTimeout = 1000;
        this.updateStateWithSensorsTimeout = 1000;
        this.initializationTimeout = 2000;
        
        // ---- Filters
        // filters only the messages coming from my subsystem.
        originFilter = BusMessageFilter.messageOrigin(destination);
        // filters messages that are subsystem data statuses
        dataStatusFilter = BusMessageFilter.messageClass(StatusSubsystemData.class);
        // -----

    }

    public String getName() {
        return name;
    }

    public void initGui() {

        FCSLOG.info(name + ":initGUI");
        // This GUI listens to agent connectio/disconnection events
        aml.getAgentPresenceManager().addAgentPresenceListener(this);
        
    
        
        aml.addStatusMessageListener(this, originFilter);

      
    }

    public JComponent getGuiLayout() {
        return pane;
    }

    public void resetGui() {
        generalPanel.resetPanel();
    }

    public void shutdownMyDestinationSubsystem() {
        System.out.println("Shutting down " + destination);
        //the following doesn't work with Toolkit-V2.0
        //sendAsynchronousCommand(new CommandRequest("shutdown", destination));
        //This does not work properly because the abort method return code is void so it
        //always opens a JOPtionPane with ok : DONE
        new CommandSwingWorker("shutdown", abortTimeout).execute();
    }

    public void abortAction() {
        System.out.println("Aborting " + destination);
        //the following doesn't work with Toolkit-V2.0
        //sendAsynchronousCommand(new CommandRequest("abort", destination));
        //This does not work properly because the abort method return code is void so it
        //always opens a JOPtionPane with ok : DONE
        new CommandSwingWorker("abort", abortTimeout).execute();
    }

    public void stopMyDestinationSubsystem() {
        System.out.println("Stopping " + destination);
        cmu.sendAsynchronousCommand(new CommandRequest(destination, "stop", 1000L));
    }

    public void updateStateWithSensors() {
        FCSLOG.info("Reading sensors");
        FCSLOG.info("/timeout=" + updateStateWithSensorsTimeout);
        new CommandSwingWorker("updateStateWithSensors", updateStateWithSensorsTimeout).execute();
    }

    public void completeInitialization() {
        FCSLOG.info("Reading sensors");
        FCSLOG.info("/timeout=" + initializationTimeout);
        new CommandSwingWorker("completeInitialization", initializationTimeout).execute();
    }

    public void sendCommandSwingWorker(String commandName, long timeout, String moduleName) {
        FCSLOG.info(getName() + ": sending command= " + commandName + " to Module=" + moduleName);
        new CommandSwingWorker(commandName, timeout, moduleName).execute();
    }

    public void sendCommandSwingWorker(String commandName, long timeout) {
        FCSLOG.info(getName() + ": sending command= " + commandName);
        new CommandSwingWorker(commandName, timeout).execute();
    }


    public void sendAsynchronousCommand(CommandRequest cmd) {
        aml.sendCommandRequest(cmd, new CommandOriginator() {

            @Override
            public void processAck(CommandAck ack) {
            }

            @Override
            public void processResult(CommandResult result) {
            }

            @Override
            public void processNack(CommandNack nack) {
            }

            @Override
            public void processError(CommandError nack) {
            }
        });
    }

    public Object sendCommand(CommandRequest cmd, long timeout) {
        try {
            FCSLOG.debug("fac=" + String.valueOf(aml));
            SynchronousCommandAgent agent = new SynchronousCommandAgent(aml);
            FCSLOG.debug("agent=" + String.valueOf(agent));
            FCSLOG.debug("command=" + String.valueOf(cmd));
            Object response = agent.invoke(cmd, timeout);
            //ask Bernard
            String strRes = String.valueOf(response);
            FCSLOG.debug("strRes=" + strRes);
            if (response == null) {
                return "ok : DONE";
                //JOptionPane.showMessageDialog(null, strRes, cmd.getCommand(),  JOptionPane.INFORMATION_MESSAGE);
            } else {
                return strRes;
                //JOptionPane.showMessageDialog(null, strRes, cmd.getCommand(), JOptionPane.INFORMATION_MESSAGE);
            }
        } catch (Exception ex) {
            return ex;
        }
    }

    /**
     * To update the Subsystem state panel only when the subsytem state has
     * changed.
     *
     * @param s
     */
    @Override
    public void onStatusMessage(StatusMessage s) {
        if (!panelStateInitialized) {
            this.generalPanel.getSubsystemPanel().updateState(s.getState());
            panelStateInitialized = true;

        } else {
            if (!(s instanceof StatusStateChangeNotification)) {
                return;
            }
            this.generalPanel.getSubsystemPanel().updateState(s.getState());
        }
    }

    @Override
    public void connecting(AgentInfo agent) {
        //my subsystem is connecting on the bus
        if (!agent.getName().equals(this.destination)) return;
        
        FCSLOG.warning(destination + " is connecting NOW :");
        
        
        List<String> hardwareList = getHardwareNamesForBridge("bridge", "tcpProxy");
        FCSLOG.debug("retrieved the list of connected hardwares"); 
        
        List<String> filterNamesList = getFilterNames();
        FCSLOG.debug("retrieved the list of filter names"); 
        
        /* Send asynchronous command to populate panels with configuration data.*/
        CommandRequest cmd = new CommandRequest(destination, "getConfigurationInfo");
        
        aml.sendCommandRequest(cmd, new CommandOriginator() {
            
            @Override
            public void processAck(CommandAck ack) {
                FCSLOG.debug(destination + " has received a getConfigurationInfo command:" 
                        + ack.toString());
            }
            
            @Override
            public void processResult(CommandResult result) {
                try {
                    FCSLOG.debug("--- Got a reply to getConfigurationInfo from subs " + destination);
                    ConfigurationInfo configInfo = (ConfigurationInfo) result.getReply();
                    // Gui Initialization with configuration information
                    generalPanel.initializeGUIWithConfigInfo(configInfo, hardwareList);
                    generalPanel.initializeGUIWithFiltersInfo(configInfo, filterNamesList);
                    
                    // Filters only specific data status from the worker subsystem
                    aml.addStatusMessageListener(generalPanel, originFilter.and(dataStatusFilter));
                   
                    
                    
                    /* Send command to populate panels with dynamic data.*/
                    //TODO find another solution in the case of hardware cf tests in January 2016
//                    FCSLOG.debug("sending updateStateWithSensors");
//                    cmu.sendAsynchronousCommand(new CommandRequest(destination, "updateStateWithSensors"));
                } catch (InterruptedException | InvocationTargetException ex) {
                    FCSLOG.error(name+ ":Error occured when initializing GUI with config info: " +ex);
                }
            }
            
            @Override
            public void processNack(CommandNack nack) {
            }
            
            @Override
            public void processError(CommandError nack) {
                FCSLOG.error("--- Error in execution of getConfigurationInfo command:"
                        + nack.toString());
            }
        });
    }
    
    /**
     * Sends a command to retrieve the list of hardware connected to the given bridge.
     * It sends a command to the worker subsystem
     * @param bridge the bridge to list the hardware connected to.
     * @param tcpProxy the tcpProxy associated to the bridge.
     * @return a list of hardware names
     */
    private List<String> getHardwareNamesForBridge(String bridge, String tcpProxy){
        Future<Object> future = cmu.sendAsynchronousCommand(new CommandRequest(destination+"/"+bridge, "getComponentTree"));
        TreeBranch<String> treeRes;
        try {
            treeRes = (TreeBranch<String>)future.get();
        } catch (InterruptedException | ExecutionException ex){
            FCSLOG.error(" Unable to retrieve the list of hardware names." + ex);
            return Collections.EMPTY_LIST;
        }
        List<String> listRes = new ArrayList<>();
        for (TreeBranch<String> child : treeRes.getChildren()){
            if (!child.getContent().equals(tcpProxy)){
                listRes.add(child.getContent());
            }
        }
        return listRes;
    }
    
    /**
     * Sends a command to the worker subsystem to retrieve the list of filter names. 
     * This list is then used to initialize the filterListPanels.
     * When there is no filter names list, it returns an empty list.
     * @return 
     */
    private List<String> getFilterNames() {
        Future<Object> future = cmu.sendAsynchronousCommand(new CommandRequest(destination, "getFilterNames"));
        try {
            return (List<String>)future.get();
        } catch (InterruptedException | ExecutionException ex) {
            FCSLOG.error("Unable to retrieve the list of filter names." + ex);
            return Collections.emptyList();
        } 
    }
    
    @Override
    public void disconnecting(AgentInfo agent) {
        if (agent.getName().equals(this.destination)) {
            FCSLOG.warning(destination + " is disconnecting NOW");
            /*my subsystem is disconnecting from the bus.*/
            /*we have to reset all the panels*/
            //TODO
            resetGui();
        }
    }

}
