/*
 * To change this license header, choose License Headers in Project Properties.
 * To change this template file, choose Tools | Templates
 * and open the template in the editor.
 */
package org.lsst.ccs.subsystems.fcs.ui.commons;

import java.awt.Color;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import javax.swing.BorderFactory;
import javax.swing.JFrame;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.JTable;
import javax.swing.SwingUtilities;
import javax.swing.table.DefaultTableModel;
import javax.swing.table.TableColumn;
import org.lsst.ccs.bus.data.ConfigurationInfo;
import org.lsst.ccs.subsystems.fcs.FCSCst;
import org.lsst.ccs.subsystems.fcs.StatusDataPublishedByHardware;
import static org.lsst.ccs.subsystems.fcs.ui.commons.Tools.insets_std;
import org.lsst.ccs.utilities.logging.Logger;

/**
 * This Panel displays in a table the list of CANopen devices the subsystem manages.
 * It displays also the portnumber of the tcp connection of the tcpProxy and the 
 * numbers of devices on the CAN bus.
 * @author virieux
 */
public class CanOpenHardwareListPanel extends JPanel {
    
    private final String tcpProxyName;

    // Variables declaration 
    protected static final Logger FCSLOG = FCSCst.FCSLOG;
    private static final long serialVersionUID = 4473630289328745341L;

    private JLabel tcpProxyNameLabel;
    private JLabel tcpProxyLabel;
    private JLabel bootedNodesNBLabel;
    private JLabel expectedNodesNBLabel;
    private JLabel hardwareBootTimeoutLabel;
    private JPanel hardwarePanel;
    private JTable hardwareTable;
    private JLabel bootedNodesNBNameLabel;
    private JLabel hardwareBootTimeoutNameLabel;
    private JLabel expectedNodesNBNameLabel;
    private JLabel portNumberNameLabel;
    private JLabel portNumberLabel;
    private JScrollPane jScrollPane;
    // End of variables declaration 

    public void initializeGui(ConfigurationInfo configInfo, List<String> hardwareList) {
        SwingUtilities.invokeLater(new GuiInitializationWithConfigInfo(configInfo,hardwareList));
    }
    
    /**
     * Initialization of the GUI for data stored in tcpConfiguration. *
     */
    private class GuiInitializationWithConfigInfo implements Runnable {
        private final ConfigurationInfo configInfo; 
        private final List<String> hardwareList;

        public GuiInitializationWithConfigInfo(ConfigurationInfo configInfo, List<String> hardwareList) {
            this.configInfo = configInfo;
            /* copy of teh hardwareList*/
            this.hardwareList = new ArrayList<>();
            this.hardwareList.addAll(hardwareList);
        }
        
        @Override
        public void run() {
            int nodesNB = hardwareList.size();
            Map<String, String> config = configInfo.getCurrentValuesForComponent(tcpProxyName);
            portNumberLabel.setText(Tools.getText(config,tcpProxyName, "portNumber"));
            expectedNodesNBLabel.setText(Integer.toString(nodesNB));
            hardwareBootTimeoutLabel.setText(Tools.getText(config,tcpProxyName, "hardwareBootTimeout"));

            //configure HardwareTable           
            bootedNodesNBLabel.setText(Integer.toString(nodesNB)); // not a configuration data 
            //TODO : update bootedNodesNBLabel from status data.
            DefaultTableModel hardwareTableModel = (DefaultTableModel) hardwareTable.getModel();
            for (String hardwareName: hardwareList) {
                Map<String, String> hardwareConfig = configInfo.getCurrentValuesForComponent(hardwareName);
                String nodeID = Tools.getText(hardwareConfig, hardwareName, "nodeID");
                String serialNB = Tools.getText(hardwareConfig, hardwareName, "serialNB");
                Object[] tab = {hardwareName, nodeID, serialNB,
                    false, false};
                hardwareTableModel.addRow(tab);
            }
            hardwareTableModel.fireTableDataChanged();
        }
        
    }
    


    /**
     * A Runnable to update the Panel when the tcpProxy publishes on the Status 
     * bus the status of a CANopen hardware.
     */
    private class UpdateCanOpenHardwarePanel implements Runnable {

        private StatusDataPublishedByHardware s;

        public UpdateCanOpenHardwarePanel(StatusDataPublishedByHardware status) {
            this.s = status;
        }

        @Override
        public void run() {
            DefaultTableModel hardwareTableModel = (DefaultTableModel) hardwareTable.getModel();
            int rowNB = hardwareTableModel.getRowCount();
            for (int i = 0; i < rowNB; i++) {
                if ((hardwareTableModel.getValueAt(i, 0)).equals(s.getName())) {
                    hardwareTableModel.setValueAt(s.isBooted(), i, 3);
                    hardwareTableModel.setValueAt(s.isInitialized(), i, 4);
                }
            }
            hardwareTableModel.fireTableDataChanged();
        }
    }

    /**
     * Creates new form CanOpenHardware with a tcpProxy name.
     * @param tcpProxyName
     */
    public CanOpenHardwareListPanel(String tcpProxyName) {
        this.tcpProxyName = tcpProxyName;
        initComponents();
    }

   

    public void updateCanOpenHardwarePanel(StatusDataPublishedByHardware status) {
        SwingUtilities.invokeLater(new UpdateCanOpenHardwarePanel(status));
    }

    // <editor-fold defaultstate="collapsed" desc="Form description">                          
    private void initComponents() {

        hardwarePanel = new JPanel();
        
        tcpProxyNameLabel = new JLabel("Tcp Proxy name:");
        tcpProxyLabel = new JLabel(this.tcpProxyName);
        
        bootedNodesNBNameLabel = new JLabel("Number of booted CANopen Nodes:");
        bootedNodesNBLabel = new JLabel("0");

        hardwareBootTimeoutNameLabel = new JLabel("Hardware boot timeout:");
        hardwareBootTimeoutLabel = new JLabel("0");

        expectedNodesNBNameLabel = new JLabel("Number of expected CANopen Nodes:");
        expectedNodesNBLabel = new JLabel("0");

        portNumberNameLabel = new JLabel("Port number:");
        portNumberLabel = new JLabel("0");

        /**
         ******************************************************************************
         ** CANopen Hardware Panel
         * *****************************************************************************
         */
        jScrollPane = new JScrollPane();
        hardwareTable = new JTable();
        hardwareTable.setModel(new DefaultTableModel(
                new Object[][]{},
                new String[]{
                    "Name", "Node ID", "Serial NB", "Booted", "Initialized"
                }
        ) {
            private static final long serialVersionUID = 1L;
            private final Class[] types = new Class[]{
                String.class, String.class, Object.class, Boolean.class, Boolean.class
            };
            private final boolean[] canEdit = new boolean[]{
                false, false, false, false, false
            };

            @Override
            public Class getColumnClass(int columnIndex) {
                return types[columnIndex];
            }

            @Override
            public boolean isCellEditable(int rowIndex, int columnIndex) {
                return canEdit[columnIndex];
            }
        });
        TableColumn column;
        for (int i = 0; i < 5; i++) {
            column = hardwareTable.getColumnModel().getColumn(i);
            if (i == 0) {
                column.setPreferredWidth(150); //first column is bigger
            }
        }
        jScrollPane.setViewportView(hardwareTable);

        /**
         * General info panel *
         */
        JPanel generalInfoPanel = new JPanel();
        generalInfoPanel.setBorder(BorderFactory.createTitledBorder("General Informations"));
        generalInfoPanel.setLayout(new GridBagLayout());
        GridBagConstraints gbc4 = new GridBagConstraints();
        gbc4.insets = insets_std;
        //first colomn
        gbc4.gridx = 0;
        gbc4.gridy = 0;
        gbc4.anchor = GridBagConstraints.LINE_START;
        generalInfoPanel.add(tcpProxyNameLabel, gbc4);
        gbc4.gridy++;
        generalInfoPanel.add(portNumberNameLabel, gbc4);
        gbc4.gridy++;
        generalInfoPanel.add(hardwareBootTimeoutNameLabel, gbc4);
        gbc4.gridy++;
        generalInfoPanel.add(expectedNodesNBNameLabel, gbc4);
        gbc4.gridy++;
        generalInfoPanel.add(bootedNodesNBNameLabel, gbc4);
        //second colomn
        gbc4.gridx++;
        gbc4.gridy = 0;
        gbc4.anchor = GridBagConstraints.LINE_END;
        generalInfoPanel.add(tcpProxyLabel, gbc4);
        gbc4.gridy++;
        generalInfoPanel.add(portNumberLabel, gbc4);
        gbc4.gridy++;
        generalInfoPanel.add(hardwareBootTimeoutLabel, gbc4);
        gbc4.gridy++;
        generalInfoPanel.add(expectedNodesNBLabel, gbc4);
        gbc4.gridy++;
        generalInfoPanel.add(bootedNodesNBLabel, gbc4);
        /**
         * end of General info panel *
         */

        hardwarePanel.setBorder(BorderFactory.createLineBorder(new Color(255, 0, 255)));
        hardwarePanel.setLayout(new GridBagLayout());
        GridBagConstraints gbc0 = new GridBagConstraints();
        gbc0.anchor = GridBagConstraints.NORTHWEST;
        gbc0.insets = insets_std;
        gbc0.gridx = 0;
        gbc0.gridy = 0;
        hardwarePanel.add(generalInfoPanel, gbc0);
        gbc0.gridx = 0;
        gbc0.gridy++;
        hardwarePanel.add(jScrollPane, gbc0);
        /**
         ******************************************************************************
         ** end of CANopen Hardware Panel
         * *****************************************************************************
         */
        //Whole Panel
        GridBagConstraints gbc = new GridBagConstraints();

        gbc.anchor = java.awt.GridBagConstraints.BASELINE_LEADING;
        gbc.insets = insets_std;
        gbc.gridx = 0;
        gbc.gridy = 0;
        add(hardwarePanel, gbc);

    }

    
    @Override
    public String toString() {
        return "CANopen hardware list";
    }

    /**
     * Just for tests.
     * @param argv 
     */
    public static void main(String[] argv) {

        CanOpenHardwareListPanel p = new CanOpenHardwareListPanel("fakeTcpProxy");

        JFrame frame = new JFrame("CANopen Hardware");
        frame.setContentPane(p);
        frame.pack();
        frame.setVisible(true);
        frame.setDefaultCloseOperation(JFrame.EXIT_ON_CLOSE);

    }

}
