package org.lsst.ccs.subsystems.fcs.ui;

import java.awt.BasicStroke;
import java.awt.BorderLayout;
import java.awt.Dimension;
import java.awt.Graphics;
import java.awt.Graphics2D;
import java.awt.RenderingHints;
import java.awt.Shape;
import java.awt.Stroke;
import java.awt.geom.Arc2D;
import javax.swing.JComponent;
import javax.swing.JPanel;
import javax.swing.event.ChangeEvent;
import javax.swing.event.ChangeListener;

/**
 *
 * @author Tony Johnson
 */
public final class Carousel extends JPanel {
    private static final long serialVersionUID = -8531076243930890565L;

    private transient CarouselModel model;
    private transient final ChangeListener cl = new CarouselChangeListener();

    Carousel() {
        super(new BorderLayout());
        setModel(new CarouselModel());
        add(new CarouselDisk(), BorderLayout.CENTER);
    }

    public void setModel(CarouselModel model) {
        if (this.model != null) {
            model.removeChangeListener(cl);
        }
        this.model = model;
        model.addChangeListener(cl);
    }

    public CarouselModel getModel() {
        return model;
    }

    private class CarouselChangeListener implements ChangeListener {

        @Override
        public void stateChanged(ChangeEvent e) {
            repaint();
        }
    }

    private class CarouselDisk extends JComponent {
        private static final long serialVersionUID = 8623623490718103957L;

        CarouselDisk() {
            setPreferredSize(new Dimension(400, 400));
        }

        @Override
        protected void paintComponent(Graphics g) {
            Graphics2D g2 = (Graphics2D) g;
            RenderingHints hints = new RenderingHints(RenderingHints.KEY_ANTIALIASING, RenderingHints.VALUE_ANTIALIAS_ON);
            g2.addRenderingHints(hints);

            float width = getWidth();
            float height = getHeight();
            float actualWidth = Math.min(width, height);

            float border = 10;

            // Draw the outline
            Shape circle = new Arc2D.Double(border, border, actualWidth - 2 * border, actualWidth - 2 * border, 0, 360, Arc2D.CHORD);
            g2.draw(circle);

            double filterDegrees = 360. / model.getFilterNames().length;

            // Draw the "loading window"
            Stroke oldStroke = g2.getStroke();
            g2.setStroke(new BasicStroke(5));
            Shape loader = new Arc2D.Double(border, border, actualWidth - 2 * border, actualWidth - 2 * border, 0, -filterDegrees, Arc2D.OPEN);
            g2.draw(loader);
            g2.setStroke(oldStroke);

            g2.rotate(model.getRotationInDegrees() * Math.PI / 180, actualWidth / 2, actualWidth / 2);
            border *= 2;

            for (String filter : model.getFilterNames()) {
                Shape arc = new Arc2D.Double(border, border, actualWidth - 2 * border, actualWidth - 2 * border, 0, -filterDegrees, Arc2D.PIE);
                boolean filterPresent = !filter.equals(model.getFilterInCamera());
                if (filterPresent) {
                    g2.draw(arc);
                }
                g2.rotate(filterDegrees * Math.PI / 360, actualWidth / 2, actualWidth / 2);
                if (filterPresent) {
                    g2.drawString(filter, actualWidth * 3 / 4, actualWidth / 2);
                }
                g2.rotate(filterDegrees * Math.PI / 360, actualWidth / 2, actualWidth / 2);
            }
        }
    }
}
